package com.paypal.hybris.core.commands.impl;

import com.paypal.core.PayPalEnvironment;
import com.paypal.core.PayPalHttpClient;
import com.paypal.http.Headers;
import com.paypal.http.HttpResponse;
import com.paypal.hybris.core.commands.PayPalAbstractCommandTest;
import com.paypal.hybris.core.exception.PayPalCaptureAdapterException;
import com.paypal.hybris.core.results.PayPalCaptureResult;
import com.paypal.hybris.core.service.PayPalConfigurationService;
import com.paypal.payments.AuthorizationsCaptureRequest;
import com.paypal.payments.Capture;
import com.paypal.payments.Money;
import de.hybris.bootstrap.annotations.UnitTest;
import de.hybris.platform.payment.AdapterException;
import de.hybris.platform.payment.commands.request.CaptureRequest;
import de.hybris.platform.payment.dto.TransactionStatus;
import de.hybris.platform.payment.dto.TransactionStatusDetails;
import de.hybris.platform.servicelayer.user.UserService;
import org.apache.commons.io.IOUtils;
import org.junit.Before;
import org.junit.Test;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;
import org.mockito.Spy;

import java.io.IOException;
import java.math.BigDecimal;
import java.nio.charset.StandardCharsets;
import java.time.Instant;
import java.util.Currency;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;

import static org.assertj.core.api.Assertions.assertThat;
import static org.junit.Assert.assertEquals;
import static org.mockito.Matchers.any;
import static org.mockito.Mockito.never;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

@UnitTest
public class DefaultPaypalOnDemandCaptureCommandTest {

	private static final String REQUEST_ID = "requestId";
	private static final String TOKEN = "token";
	private static final String CLIENT_ID = "clientId";
	private static final String SECRET_KEY = "secretKey";
	private static final String RESULT_STATUS = "ACCEPTED";
	private static final String SUBSCRIPTION_ID = "subscriptionId";
	private static final String MERCHANT_TRANSACTION_CODE = "merchantTransactionCode";
	private static final String CAPTURE_ID = "captureId";
	private static final String EXCEPTION = "{\"details\": [{\"description\": \"Error\"}]}";

	private static final String PAYPAL_DEBUG_ID = "Paypal-Debug-Id";
	private static final String PAYPAL_DEBUG_ID_VALUE = "PAYPAL_DEBUG_ID_VALUE";
	private static final String CURRENCY = "USD";
	private static final String PAYMENT_PROVIDER = "paymentProvider";
	private static final String PAYMENT_AMOUNT = "123.45";

	private static final BigDecimal AMOUNT = BigDecimal.valueOf(123.45);
	private String expectedCaptureResponse;
	private String expectedCaptureRequest;
	private String expectedCaptureRequestWithNullToken;

	@Spy
	@InjectMocks
	private DefaultPayPalOnDemandCaptureCommand captureCommand;
	@Mock
	private UserService userService;
	@Mock
	private CaptureRequest wrongCaptureRequest;
	@Mock
	private AuthorizationsCaptureRequest authorizationsCaptureRequest;
	@Mock
	private PayPalConfigurationService defaultPayPalConfigurationService;
	@Mock
	private PayPalEnvironment payPalEnvironment;
	@Mock
	private PayPalHttpClient client;
	@Mock
	private TransactionStatus transactionStatus;
	@Mock
	private TransactionStatusDetails transactionStatusDetails;
	@Mock
	private CaptureRequest captureRequest;
	@Mock
	private HttpResponse<Capture> httpResponse;
	@Spy
	private Map<String, TransactionStatus> transactionStatusMap = new HashMap<>() {{
		put(RESULT_STATUS, TransactionStatus.ACCEPTED);
	}};
	@Spy
	private Map<String, TransactionStatusDetails> transactionStatusDetailsMap = new HashMap<>() {{
		put(RESULT_STATUS, TransactionStatusDetails.SUCCESFULL);
	}};
	private final Currency currency = Currency.getInstance(CURRENCY);
	private CaptureRequest request;
	private Capture response;

	@Before
	public void setUp() throws Exception {
		MockitoAnnotations.initMocks(this);
		when(defaultPayPalConfigurationService.getClientID()).thenReturn(CLIENT_ID);
		when(defaultPayPalConfigurationService.getSecretKey()).thenReturn(SECRET_KEY);

		Headers headers = new Headers();
		headers.header(PAYPAL_DEBUG_ID, PAYPAL_DEBUG_ID_VALUE);

		when(httpResponse.headers()).thenReturn(headers);
		request = new CaptureRequest(MERCHANT_TRANSACTION_CODE, REQUEST_ID, TOKEN,
				currency, AMOUNT, PAYMENT_PROVIDER, SUBSCRIPTION_ID);
		response = new Capture();
		response.status(RESULT_STATUS);
		response.id(CAPTURE_ID);
		response.createTime("2021-10-08T23:37:39Z");
		response.updateTime("2021-10-08T23:37:39Z");
		Money amount = new Money();
		amount.currencyCode(CURRENCY);
		amount.value(PAYMENT_AMOUNT);
		response.amount(amount);

		expectedCaptureResponse = IOUtils.resourceToString("/test/OnDemandCaptureResponse.json", StandardCharsets.UTF_8);
		expectedCaptureRequestWithNullToken = IOUtils.resourceToString("/test/OnDemandCaptureRequestWithNullToken.json", StandardCharsets.UTF_8);
		expectedCaptureRequest = IOUtils.resourceToString("/test/OnDemandCaptureRequest.json", StandardCharsets.UTF_8);
	}

	@Test
	public void shouldReturnCaptureResultWithStatusAcceptedWhenRequestTokenIsNotNull() throws IOException {
		PayPalAbstractCommandTest.mockHttpClientToReturnResponseWith204StatusCode(captureCommand, httpResponse);
		when(httpResponse.result()).thenReturn(response);

		PayPalCaptureResult captureResult = (PayPalCaptureResult) captureCommand.perform(request);

		assertCaptureResultContainsProperData(captureResult, expectedCaptureRequest);
		verify(captureCommand).prepareRequestBody(any(CaptureRequest.class));
		verify(captureCommand, never()).preparePartialRequestBody(any(CaptureRequest.class));
	}

	@Test
	public void shouldReturnCaptureResultWithStatusAcceptedWhenRequestTokenIsNull() throws IOException {
		request = new CaptureRequest(MERCHANT_TRANSACTION_CODE, REQUEST_ID, null,
				currency, AMOUNT, PAYMENT_PROVIDER, SUBSCRIPTION_ID);
		PayPalAbstractCommandTest.mockHttpClientToReturnResponseWith204StatusCode(captureCommand, httpResponse);
		when(httpResponse.result()).thenReturn(response);

		PayPalCaptureResult captureResult = (PayPalCaptureResult) captureCommand.perform(request);

		assertCaptureResultContainsProperData(captureResult, expectedCaptureRequestWithNullToken);
		verify(captureCommand).preparePartialRequestBody(any(CaptureRequest.class));
		verify(captureCommand, never()).prepareRequestBody(any(CaptureRequest.class));
	}

	@Test(expected = PayPalCaptureAdapterException.class)
	public void shouldThrowAdapterExceptionWhenCommandExecutionFails() throws IOException {
		PayPalAbstractCommandTest.mockHttpClientToThrowIOException(captureCommand);

		PayPalCaptureResult captureResult = (PayPalCaptureResult) captureCommand.perform(request);

		assertCaptureResultContainsProperData(captureResult, expectedCaptureRequest);
		verify(captureCommand).prepareRequestBody(any(CaptureRequest.class));
		verify(captureCommand, never()).preparePartialRequestBody(any(CaptureRequest.class));
	}

	@Test(expected = IllegalArgumentException.class)
	public void shouldThrowIllegalArgumentExceptionWhenRequestIdIsNull() {
		request = new CaptureRequest(MERCHANT_TRANSACTION_CODE, null, null,
				currency, AMOUNT, PAYMENT_PROVIDER, SUBSCRIPTION_ID);

		captureCommand.perform(request);

		verify(captureCommand, never()).prepareRequestBody(any(CaptureRequest.class));
		verify(captureCommand, never()).preparePartialRequestBody(any(CaptureRequest.class));
	}

	private void assertCaptureResultContainsProperData(PayPalCaptureResult captureResult, String expectedCaptureRequest) {
		assertThat(captureResult.getTransactionStatus().name()).isEqualTo(TransactionStatus.ACCEPTED.name());
		assertThat(captureResult.getRequestId()).isEqualTo(CAPTURE_ID);
		assertThat(captureResult.getCurrency()).isEqualTo(currency);
		assertThat(captureResult.getTotalAmount()).isEqualTo(AMOUNT);
		assertThat(captureResult.getReconciliationId()).isEqualTo(SUBSCRIPTION_ID);
		assertThat(captureResult.getMerchantTransactionCode()).isEqualTo(MERCHANT_TRANSACTION_CODE);
		assertThat(captureResult.getTransactionStatusDetails()).isEqualTo(TransactionStatusDetails.SUCCESFULL);
		assertEquals(expectedCaptureRequest, captureResult.getRequestField());
		assertEquals(expectedCaptureResponse, captureResult.getResponseField());
		assertEquals(PAYPAL_DEBUG_ID_VALUE, captureResult.getDebugId());
		assertEquals(Date.from(Instant.parse("2021-10-08T23:37:39Z")), captureResult.getCreateTime());
		assertEquals(Date.from(Instant.parse("2021-10-08T23:37:39Z")), captureResult.getUpdateTime());
	}

}
