package com.paypal.hybris.core.commands.impl;

import com.paypal.http.Headers;
import com.paypal.hybris.core.commands.PayPalAbstractCommandTest;
import com.paypal.hybris.core.exception.PayPalVoidAdapterException;
import com.paypal.hybris.core.results.PayPalVoidResult;
import de.hybris.bootstrap.annotations.UnitTest;
import de.hybris.platform.payment.commands.request.VoidRequest;
import de.hybris.platform.payment.dto.TransactionStatus;
import de.hybris.platform.payment.dto.TransactionStatusDetails;
import org.apache.commons.io.IOUtils;
import org.junit.Before;
import org.junit.Test;
import org.mockito.InjectMocks;
import org.mockito.MockitoAnnotations;
import org.mockito.Spy;

import java.io.IOException;
import java.math.BigDecimal;
import java.nio.charset.StandardCharsets;
import java.util.Currency;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertThrows;

@UnitTest
public class DefaultPayPalVoidCommandTest {

    private static final String MERCHANT_TRANSACTION_CODE = "merchantTransactionCode";
    private static final String REQUEST_ID = "requestId";
    private static final String REQUEST_TOKEN = "requestToken";
    private static final String PAYMENT_PROVIDER = "paymentProvider";
    private static final String USD = "USD";
    private static final String AMOUNT = "123.45";
    private static final String EXPECTED_ILLEGAL_ARGUMENT_EXCEPTION_MESSAGE = "Request ID is undefined, actual Request ID is: null";
    private static final String ERROR_MESSAGE_FROM_IO_EXCEPTION = "testErrorMessageFromIOException";
    private static final String PAYPAL_DEBUG_ID = "Paypal-Debug-Id";
    private static final String PAYPAL_DEBUG_ID_VALUE = "PAYPAL_DEBUG_ID_VALUE";
    private String expectedReVoidRequest;

    @InjectMocks
    @Spy
    private DefaultPayPalVoidCommand unit;
    private VoidRequest voidRequest;

    @Before
    public void setUp() throws Exception {
        MockitoAnnotations.initMocks(this);
        voidRequest = new VoidRequest(MERCHANT_TRANSACTION_CODE,
                REQUEST_ID,
                REQUEST_TOKEN,
                PAYMENT_PROVIDER,
                Currency.getInstance(USD),
                new BigDecimal(AMOUNT));

        expectedReVoidRequest = IOUtils.resourceToString("/test/VoidRequest.json", StandardCharsets.UTF_8);
    }

    @Test
    public void shouldThrowIllegalArgumentExceptionWhenRequestIdIsNull() {
        voidRequest = new VoidRequest(MERCHANT_TRANSACTION_CODE,
                null,
                REQUEST_TOKEN,
                PAYMENT_PROVIDER,
                Currency.getInstance(USD),
                new BigDecimal(AMOUNT));

        assertThrows(EXPECTED_ILLEGAL_ARGUMENT_EXCEPTION_MESSAGE,
                IllegalArgumentException.class, () -> unit.perform(voidRequest));

        Headers headers = new Headers();
        headers.header(PAYPAL_DEBUG_ID, PAYPAL_DEBUG_ID_VALUE);
    }

    @Test
    public void shouldThrowPayPalVoidAdapterExceptionWhenHttpClientThrowsIOException() throws IOException {
        PayPalAbstractCommandTest.mockHttpClientToThrowIOException(unit);

        assertThrows(ERROR_MESSAGE_FROM_IO_EXCEPTION,
                PayPalVoidAdapterException.class, () -> unit.perform(voidRequest));
    }

    @Test
    public void shouldReturnRejectedTransactionStatusWhenApiReturnsNon204StatusCode() throws IOException {
        PayPalAbstractCommandTest.mockHttpClientToReturnResponseWith200StatusCode(unit);

        PayPalVoidResult voidResult = (PayPalVoidResult) unit.perform(voidRequest);

        assertEquals(TransactionStatus.REJECTED, voidResult.getTransactionStatus());
        assertEquals(TransactionStatusDetails.PROCESSOR_DECLINE, voidResult.getTransactionStatusDetails());
        assertEquals(voidRequest.getMerchantTransactionCode(), voidResult.getMerchantTransactionCode());
        assertEquals(voidRequest.getRequestId(), voidResult.getRequestId());
        assertEquals(voidRequest.getTotalAmount(), voidResult.getAmount());
        assertEquals(voidRequest.getCurrency(), voidResult.getCurrency());
        assertEquals(PAYPAL_DEBUG_ID_VALUE, voidResult.getDebugId());
        assertEquals(expectedReVoidRequest, voidResult.getRequestField());
    }

    @Test
    public void shouldReturnAcceptedTransactionStatusWhenApiReturns204StatusCode() throws IOException {
        PayPalAbstractCommandTest.mockHttpClientToReturnResponseWith204StatusCode(unit);

        PayPalVoidResult voidResult = (PayPalVoidResult) unit.perform(voidRequest);

        assertEquals(TransactionStatus.ACCEPTED, voidResult.getTransactionStatus());
        assertEquals(TransactionStatusDetails.SUCCESFULL, voidResult.getTransactionStatusDetails());
        assertEquals(voidRequest.getMerchantTransactionCode(), voidResult.getMerchantTransactionCode());
        assertEquals(voidRequest.getRequestId(), voidResult.getRequestId());
        assertEquals(voidRequest.getTotalAmount(), voidResult.getAmount());
        assertEquals(voidRequest.getCurrency(), voidResult.getCurrency());
        assertEquals(PAYPAL_DEBUG_ID_VALUE, voidResult.getDebugId());
        assertEquals(expectedReVoidRequest, voidResult.getRequestField());
    }
}
