package com.paypal.hybris.core.commands.impl;

import com.paypal.core.AuthorizationProvider;
import com.paypal.core.PayPalHttpClient;
import com.paypal.core.object.AccessToken;
import com.paypal.hybris.core.commands.PayPalAbstractCommandTest;
import com.paypal.hybris.data.AmountWithBreakdownData;
import com.paypal.hybris.data.BreakdownData;
import com.paypal.hybris.data.Level2Level3CardData;
import com.paypal.hybris.data.OrderItemData;
import com.paypal.hybris.data.PayPalIntegrationPatch;
import com.paypal.hybris.data.PayPalOrderRequestData;
import com.paypal.hybris.data.SimpleAmount;
import com.paypal.hybris.data.SupplementaryData;
import de.hybris.bootstrap.annotations.UnitTest;
import de.hybris.platform.payment.AdapterException;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.MockedStatic;
import org.mockito.Mockito;
import org.mockito.Spy;
import org.mockito.junit.MockitoJUnitRunner;
import org.springframework.http.HttpEntity;
import org.springframework.web.client.RestClientException;
import org.springframework.web.client.RestTemplate;

import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;

import static org.junit.Assert.assertThrows;
import static org.junit.Assert.assertTrue;
import static org.mockito.ArgumentMatchers.eq;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;
import static org.assertj.core.api.Assertions.assertThat;

@UnitTest
@RunWith(MockitoJUnitRunner.StrictStubs.class)
public class DefaultPayPalUpdateOrderAmountCommandTest {

    private static final String ID = "id";
    private static final String ORDER_ID = "orderId";
    private static final String USD = "USD";
    private static final String AMOUNT = "123.45";
    private static final String ITEMS_AMOUNT = "100.00";
    private static final String SHIPPING_AMOUNT = "23.45";
    private static final String DISCOUNT_AMOUNT = "10.00";
    private static final String TAX_AMOUNT = "5.00";
    private static final String REPLACE = "replace";
    private static final String AMOUNT_PATH = "/purchase_units/@reference_id=='default'/amount";
    private static final String ITEMS_PATH = "/purchase_units/@reference_id=='default'/items";
    private static final String L2_L3_DATA_PATH = "/purchase_units/@reference_id=='default'/supplementary_data/card";
    private static final String ITEM_NAME = "Product";
    private static final String ITEM_CATEGORY = "Category";
    private static final String ITEM_SKU = "sku";
    private static final String ITEM_DESCRIPTION = "Description";
    private static final String URL = "url";
    private static final String ACCESS_TOKEN = "access_token";
    private static final String SANDBOX_ORDERS_URL = "https://api.sandbox.paypal.com/v2/checkout/orders/{id}";
    private static final String ERROR_MESSAGE_FROM_IO_EXCEPTION = "testErrorMessageFromIOException";
    private static final String API_ERROR_MESSAGE_JSON = "{details:[{description:testErrorMessageFromIOException}]}";


    @InjectMocks
    @Spy
    private DefaultPayPalUpdateOrderAmountCommand unit;
    @Mock
    private AuthorizationProvider authorizationProvider;
    @Mock
    private AccessToken accessToken;
    @Mock
    private RestTemplate restTemplate;
    private HashMap<String, String> hashMap;
    private PayPalOrderRequestData orderRequestData;
    private Level2Level3CardData level2Level3CardData;

    @Before
    public void setUp() throws Exception {
        orderRequestData = new PayPalOrderRequestData();
        level2Level3CardData = new Level2Level3CardData();
        SupplementaryData supplementaryData = new SupplementaryData();
        supplementaryData.setCard(level2Level3CardData);
        orderRequestData.setAmount(AMOUNT);
        orderRequestData.setCurrency(USD);
        orderRequestData.setOrderId(ORDER_ID);
        orderRequestData.setShippingAmount(SHIPPING_AMOUNT);
//        orderRequestData.setDiscountAmount(DISCOUNT_AMOUNT);
//        orderRequestData.setTaxAmount(TAX_AMOUNT);
        orderRequestData.setSupplementaryData(supplementaryData);

        when(accessToken.accessToken()).thenReturn(ACCESS_TOKEN);

        hashMap = new HashMap<>();
        hashMap.put(ID, ORDER_ID);

        when(restTemplate.patchForObject(eq(SANDBOX_ORDERS_URL), any(HttpEntity.class),
                eq(Boolean.class), eq(hashMap))).thenReturn(true);
    }

    @Test
    public void shouldReturnTrueWhenExceptionDoesNotOccur() throws IOException {
        PayPalAbstractCommandTest.mockCreatePayPalEnvironmentMethodToReturnSandboxEnv(unit);

        try (MockedStatic<AuthorizationProvider> authorizationProviderMockedStatic = Mockito.mockStatic(AuthorizationProvider.class)) {
            when(authorizationProvider.authorize(any(PayPalHttpClient.class), any())).thenReturn(accessToken);
            authorizationProviderMockedStatic.when(AuthorizationProvider::sharedInstance).thenReturn(authorizationProvider);

            assertTrue(unit.perform(orderRequestData));
            verify(restTemplate).patchForObject(eq(SANDBOX_ORDERS_URL), any(HttpEntity.class),
                    eq(Boolean.class), eq(hashMap));
        }
    }

    @Test
    public void shouldThrowAdapterExceptionWhenRequestThrowsIOException() throws IOException {
        PayPalAbstractCommandTest.mockCreatePayPalEnvironmentMethodToReturnSandboxEnv(unit);

        try (MockedStatic<AuthorizationProvider> authorizationProviderMockedStatic = Mockito.mockStatic(AuthorizationProvider.class)) {
            when(authorizationProvider.authorize(any(PayPalHttpClient.class), any())).thenReturn(accessToken);
            authorizationProviderMockedStatic.when(AuthorizationProvider::sharedInstance).thenReturn(authorizationProvider);
            RestClientException ioException = new RestClientException(API_ERROR_MESSAGE_JSON);
            when(restTemplate.patchForObject(eq(SANDBOX_ORDERS_URL), any(HttpEntity.class),
                    eq(Boolean.class), eq(hashMap))).thenThrow(ioException);

            assertThrows(ERROR_MESSAGE_FROM_IO_EXCEPTION, AdapterException.class, () -> unit.perform(orderRequestData));
        }
    }

    @Test
    public void shouldCreateListOfPayPalIntegrationPatchWithBreakdownDataAndOrderItems() {
        List<OrderItemData> itemsList = new ArrayList<>();
        OrderItemData item = createOrderItem();

        itemsList.add(item);
        orderRequestData.setOrderItems(itemsList);
        orderRequestData.setBreakdownAmountData(createAmountWithBreakdownData());

        PayPalIntegrationPatch amountPatch = createReplacePatch(AMOUNT_PATH, createAmountWithBreakdownData());
        PayPalIntegrationPatch itemsPatch = createReplacePatch(ITEMS_PATH, itemsList);
        PayPalIntegrationPatch l2L3DataPatch = createReplacePatch(L2_L3_DATA_PATH, level2Level3CardData);

        List<PayPalIntegrationPatch> patchList = unit.preparePatches(orderRequestData);

        assertThat(patchList).hasSize(3);
        assertThat(patchList.get(0)).isEqualToComparingFieldByFieldRecursively(amountPatch);
        assertThat(patchList.get(1)).isEqualToComparingFieldByFieldRecursively(itemsPatch);
        assertThat(patchList.get(2)).isEqualToComparingFieldByFieldRecursively(l2L3DataPatch);
    }

    private OrderItemData createOrderItem() {
        SimpleAmount simpleAmount = createSimpleAmount(AMOUNT);

        OrderItemData item = new OrderItemData();
        item.setName(ITEM_NAME);
        item.setQuantity(2);
        item.setAmount(simpleAmount);
        item.setCategory(ITEM_CATEGORY);
        item.setSku(ITEM_SKU);
        item.setDescription(ITEM_DESCRIPTION);
        item.setUrl(URL);
        item.setImageUrl(URL);
        return item;
    }

    private AmountWithBreakdownData createAmountWithBreakdownData() {
        BreakdownData breakdownData = new BreakdownData();
        breakdownData.setItemTotal(createSimpleAmount(ITEMS_AMOUNT));
        breakdownData.setTax(createSimpleAmount(TAX_AMOUNT));
        breakdownData.setDiscount(createSimpleAmount(DISCOUNT_AMOUNT));
        breakdownData.setShipping(createSimpleAmount(SHIPPING_AMOUNT));

        AmountWithBreakdownData amountWithBreakdownData = new AmountWithBreakdownData();
        amountWithBreakdownData.setCurrencyCode(USD);
        amountWithBreakdownData.setBreakdown(breakdownData);
        amountWithBreakdownData.setValue(AMOUNT);

        return amountWithBreakdownData;
    }

    private SimpleAmount createSimpleAmount(String value) {
        SimpleAmount simpleAmount = new SimpleAmount();
        simpleAmount.setValue(value);
        simpleAmount.setCurrencyCode(USD);
        return simpleAmount;
    }

    private PayPalIntegrationPatch createReplacePatch(String path, Object value) {
        PayPalIntegrationPatch patch = new PayPalIntegrationPatch();
        patch.setOp(REPLACE);
        patch.setPath(path);
        patch.setValue(value);
        return patch;
    }

}
