package com.paypal.hybris.core.commands.impl;

import com.paypal.base.rest.PayPalRESTException;
import com.paypal.hybris.core.commands.PayPalAbstractCommandTest;
import com.paypal.hybris.core.exception.PayPalProcessHttpClientErrorException;
import com.paypal.hybris.core.exception.PayPalProcessPaymentException;
import com.paypal.hybris.core.service.PayPalConfigurationService;
import com.paypal.hybris.data.AmountWithBreakdownData;
import com.paypal.hybris.data.PayPalOrderResponseData;
import com.paypal.hybris.data.PayPalPaymentCollectionData;
import com.paypal.hybris.data.PayPalPaymentData;
import com.paypal.hybris.data.PurchaseUnitData;
import de.hybris.bootstrap.annotations.UnitTest;
import de.hybris.platform.core.model.order.CartModel;
import de.hybris.platform.order.CartService;
import de.hybris.platform.payment.commands.request.SubscriptionAuthorizationRequest;
import de.hybris.platform.payment.commands.result.AuthorizationResult;
import de.hybris.platform.payment.dto.BillingInfo;
import de.hybris.platform.payment.dto.TransactionStatus;
import de.hybris.platform.payment.dto.TransactionStatusDetails;
import de.hybris.platform.store.BaseStoreModel;
import de.hybris.platform.store.services.BaseStoreService;
import org.apache.commons.lang.StringUtils;
import org.junit.Before;
import org.junit.Test;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;
import org.mockito.Spy;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.util.MultiValueMap;
import org.springframework.web.client.HttpClientErrorException;
import org.springframework.web.client.RestTemplate;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Currency;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import static com.paypal.hybris.core.constants.PaypalcoreConstants.CHECKOUT_URL;
import static com.paypal.hybris.core.constants.PaypalcoreConstants.PAYPAL_CHECKOUT_PAYMENT_ERROR_MSG;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertThrows;
import static org.mockito.Matchers.eq;
import static org.mockito.Matchers.any;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

@UnitTest
public class DefaultPayPalSubscriptionAuthorizationCommandTest {

    private static final String MERCHANT_TRANSACTION_CODE = "merchantTransactionCode";
    private static final String PAYMENT_PROVIDER = "paymentProvider";
    private static final String USD = "USD";
    private static final String AMOUNT = "123.45";
    private static final BigDecimal TOTAL_AMOUNT_BIG_DECIMAL = new BigDecimal(AMOUNT);
    private static final String SUBSCRIPTION_ID = "subscriptionId";
    private static final String CV_2 = "cv2";
    private static final String API_URL = "https://api.sandbox.paypal.com" + CHECKOUT_URL + SUBSCRIPTION_ID + "/authorize";
    private static final String PAY_PAL_PAYMENT_DATA_ID = "payPalPaymentDataId";
    private static final String CREATED = "CREATED";
    private static final String PURCHASE_UNIT_DATA_ID = "purchaseUnitDataId";
    private static final String DEFAULT = "default";
    private static final String COMPLETED = "COMPLETED";
    private static final String SAVED = "SAVED";
    private static final String PENDING = "PENDING";
    private static final String DECLINED = "DECLINED";
    private static final String PAY_PAL_CLIENT_META_DATA_ID = "payPalClientMetaDataId";
    private static final String PAYPAL_DEBUG_ID = "Paypal-Debug-Id";
    private static final String PAYPAL_DEBUG_ID_VALUE = "PAYPAL_DEBUG_ID_VALUE";

    @InjectMocks
    @Spy
    private DefaultPayPalSubscriptionAuthorizationCommand unit;
    @Mock
    private BaseStoreService baseStoreService;
    @Mock
    private BaseStoreModel baseStoreModel;
    @Mock
    private PayPalConfigurationService payPalConfigurationService;
    @Mock
    private RestTemplate restTemplate;
    @Mock
    private CartService cartService;
    @Mock
    private CartModel cartModel;
    private SubscriptionAuthorizationRequest request;

    @Before
    public void setUp() throws Exception {
        MockitoAnnotations.initMocks(this);
        Map<String, TransactionStatusDetails> transactionStatusDetailsMap = new HashMap<>();
        transactionStatusDetailsMap.put(COMPLETED, TransactionStatusDetails.SUCCESFULL);
        transactionStatusDetailsMap.put(SAVED, TransactionStatusDetails.SUCCESFULL);
        transactionStatusDetailsMap.put(PENDING, TransactionStatusDetails.REVIEW_NEEDED);
        transactionStatusDetailsMap.put(DECLINED, TransactionStatusDetails.PROCESSOR_DECLINE);

        Map<String, TransactionStatus> transactionStatusMap = new HashMap<>();
        transactionStatusMap.put(COMPLETED, TransactionStatus.ACCEPTED);
        transactionStatusMap.put(SAVED, TransactionStatus.ACCEPTED);
        transactionStatusMap.put(PENDING, TransactionStatus.REVIEW);
        transactionStatusMap.put(DECLINED, TransactionStatus.REJECTED);

        unit.setTransactionStatusDetailsMap(transactionStatusDetailsMap);
        unit.setTransactionStatusMap(transactionStatusMap);

        request = new SubscriptionAuthorizationRequest(MERCHANT_TRANSACTION_CODE, SUBSCRIPTION_ID,
                Currency.getInstance(USD), TOTAL_AMOUNT_BIG_DECIMAL,
                new BillingInfo(), CV_2, PAYMENT_PROVIDER);

        PayPalAbstractCommandTest.mockCreatePayPalEnvironmentMethodToReturnSandboxEnv(unit);
        PayPalAbstractCommandTest.mockCreateApiContextMethodToReturnSandboxApiContext(unit);

        when(baseStoreService.getCurrentBaseStore()).thenReturn(baseStoreModel);
        when(payPalConfigurationService.isFraudNetEnabled()).thenReturn(false);
    }

    @Test
    public void shouldSendRequestWithoutException() {
        ResponseEntity<PayPalOrderResponseData> responseEntity = getResponseEntity();
        when(restTemplate.postForEntity(eq(API_URL), any(HttpEntity.class), eq(PayPalOrderResponseData.class)))
                .thenReturn(responseEntity);

        AuthorizationResult result = unit.perform(request);

        verify(restTemplate).postForEntity(eq(API_URL), any(HttpEntity.class), eq(PayPalOrderResponseData.class));

        assertEquals(PAY_PAL_PAYMENT_DATA_ID, result.getAuthorizationCode());
        assertEquals(TransactionStatus.ACCEPTED, result.getTransactionStatus());
        assertEquals(TransactionStatusDetails.SUCCESFULL, result.getTransactionStatusDetails());
    }

    @Test
    public void shouldSendRequestWithoutExceptionWhenCv2IsNull() {
        request = new SubscriptionAuthorizationRequest(MERCHANT_TRANSACTION_CODE, SUBSCRIPTION_ID,
                Currency.getInstance(USD), TOTAL_AMOUNT_BIG_DECIMAL,
                new BillingInfo(), null, PAYMENT_PROVIDER);

        ResponseEntity<PayPalOrderResponseData> responseEntity = getResponseEntity();
        when(restTemplate.postForEntity(eq(API_URL), any(HttpEntity.class), eq(PayPalOrderResponseData.class)))
                .thenReturn(responseEntity);

        AuthorizationResult result = unit.perform(request);

        verify(restTemplate).postForEntity(eq(API_URL), any(HttpEntity.class), eq(PayPalOrderResponseData.class));

        assertEquals(PAY_PAL_PAYMENT_DATA_ID, result.getAuthorizationCode());
        assertEquals(TransactionStatus.ACCEPTED, result.getTransactionStatus());
        assertEquals(TransactionStatusDetails.SUCCESFULL, result.getTransactionStatusDetails());
    }

    @Test
    public void shouldThrowPayPalProcessHttpClientErrorExceptionWhenHttpClientErrorExceptionOccurs() {
        when(restTemplate.postForEntity(eq(API_URL), any(HttpEntity.class), eq(PayPalOrderResponseData.class)))
                .thenThrow(HttpClientErrorException.class);

        assertThrows(PAYPAL_CHECKOUT_PAYMENT_ERROR_MSG, PayPalProcessHttpClientErrorException.class,
                () -> unit.perform(request));
        verify(restTemplate).postForEntity(eq(API_URL), any(HttpEntity.class), eq(PayPalOrderResponseData.class));
    }

    @Test(expected = IllegalArgumentException.class)
    public void shouldThrowIllegalArgumentExceptionWhenPaymentsAreEmpty() {
        PayPalOrderResponseData orderResponseData = getPayPalOrderResponseData();
        ResponseEntity<PayPalOrderResponseData> responseEntity = new ResponseEntity<>(orderResponseData, HttpStatus.OK);
        when(restTemplate.postForEntity(eq(API_URL), any(HttpEntity.class), eq(PayPalOrderResponseData.class)))
                .thenReturn(responseEntity);
        orderResponseData.getPurchaseUnits().stream().findFirst().ifPresent(unit -> unit.getPayments().setAuthorizations(null));

        unit.perform(request);
    }

    @Test(expected = PayPalProcessPaymentException.class)
    public void shouldThrowPayPalProcessPaymentExceptionWhenPaymentsAreNotCreated() {
        PayPalOrderResponseData orderResponseData = getPayPalOrderResponseData();
        ResponseEntity<PayPalOrderResponseData> responseEntity = new ResponseEntity<>(orderResponseData, HttpStatus.OK);
        when(restTemplate.postForEntity(eq(API_URL), any(HttpEntity.class), eq(PayPalOrderResponseData.class)))
                .thenReturn(responseEntity);
        orderResponseData.getPurchaseUnits().stream().findFirst()
                .flatMap(unit -> unit.getPayments().getAuthorizations().stream().findFirst())
                .ifPresent(payPalPaymentData -> payPalPaymentData.setStatus(StringUtils.EMPTY));

        unit.perform(request);
    }

    @Test
    public void shouldSendRequestIfPayPalRESTExceptionOccursWhenCreatingApiContext() throws PayPalRESTException {
        PayPalAbstractCommandTest.mockApiContextFetchAccessTokenMethodToThrowPayPalRESTException(unit);

        ResponseEntity<PayPalOrderResponseData> responseEntity = getResponseEntity();
        when(restTemplate.postForEntity(eq(API_URL), any(HttpEntity.class), eq(PayPalOrderResponseData.class)))
                .thenReturn(responseEntity);

        AuthorizationResult result = unit.perform(request);

        verify(restTemplate).postForEntity(eq(API_URL), any(HttpEntity.class), eq(PayPalOrderResponseData.class));

        assertEquals(PAY_PAL_PAYMENT_DATA_ID, result.getAuthorizationCode());
        assertEquals(TransactionStatus.ACCEPTED, result.getTransactionStatus());
        assertEquals(TransactionStatusDetails.SUCCESFULL, result.getTransactionStatusDetails());
    }

    @Test
    public void shouldSendRequestWithoutExceptionWhenFraudNetIsEnabled() {
        when(payPalConfigurationService.isFraudNetEnabled()).thenReturn(true);
        when(cartService.getSessionCart()).thenReturn(cartModel);
        when(cartModel.getPayPalClientMetadataId()).thenReturn(PAY_PAL_CLIENT_META_DATA_ID);

        ResponseEntity<PayPalOrderResponseData> responseEntity = getResponseEntity();
        when(restTemplate.postForEntity(eq(API_URL), any(HttpEntity.class), eq(PayPalOrderResponseData.class)))
                .thenReturn(responseEntity);

        AuthorizationResult result = unit.perform(request);

        verify(restTemplate).postForEntity(eq(API_URL), any(HttpEntity.class), eq(PayPalOrderResponseData.class));

        assertEquals(PAY_PAL_PAYMENT_DATA_ID, result.getAuthorizationCode());
        assertEquals(TransactionStatus.ACCEPTED, result.getTransactionStatus());
        assertEquals(TransactionStatusDetails.SUCCESFULL, result.getTransactionStatusDetails());
    }

    private ResponseEntity<PayPalOrderResponseData> getResponseEntity() {
        MultiValueMap<String, String> headers = new HttpHeaders();
        headers.put(PAYPAL_DEBUG_ID, List.of(PAYPAL_DEBUG_ID_VALUE));
        PayPalOrderResponseData orderResponseData = getPayPalOrderResponseData();
        return new ResponseEntity<>(orderResponseData, headers, HttpStatus.OK);
    }

    private PayPalOrderResponseData getPayPalOrderResponseData() {
        PayPalOrderResponseData orderResponseData = new PayPalOrderResponseData();

        PurchaseUnitData purchaseUnitData = new PurchaseUnitData();

        AmountWithBreakdownData amountWithBreakdownData = new AmountWithBreakdownData();
        amountWithBreakdownData.setCurrencyCode(USD);
        amountWithBreakdownData.setValue(AMOUNT);

        ArrayList<PayPalPaymentData> payPalPaymentDataList = new ArrayList<>();
        PayPalPaymentData palPaymentData = new PayPalPaymentData();
        palPaymentData.setAmount(amountWithBreakdownData);
        palPaymentData.setId(PAY_PAL_PAYMENT_DATA_ID);
        palPaymentData.setStatus(CREATED);
        payPalPaymentDataList.add(palPaymentData);

        PayPalPaymentCollectionData payPalPaymentCollectionData = new PayPalPaymentCollectionData();
        payPalPaymentCollectionData.setAuthorizations(payPalPaymentDataList);

        purchaseUnitData.setAmount(amountWithBreakdownData);
        purchaseUnitData.setId(PURCHASE_UNIT_DATA_ID);
        purchaseUnitData.setReferenceId(DEFAULT);
        purchaseUnitData.setPayments(payPalPaymentCollectionData);

        ArrayList<PurchaseUnitData> purchaseUnitDataList = new ArrayList<>();
        purchaseUnitDataList.add(purchaseUnitData);
        orderResponseData.setPurchaseUnits(purchaseUnitDataList);
        orderResponseData.setStatus(COMPLETED);
        return orderResponseData;
    }

}
