package com.paypal.hybris.core.commands.impl;

import com.paypal.hybris.core.commands.PayPalAbstractCommandTest;
import com.paypal.hybris.data.AmountWithBreakdownData;
import com.paypal.hybris.data.PayPalPaymentCollectionData;
import com.paypal.hybris.data.PayPalPaymentData;
import com.paypal.hybris.data.PayPalSaveOrderResponseData;
import com.paypal.hybris.data.PurchaseUnitData;
import de.hybris.bootstrap.annotations.UnitTest;
import de.hybris.platform.payment.AdapterException;
import org.junit.Before;
import org.junit.Test;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;
import org.mockito.Spy;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.web.client.RestTemplate;

import java.util.ArrayList;

import static com.paypal.hybris.core.constants.PaypalcoreConstants.CHECKOUT_URL;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertThrows;
import static org.mockito.Matchers.eq;
import static org.mockito.Matchers.any;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

@UnitTest
public class DefaultPayPalSaveOrderCommandTest {
    private static final String USD = "USD";
    private static final String AMOUNT = "123.45";
    private static final String SUBSCRIPTION_ID = "subscriptionId";
    private static final String API_URL = "https://api.sandbox.paypal.com" + CHECKOUT_URL + SUBSCRIPTION_ID + "/save";
    private static final String PAY_PAL_PAYMENT_DATA_ID = "payPalPaymentDataId";
    private static final String CREATED = "CREATED";
    private static final String PURCHASE_UNIT_DATA_ID = "purchaseUnitDataId";
    private static final String DEFAULT = "default";
    private static final String COMPLETED = "COMPLETED";
    private static final String MESSAGE_FROM_RUNTIME_EXCEPTION = "Message from RuntimeException";

    @InjectMocks
    @Spy
    private DefaultPayPalSaveOrderCommand unit;
    @Mock
    private RestTemplate restTemplate;

    @Before
    public void setUp() throws Exception {
        MockitoAnnotations.initMocks(this);
    }

    @Test
    public void shouldReturnPurchaseUnitIdWhenPostedSuccessfully() {
        PayPalAbstractCommandTest.mockCreatePayPalEnvironmentMethodToReturnSandboxEnv(unit);
        PayPalAbstractCommandTest.mockCreateApiContextMethodToReturnSandboxApiContext(unit);

        PayPalSaveOrderResponseData saveOrderResponseData = getPayPalSaveOrderResponseData();
        ResponseEntity<PayPalSaveOrderResponseData> responseEntity = new ResponseEntity<>(saveOrderResponseData, HttpStatus.OK);
        when(restTemplate.postForEntity(eq(API_URL), any(HttpEntity.class), eq(PayPalSaveOrderResponseData.class)))
                .thenReturn(responseEntity);

        String result = unit.perform(SUBSCRIPTION_ID);
        assertEquals(PURCHASE_UNIT_DATA_ID, result);
    }

    @Test
    public void shouldThrowAdapterExceptionExceptionWhenPurchaseUnitsAreEmpty() {
        PayPalAbstractCommandTest.mockCreatePayPalEnvironmentMethodToReturnSandboxEnv(unit);
        PayPalAbstractCommandTest.mockCreateApiContextMethodToReturnSandboxApiContext(unit);

        PayPalSaveOrderResponseData orderResponseData = new PayPalSaveOrderResponseData();
        ArrayList<PurchaseUnitData> purchaseUnitDataList = new ArrayList<>();
        orderResponseData.setPurchaseUnits(purchaseUnitDataList);

        ResponseEntity<PayPalSaveOrderResponseData> responseEntity = new ResponseEntity<>(orderResponseData, HttpStatus.OK);
        when(restTemplate.postForEntity(eq(API_URL), any(HttpEntity.class), eq(PayPalSaveOrderResponseData.class)))
                .thenReturn(responseEntity);

        assertThrows(MESSAGE_FROM_RUNTIME_EXCEPTION, AdapterException.class,
                () -> unit.perform(SUBSCRIPTION_ID));
        verify(restTemplate).postForEntity(eq(API_URL), any(HttpEntity.class), eq(PayPalSaveOrderResponseData.class));
    }

    @Test
    public void shouldThrowAdapterExceptionWhenRestTemplateThrowsAnyException() {
        PayPalAbstractCommandTest.mockCreatePayPalEnvironmentMethodToReturnSandboxEnv(unit);
        PayPalAbstractCommandTest.mockCreateApiContextMethodToReturnSandboxApiContext(unit);

        when(restTemplate.postForEntity(eq(API_URL), any(HttpEntity.class), eq(PayPalSaveOrderResponseData.class)))
                .thenThrow(new RuntimeException(MESSAGE_FROM_RUNTIME_EXCEPTION));

        assertThrows(MESSAGE_FROM_RUNTIME_EXCEPTION, AdapterException.class,
                () -> unit.perform(SUBSCRIPTION_ID));
        verify(restTemplate).postForEntity(eq(API_URL), any(HttpEntity.class), eq(PayPalSaveOrderResponseData.class));
    }

    private PayPalSaveOrderResponseData getPayPalSaveOrderResponseData() {
        PayPalSaveOrderResponseData orderResponseData = new PayPalSaveOrderResponseData();

        PurchaseUnitData purchaseUnitData = new PurchaseUnitData();

        AmountWithBreakdownData amountWithBreakdownData = new AmountWithBreakdownData();
        amountWithBreakdownData.setCurrencyCode(USD);
        amountWithBreakdownData.setValue(AMOUNT);

        ArrayList<PayPalPaymentData> payPalPaymentDataList = new ArrayList<>();
        PayPalPaymentData palPaymentData = new PayPalPaymentData();
        palPaymentData.setAmount(amountWithBreakdownData);
        palPaymentData.setId(PAY_PAL_PAYMENT_DATA_ID);
        palPaymentData.setStatus(CREATED);
        payPalPaymentDataList.add(palPaymentData);

        PayPalPaymentCollectionData payPalPaymentCollectionData = new PayPalPaymentCollectionData();
        payPalPaymentCollectionData.setAuthorizations(payPalPaymentDataList);

        purchaseUnitData.setAmount(amountWithBreakdownData);
        purchaseUnitData.setId(DefaultPayPalSaveOrderCommandTest.PURCHASE_UNIT_DATA_ID);
        purchaseUnitData.setReferenceId(DEFAULT);
        purchaseUnitData.setPayments(payPalPaymentCollectionData);

        ArrayList<PurchaseUnitData> purchaseUnitDataList = new ArrayList<>();
        purchaseUnitDataList.add(purchaseUnitData);
        orderResponseData.setPurchaseUnits(purchaseUnitDataList);
        orderResponseData.setStatus(COMPLETED);
        return orderResponseData;
    }

}
