package com.paypal.hybris.core.commands.impl;

import com.paypal.http.Headers;
import com.paypal.http.HttpResponse;
import com.paypal.hybris.core.commands.PayPalAbstractCommandTest;
import com.paypal.hybris.core.exception.PayPalReauthorizationTooSoonException;
import com.paypal.hybris.core.exception.PayPalReauthorizeAdapterException;
import com.paypal.hybris.core.results.PayPalAuthorizationResult;
import com.paypal.payments.Authorization;
import de.hybris.bootstrap.annotations.UnitTest;
import de.hybris.platform.payment.AdapterException;
import de.hybris.platform.payment.commands.request.SubscriptionAuthorizationRequest;
import de.hybris.platform.payment.dto.BillingInfo;
import de.hybris.platform.payment.dto.TransactionStatus;
import de.hybris.platform.payment.dto.TransactionStatusDetails;
import org.apache.commons.io.IOUtils;
import org.junit.Before;
import org.junit.Test;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;
import org.mockito.Spy;

import java.io.IOException;
import java.math.BigDecimal;
import java.nio.charset.StandardCharsets;
import java.time.Instant;
import java.util.Currency;
import java.util.Date;

import static org.junit.Assert.assertEquals;
import static org.mockito.Mockito.when;

@UnitTest
public class DefaultPayPalReauthorizationRequestCommandTest {

    private static final String MERCHANT_TRANSACTION_CODE = "merchantTransactionCode";
    private static final String USD = "USD";
    private static final String AMOUNT = "123.45";
    private static final String CREATED = "CREATED";
    private static final String NOT_CREATED = "NOT_CREATED";
    private static final String PAYMENT_PROVIDER = "paymentProvider";
    private static final BigDecimal TOTAL_AMOUNT_BIG_DECIMAL = new BigDecimal(AMOUNT);
    private static final String SUBSCRIPTION_ID = "subscriptionId";
    private static final String CV_2 = "cv2";
    private static final String REAUTHORIZATION_TOO_SOON = "REAUTHORIZATION_TOO_SOON";
    private static final String AUTHORIZATION_ID = "authorizationId";
    private static final String PAYPAL_DEBUG_ID = "Paypal-Debug-Id";
    private static final String PAYPAL_DEBUG_ID_VALUE = "PAYPAL_DEBUG_ID_VALUE";
    private String expectedReAuthResponse;
    private String expectedReAuthRequest;
    private String expectedReAuthResponseNotCreated;

    @InjectMocks
    @Spy
    private DefaultPayPalReauthorizationRequestCommand unit;
    @Mock
    private HttpResponse<Authorization> httpResponse;
    private SubscriptionAuthorizationRequest request;
    private Authorization authorization;

    @Before
    public void setUp() throws Exception {
        MockitoAnnotations.initMocks(this);

        request = new SubscriptionAuthorizationRequest(MERCHANT_TRANSACTION_CODE, SUBSCRIPTION_ID,
                Currency.getInstance(USD), TOTAL_AMOUNT_BIG_DECIMAL,
                new BillingInfo(), CV_2, PAYMENT_PROVIDER);

        authorization = new Authorization();
        authorization.id(AUTHORIZATION_ID);
        authorization.status(CREATED);
        authorization.expirationTime("2021-10-08T23:37:39Z");
        when(httpResponse.result()).thenReturn(authorization);

        Headers headers = new Headers();
        headers.header(PAYPAL_DEBUG_ID, PAYPAL_DEBUG_ID_VALUE);

        when(httpResponse.headers()).thenReturn(headers);

        expectedReAuthResponse = IOUtils.resourceToString("/test/ReAuthorizationResponse.json", StandardCharsets.UTF_8);
        expectedReAuthResponseNotCreated = IOUtils.resourceToString("/test/ReAuthResponseNotCreated.json", StandardCharsets.UTF_8);
        expectedReAuthRequest = IOUtils.resourceToString("/test/ReAuthorizationRequest.json", StandardCharsets.UTF_8);
    }

    @Test(expected = IllegalArgumentException.class)
    public void shouldThrowIllegalArgumentExceptionWhenSubscriptionIdIsNull() {
        request = new SubscriptionAuthorizationRequest(MERCHANT_TRANSACTION_CODE, null,
                Currency.getInstance(USD), TOTAL_AMOUNT_BIG_DECIMAL,
                new BillingInfo(), CV_2, PAYMENT_PROVIDER);

        unit.perform(request);
    }

    @Test(expected = PayPalReauthorizeAdapterException.class)
    public void shouldThrowPayPalReauthorizeAdapterExceptionWhenIOExceptionOccurs() throws IOException {
        PayPalAbstractCommandTest.mockHttpClientToThrowIOException(unit);

        unit.perform(request);
    }

    @Test(expected = PayPalReauthorizeAdapterException.class)
    public void shouldThrowPayPalReauthorizeAdapterExceptionWhenIOExceptionWithAuthorizationTooSoonMessageOccurs() throws IOException {
        PayPalAbstractCommandTest.mockHttpClientToThrowIOExceptionWithIssueMessage(unit, REAUTHORIZATION_TOO_SOON);

        unit.perform(request);
    }

    @Test
    public void shouldReturnTransactionStatusAccepted() throws IOException {
        PayPalAbstractCommandTest.mockHttpClientToReturnResponseWith204StatusCode(unit, httpResponse);

        PayPalAuthorizationResult authorizationResult = (PayPalAuthorizationResult) unit.perform(request);

        assertEquals(TransactionStatus.ACCEPTED, authorizationResult.getTransactionStatus());
        assertEquals(TransactionStatusDetails.SUCCESFULL, authorizationResult.getTransactionStatusDetails());
        assertEquals(request.getTotalAmount(), authorizationResult.getTotalAmount());
        assertEquals(expectedReAuthRequest, authorizationResult.getRequestField());
        assertEquals(expectedReAuthResponse, authorizationResult.getResponseField());
        assertEquals(PAYPAL_DEBUG_ID_VALUE, authorizationResult.getDebugId());
        assertEquals(Date.from(Instant.parse("2021-10-08T23:37:39Z")), authorizationResult.getExpirationTime());
    }

    @Test
    public void shouldReturnTransactionStatusRejected() throws IOException {
        PayPalAbstractCommandTest.mockHttpClientToReturnResponseWith204StatusCode(unit, httpResponse);
        authorization.status(NOT_CREATED);

        PayPalAuthorizationResult authorizationResult = (PayPalAuthorizationResult) unit.perform(request);

        assertEquals(TransactionStatus.REJECTED, authorizationResult.getTransactionStatus());
        assertEquals(TransactionStatusDetails.PROCESSOR_DECLINE, authorizationResult.getTransactionStatusDetails());
        assertEquals(request.getTotalAmount(), authorizationResult.getTotalAmount());
        assertEquals(expectedReAuthRequest, authorizationResult.getRequestField());
        assertEquals(expectedReAuthResponseNotCreated, authorizationResult.getResponseField());
        assertEquals(PAYPAL_DEBUG_ID_VALUE, authorizationResult.getDebugId());
        assertEquals(Date.from(Instant.parse("2021-10-08T23:37:39Z")), authorizationResult.getExpirationTime());
    }

}
