package com.paypal.hybris.core.commands.impl;

import com.paypal.hybris.core.commands.PayPalAbstractCommandTest;
import com.paypal.hybris.core.exception.PayPalProcessPaymentException;
import com.paypal.hybris.core.results.PayPalVaultedPaymentResult;
import com.paypal.hybris.core.service.PayPalConfigurationService;
import com.paypal.hybris.data.AmountWithBreakdownData;
import com.paypal.hybris.data.OrderItemData;
import com.paypal.hybris.data.PayPalHostedFieldsOrderRequestData;
import com.paypal.hybris.data.PayPalOrderResponseData;
import com.paypal.hybris.data.PayPalPaymentCollectionData;
import com.paypal.hybris.data.PayPalPaymentData;
import com.paypal.hybris.data.PayPalVaultedPaymentProcessResultData;
import com.paypal.hybris.data.PurchaseUnitData;
import com.paypal.hybris.data.StoredCredentialData;
import com.paypal.hybris.data.SupplementaryData;
import de.hybris.bootstrap.annotations.UnitTest;
import de.hybris.platform.commerceservices.util.GuidKeyGenerator;
import de.hybris.platform.core.model.order.CartModel;
import de.hybris.platform.order.CartService;
import de.hybris.platform.payment.dto.TransactionStatus;
import de.hybris.platform.payment.dto.TransactionStatusDetails;
import org.junit.Before;
import org.junit.Test;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;
import org.mockito.Spy;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.util.MultiValueMap;
import org.springframework.web.client.RestTemplate;

import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import static com.paypal.hybris.core.constants.PaypalcoreConstants.COMPLETED_STATUS_RESULT;
import static com.paypal.hybris.core.constants.PaypalcoreConstants.CREATED_STATUS_RESULT;
import static com.paypal.hybris.core.constants.PaypalcoreConstants.PAYPAL_INTENT_AUTHORIZE;
import static com.paypal.hybris.core.constants.PaypalcoreConstants.PAYPAL_INTENT_CAPTURE;
import static org.junit.Assert.assertEquals;
import static org.mockito.Matchers.any;
import static org.mockito.Matchers.eq;
import static org.mockito.Mockito.when;

@UnitTest
public class DefaultPayPalProcessVaultedPaymentCommandTest {

    private static final String ID = "id";
    private static final String ORDER_ID = "orderId";
    private static final String USD = "USD";
    private static final String SANDBOX_ORDERS_URL = "https://api.sandbox.paypal.com/v2/checkout/orders/";
    private static final String USAGE = "usage";
    private static final String PAYMENT_INITIATOR = "paymentInitiator";
    private static final String PAYMENT_TYPE = "paymentType";
    private static final String PAYMENT_TOKEN_TEST = "paymentTokenTest";
    private static final String COMPLETED = "COMPLETED";
    private static final String SAVED = "SAVED";
    private static final String PENDING = "PENDING";
    private static final String DECLINED = "DECLINED";
    private static final String UNIQUE_UID = "uniqueUid";
    private static final String PAY_PAL_CLIENT_META_DATA_ID = "payPalClientMetaDataId";
    private static final String DEBUG_ID = "debug_id";
    protected static final String PAYPAL_DEBUG_ID = "Paypal-Debug-Id";

    @InjectMocks
    @Spy
    private DefaultPayPalProcessVaultedPaymentCommand unit;
    @Mock
    private RestTemplate restTemplate;
    @Mock
    private PayPalConfigurationService defaultPayPalConfigurationService;
    @Mock
    private GuidKeyGenerator guidKeyGenerator;
    @Mock
    private CartService cartService;
    @Mock
    private CartModel cartModel;
    private PayPalHostedFieldsOrderRequestData requestData;
    private Date expirationDate;
    private SupplementaryData supplementaryData;
    private AmountWithBreakdownData amountWithBreakdownData;
    private List<OrderItemData> orderItemDataList;

    @Before
    public void setUp() throws Exception {
        MockitoAnnotations.initMocks(this);

        Map<String, TransactionStatusDetails> transactionStatusDetailsMap = new HashMap<>();
        transactionStatusDetailsMap.put(COMPLETED, TransactionStatusDetails.SUCCESFULL);
        transactionStatusDetailsMap.put(SAVED, TransactionStatusDetails.SUCCESFULL);
        transactionStatusDetailsMap.put(PENDING, TransactionStatusDetails.REVIEW_NEEDED);
        transactionStatusDetailsMap.put(DECLINED, TransactionStatusDetails.PROCESSOR_DECLINE);

        Map<String, TransactionStatus> transactionStatusMap = new HashMap<>();
        transactionStatusMap.put(COMPLETED, TransactionStatus.ACCEPTED);
        transactionStatusMap.put(SAVED, TransactionStatus.ACCEPTED);
        transactionStatusMap.put(PENDING, TransactionStatus.REVIEW);
        transactionStatusMap.put(DECLINED, TransactionStatus.REJECTED);

        unit.setTransactionStatusDetailsMap(transactionStatusDetailsMap);
        unit.setTransactionStatusMap(transactionStatusMap);

        StoredCredentialData storedCredential = new StoredCredentialData();
        storedCredential.setPaymentInitiator(PAYMENT_INITIATOR);
        storedCredential.setUsage(USAGE);
        storedCredential.setPaymentType(PAYMENT_TYPE);

        ArrayList<PurchaseUnitData> purchaseUnits = new ArrayList<>();

        PurchaseUnitData purchaseUnit = new PurchaseUnitData();
        purchaseUnit.setId(ORDER_ID);
        purchaseUnits.add(purchaseUnit);
        PayPalPaymentCollectionData paymentCollection = new PayPalPaymentCollectionData();

        expirationDate = new Date();
        ArrayList<PayPalPaymentData> authorizations = new ArrayList<>();
        PayPalPaymentData palPaymentDataAuth = new PayPalPaymentData();
        palPaymentDataAuth.setStatus(CREATED_STATUS_RESULT);
        palPaymentDataAuth.setId(ORDER_ID);
        palPaymentDataAuth.setExpirationTime(expirationDate);
        authorizations.add(palPaymentDataAuth);

        ArrayList<PayPalPaymentData> captures = new ArrayList<>();
        PayPalPaymentData palPaymentDataCapture = new PayPalPaymentData();
        palPaymentDataCapture.setStatus(COMPLETED_STATUS_RESULT);
        palPaymentDataCapture.setId(ORDER_ID);
        captures.add(palPaymentDataCapture);

        paymentCollection.setAuthorizations(authorizations);
        paymentCollection.setCaptures(captures);
        purchaseUnit.setPayments(paymentCollection);

        amountWithBreakdownData = new AmountWithBreakdownData();
        supplementaryData = new SupplementaryData();
        orderItemDataList = List.of(new OrderItemData());

        requestData = new PayPalHostedFieldsOrderRequestData();
        requestData.setCurrency(USD);
        requestData.setIntent(PAYPAL_INTENT_AUTHORIZE);
        requestData.setVaultId(PAYMENT_TOKEN_TEST);
        requestData.setStoredCredential(storedCredential);
        requestData.setSupplementaryData(supplementaryData);
        requestData.setBreakdownAmountData(amountWithBreakdownData);
        requestData.setOrderItems(orderItemDataList);

        PayPalOrderResponseData responseData = new PayPalOrderResponseData();
        responseData.setStatus(COMPLETED);
        responseData.setId(ID);
        responseData.setPurchaseUnits(purchaseUnits);

        MultiValueMap<String, String> headers = new HttpHeaders();
        headers.put(PAYPAL_DEBUG_ID, List.of(DEBUG_ID));
        ResponseEntity<PayPalOrderResponseData> responseEntity = new ResponseEntity<>(responseData, headers, HttpStatus.OK);

        when(restTemplate.postForEntity(eq(SANDBOX_ORDERS_URL), any(HttpEntity.class), eq(PayPalOrderResponseData.class)))
                .thenReturn(responseEntity);
        when(defaultPayPalConfigurationService.isFraudNetEnabled()).thenReturn(Boolean.FALSE);
        when(guidKeyGenerator.generate()).thenReturn(UNIQUE_UID);
        when(cartService.getSessionCart()).thenReturn(cartModel);
        when(cartModel.getPayPalClientMetadataId()).thenReturn(PAY_PAL_CLIENT_META_DATA_ID);
    }

    @Test
    public void shouldReturnSuccessfulTransactionStatusWhenFraudNetDisabledForAuthorizationIntent() {
        PayPalAbstractCommandTest.mockCreatePayPalEnvironmentMethodToReturnSandboxEnv(unit);
        PayPalAbstractCommandTest.mockCreateApiContextMethodToReturnSandboxApiContext(unit);
        when(defaultPayPalConfigurationService.getPayPalIntent()).thenReturn(PAYPAL_INTENT_AUTHORIZE);

        PayPalVaultedPaymentProcessResultData resultData = unit.perform(requestData);

        assertEquals(TransactionStatus.ACCEPTED, resultData.getTransactionStatus());
        assertEquals(TransactionStatusDetails.SUCCESFULL, resultData.getTransactionStatusDetails());
        assertEquals(ID, resultData.getOrderId());
        assertEquals(ORDER_ID, resultData.getRequestId());
    }

    @Test
    public void shouldReturnSuccessfulTransactionStatusWhenFraudNetDisabledForCaptureIntent() {
        PayPalAbstractCommandTest.mockCreatePayPalEnvironmentMethodToReturnSandboxEnv(unit);
        PayPalAbstractCommandTest.mockCreateApiContextMethodToReturnSandboxApiContext(unit);
        when(defaultPayPalConfigurationService.getPayPalIntent()).thenReturn(PAYPAL_INTENT_CAPTURE);

        requestData.setIntent(PAYPAL_INTENT_CAPTURE);

        PayPalVaultedPaymentProcessResultData resultData = unit.perform(requestData);

        assertEquals(TransactionStatus.ACCEPTED, resultData.getTransactionStatus());
        assertEquals(TransactionStatusDetails.SUCCESFULL, resultData.getTransactionStatusDetails());
        assertEquals(ID, resultData.getOrderId());
        assertEquals(ORDER_ID, resultData.getRequestId());
    }

    @Test
    public void shouldReturnSuccessfulTransactionStatusWhenFraudNetEnabledForAuthorizationIntent() {
        PayPalAbstractCommandTest.mockCreatePayPalEnvironmentMethodToReturnSandboxEnv(unit);
        PayPalAbstractCommandTest.mockCreateApiContextMethodToReturnSandboxApiContext(unit);
        when(defaultPayPalConfigurationService.isFraudNetEnabled()).thenReturn(Boolean.TRUE);
        when(defaultPayPalConfigurationService.getPayPalIntent()).thenReturn(PAYPAL_INTENT_AUTHORIZE);

        PayPalVaultedPaymentProcessResultData resultData = unit.perform(requestData);

        assertEquals(TransactionStatus.ACCEPTED, resultData.getTransactionStatus());
        assertEquals(TransactionStatusDetails.SUCCESFULL, resultData.getTransactionStatusDetails());
        assertEquals(ID, resultData.getOrderId());
        assertEquals(ORDER_ID, resultData.getRequestId());
    }

    @Test
    public void shouldReturnSuccessfulTransactionStatusWhenFraudNetEnabledForCaptureIntent() {
        PayPalAbstractCommandTest.mockCreatePayPalEnvironmentMethodToReturnSandboxEnv(unit);
        PayPalAbstractCommandTest.mockCreateApiContextMethodToReturnSandboxApiContext(unit);
        when(defaultPayPalConfigurationService.isFraudNetEnabled()).thenReturn(Boolean.TRUE);
        when(defaultPayPalConfigurationService.getPayPalIntent()).thenReturn(PAYPAL_INTENT_CAPTURE);

        requestData.setIntent(PAYPAL_INTENT_CAPTURE);

        PayPalVaultedPaymentProcessResultData resultData = unit.perform(requestData);

        assertEquals(TransactionStatus.ACCEPTED, resultData.getTransactionStatus());
        assertEquals(TransactionStatusDetails.SUCCESFULL, resultData.getTransactionStatusDetails());
        assertEquals(ID, resultData.getOrderId());
        assertEquals(ORDER_ID, resultData.getRequestId());
        assertEquals(expirationDate, ((PayPalVaultedPaymentResult)resultData).getExpirationTime());
    }

    @Test(expected = PayPalProcessPaymentException.class)
    public void shouldThrowPayPalProcessPaymentExceptionWhenRestTemplateThrowsAnyException() {
        PayPalAbstractCommandTest.mockCreatePayPalEnvironmentMethodToReturnSandboxEnv(unit);
        PayPalAbstractCommandTest.mockCreateApiContextMethodToReturnSandboxApiContext(unit);
        when(restTemplate.postForEntity(eq(SANDBOX_ORDERS_URL), any(HttpEntity.class), eq(PayPalOrderResponseData.class)))
                .thenThrow(RuntimeException.class);

        unit.perform(requestData);
    }

    @Test
    public void shouldPopulatePurchaseUnitData() {
        PurchaseUnitData purchaseUnitData = unit.getPurchaseUnitData(requestData).get(0);

        assertEquals(supplementaryData, purchaseUnitData.getSupplementaryData());
        assertEquals(amountWithBreakdownData, purchaseUnitData.getAmount());
        assertEquals(orderItemDataList, purchaseUnitData.getItems());
    }
}
