package com.paypal.hybris.core.commands.impl;

import com.paypal.hybris.core.commands.PayPalAbstractCommandTest;
import com.paypal.hybris.data.PayPalOrderDetailsData;
import com.paypal.hybris.data.PayPalOrderResponseData;
import de.hybris.bootstrap.annotations.UnitTest;
import de.hybris.platform.payment.AdapterException;
import de.hybris.platform.servicelayer.dto.converter.Converter;
import org.junit.Before;
import org.junit.Test;
import org.junit.runner.RunWith;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.Spy;
import org.mockito.junit.MockitoJUnitRunner;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpMethod;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.web.client.RestTemplate;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNull;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.ArgumentMatchers.anyString;
import static org.mockito.ArgumentMatchers.eq;
import static org.mockito.Mockito.doReturn;
import static org.mockito.Mockito.when;

@UnitTest
@RunWith(MockitoJUnitRunner.StrictStubs.class)
public class DefaultPayPalGetOrderDetailsCommandTest {
	private static final String ORDER_ID = "orderId";
	private static final String ORDER_DETAILS_URL = "https://api.sandbox.paypal.com/v2/checkout/orders/" + ORDER_ID;

	@InjectMocks
	@Spy
	private DefaultPayPalGetOrderDetailsCommand unit;
	@Mock
	private RestTemplate restTemplate;
	@Mock
	private Converter<PayPalOrderResponseData, PayPalOrderDetailsData> orderResponseToDetailsConverter;
	private PayPalOrderResponseData orderResponseData;

	@Before
	public void setUp() throws Exception {
		PayPalAbstractCommandTest.mockCreatePayPalEnvironmentMethodToReturnSandboxEnv(unit);
		PayPalAbstractCommandTest.mockCreateApiContextMethodToReturnSandboxApiContext(unit);

		orderResponseData = new PayPalOrderResponseData();

		doReturn(new ResponseEntity<>(orderResponseData, HttpStatus.OK))
				.when(restTemplate).exchange(anyString(), eq(HttpMethod.GET), any(HttpEntity.class), eq(PayPalOrderResponseData.class));
	}

	@Test(expected = AdapterException.class)
	public void shouldThrowAdapterExceptionWhenRestTemplateThrowsAnyException() {
		when(restTemplate.exchange(eq(ORDER_DETAILS_URL), eq(HttpMethod.GET), any(HttpEntity.class), eq(PayPalOrderResponseData.class)))
				.thenThrow(RuntimeException.class);

		unit.perform(ORDER_ID);
	}

	@Test
	public void shouldReturnNullWhenResponseEntityIsNull() {
		when(restTemplate.exchange(eq(ORDER_DETAILS_URL), eq(HttpMethod.GET), any(HttpEntity.class), eq(PayPalOrderResponseData.class)))
				.thenReturn(null);

		assertNull(unit.perform(ORDER_ID));
	}

	@Test
	public void shouldReturnNullWhenPayPalOrderResponseDataIsNull() {
		when(restTemplate.exchange(eq(ORDER_DETAILS_URL), eq(HttpMethod.GET), any(HttpEntity.class), eq(PayPalOrderResponseData.class)))
				.thenReturn(new ResponseEntity<>(null, HttpStatus.OK));

		assertNull(unit.perform(ORDER_ID));
	}

	@Test
	public void shouldReturnPayPalOrderDetailsData() {
		PayPalOrderDetailsData data = new PayPalOrderDetailsData();
		when(orderResponseToDetailsConverter.convert(orderResponseData)).thenReturn(data);

		PayPalOrderDetailsData orderDetailsData = unit.perform(ORDER_ID);

		assertEquals(data, orderDetailsData);
	}

}
