package com.paypal.hybris.core.commands.impl;

import com.paypal.hybris.core.commands.PayPalAbstractCommandTest;
import com.paypal.hybris.data.CardData;
import com.paypal.hybris.data.PayPalGetCardDetailsResponseData;
import com.paypal.hybris.data.PaymentSourceData;
import de.hybris.bootstrap.annotations.UnitTest;
import de.hybris.platform.payment.AdapterException;
import org.junit.Before;
import org.junit.Test;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;
import org.mockito.Spy;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpMethod;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.web.client.RestTemplate;

import java.util.Optional;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertFalse;
import static org.junit.Assert.assertTrue;
import static org.mockito.Matchers.any;
import static org.mockito.Matchers.eq;
import static org.mockito.Mockito.when;

@UnitTest
public class DefaultPayPalGetCardDetailsCommandTest {
    private static final String TOKEN = "token";
    private static final String API_URL = "https://api.sandbox.paypal.com/v3/vault/payment-tokens/" + TOKEN;
    private static final String RESPONSE_DATA_ID = "responseDataId";
    private static final String TEST_VAULT_ID = "testVaultId";
    private static final String API_ERROR_MESSAGE_JSON = "{details:[{description:testErrorMessage}]}";


    @InjectMocks
    @Spy
    private DefaultPayPalGetCardDetailsCommand unit;
    @Mock
    private RestTemplate restTemplate;
    private PayPalGetCardDetailsResponseData responseData;

    @Before
    public void setUp() throws Exception {
        MockitoAnnotations.initMocks(this);
        PayPalAbstractCommandTest.mockCreatePayPalEnvironmentMethodToReturnSandboxEnv(unit);
        PayPalAbstractCommandTest.mockCreateApiContextMethodToReturnSandboxApiContext(unit);

        responseData = new PayPalGetCardDetailsResponseData();
        responseData.setId(RESPONSE_DATA_ID);
        PaymentSourceData paymentSource = new PaymentSourceData();
        responseData.setPaymentSource(paymentSource);
        CardData card = new CardData();
        paymentSource.setCard(card);
        card.setVaultId(TEST_VAULT_ID);
    }

    @Test
    public void shouldReturnCardDataWhenPresent() {
        ResponseEntity<PayPalGetCardDetailsResponseData> response = new ResponseEntity<>(responseData, HttpStatus.OK);
        when(restTemplate.exchange(eq(API_URL), eq(HttpMethod.GET), any(HttpEntity.class), eq(PayPalGetCardDetailsResponseData.class)))
                .thenReturn(response);

        Optional<CardData> result = unit.perform(TOKEN);
        assertTrue(result.isPresent());
        assertEquals(TEST_VAULT_ID, result.get().getVaultId());
    }

    @Test
    public void shouldReturnEmptyOptionalWhenCardDataIsEmpty() {
        responseData.getPaymentSource().setCard(null);

        ResponseEntity<PayPalGetCardDetailsResponseData> response = new ResponseEntity<>(responseData, HttpStatus.OK);
        when(restTemplate.exchange(eq(API_URL), eq(HttpMethod.GET), any(HttpEntity.class), eq(PayPalGetCardDetailsResponseData.class)))
                .thenReturn(response);

        assertFalse(unit.perform(TOKEN).isPresent());
    }

    @Test(expected = AdapterException.class)
    public void shouldThrowAdapterExceptionWhenRestTemplateThrowsAnyException() {
        responseData.getPaymentSource().setCard(null);

        when(restTemplate.exchange(eq(API_URL), eq(HttpMethod.GET), any(HttpEntity.class), eq(PayPalGetCardDetailsResponseData.class)))
                .thenThrow(new RuntimeException(API_ERROR_MESSAGE_JSON));

        unit.perform(TOKEN);
    }

}
