package com.paypal.hybris.core.commands.impl;

import com.paypal.http.Headers;
import com.paypal.http.HttpResponse;
import com.paypal.hybris.core.commands.PayPalAbstractCommandTest;
import com.paypal.hybris.core.exception.PayPalRefundAdapterException;
import com.paypal.hybris.core.results.PayPalRefundResult;
import com.paypal.hybris.core.service.PayPalConfigurationService;
import com.paypal.payments.Money;
import com.paypal.payments.Refund;
import de.hybris.bootstrap.annotations.UnitTest;
import de.hybris.platform.payment.AdapterException;
import de.hybris.platform.payment.commands.request.FollowOnRefundRequest;
import de.hybris.platform.payment.dto.TransactionStatus;
import de.hybris.platform.payment.dto.TransactionStatusDetails;
import org.apache.commons.io.IOUtils;
import org.junit.Before;
import org.junit.Test;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;
import org.mockito.Spy;

import java.io.IOException;
import java.math.BigDecimal;
import java.nio.charset.StandardCharsets;
import java.util.Currency;
import java.util.HashMap;
import java.util.Map;

import static org.junit.Assert.assertEquals;
import static org.mockito.Mockito.when;


@UnitTest
public class DefaultPayPalFollowOnRefundCommandTest {

    private static final String MERCHANT_TRANSACTION_CODE = "merchantTransactionCode";
    private static final String REQUEST_ID = "requestId";
    private static final String REQUEST_TOKEN = "requestToken";
    private static final String PAYMENT_PROVIDER = "paymentProvider";
    private static final String USD = "USD";
    private static final String AMOUNT = "123.45";
    private static final String COMPLETED = "COMPLETED";
    private static final String SAVED = "SAVED";
    private static final String PENDING = "PENDING";
    private static final String DECLINED = "DECLINED";
    private static final String REFUND_ID = "refundId";
    private static final String PAYPAL_DEBUG_ID = "Paypal-Debug-Id";
    private static final String PAYPAL_DEBUG_ID_VALUE = "PAYPAL_DEBUG_ID_VALUE";

    @InjectMocks
    @Spy
    private DefaultPayPalFollowOnRefundCommand unit;
    @Mock
    private PayPalConfigurationService defaultPayPalConfigurationService;
    @Mock
    private HttpResponse<Refund> httpResponse;
    private FollowOnRefundRequest request;
    private Refund response;
    private String expectedRefundResponse;
    private String expectedRefundRequest;

    @Before
    public void setUp() throws Exception {
        MockitoAnnotations.initMocks(this);

        Map<String, TransactionStatusDetails> transactionStatusDetailsMap = new HashMap<>();
        transactionStatusDetailsMap.put(COMPLETED, TransactionStatusDetails.SUCCESFULL);
        transactionStatusDetailsMap.put(SAVED, TransactionStatusDetails.SUCCESFULL);
        transactionStatusDetailsMap.put(PENDING, TransactionStatusDetails.REVIEW_NEEDED);
        transactionStatusDetailsMap.put(DECLINED, TransactionStatusDetails.PROCESSOR_DECLINE);

        Map<String, TransactionStatus> transactionStatusMap = new HashMap<>();
        transactionStatusMap.put(COMPLETED, TransactionStatus.ACCEPTED);
        transactionStatusMap.put(SAVED, TransactionStatus.ACCEPTED);
        transactionStatusMap.put(PENDING, TransactionStatus.REVIEW);
        transactionStatusMap.put(DECLINED, TransactionStatus.REJECTED);

        Headers headers = new Headers();
        headers.header(PAYPAL_DEBUG_ID, PAYPAL_DEBUG_ID_VALUE);

        when(httpResponse.headers()).thenReturn(headers);

        unit.setTransactionStatusDetailsMap(transactionStatusDetailsMap);
        unit.setTransactionStatusMap(transactionStatusMap);

        request = new FollowOnRefundRequest(MERCHANT_TRANSACTION_CODE,
                REQUEST_ID,
                REQUEST_TOKEN,
                Currency.getInstance(USD),
                new BigDecimal(AMOUNT),
                PAYMENT_PROVIDER);

        response = new Refund();
        response.status(COMPLETED);
        response.id(REFUND_ID);
        Money amount = new Money();
        amount.value(AMOUNT);
        response.amount(amount);

        expectedRefundResponse = IOUtils.resourceToString("/test/FollowOnRefundResponse.json", StandardCharsets.UTF_8);
        expectedRefundRequest = IOUtils.resourceToString("/test/FollowOnRefundRequest.json", StandardCharsets.UTF_8);
    }

    @Test(expected = IllegalArgumentException.class)
    public void shouldThrowIllegalArgumentExceptionWhenRequestIsNull() {
        unit.perform(null);
    }

    @Test(expected = PayPalRefundAdapterException.class)
    public void shouldThrowPayPalRefundAdapterExceptionWhenClientThrowsIOException() throws IOException {
        PayPalAbstractCommandTest.mockHttpClientToThrowIOException(unit);

        unit.perform(request);
    }

    @Test
    public void shouldReturnTransactionStatusAccepted() throws IOException {
        PayPalAbstractCommandTest.mockHttpClientToReturnResponseWith204StatusCode(unit, httpResponse);
        when(httpResponse.result()).thenReturn(response);

        PayPalRefundResult refundResult = (PayPalRefundResult) unit.perform(request);

        assertEquals(TransactionStatus.ACCEPTED, refundResult.getTransactionStatus());
        assertEquals(TransactionStatusDetails.SUCCESFULL, refundResult.getTransactionStatusDetails());
        assertEquals(new BigDecimal(AMOUNT), refundResult.getTotalAmount());
        assertEquals(REFUND_ID, refundResult.getRequestId());
        assertEquals(MERCHANT_TRANSACTION_CODE, refundResult.getMerchantTransactionCode());
        assertEquals(Currency.getInstance(USD), refundResult.getCurrency());
        assertEquals(PAYPAL_DEBUG_ID_VALUE, refundResult.getDebugId());
        assertEquals(expectedRefundResponse, refundResult.getResponseField());
        assertEquals(expectedRefundRequest, refundResult.getRequestField());
    }

}
