package com.paypal.hybris.core.commands.impl;

import com.paypal.hybris.core.commands.PayPalAbstractCommandTest;
import com.paypal.hybris.core.service.PayPalConfigurationService;
import com.paypal.hybris.data.ApplePayOrderRequestData;
import com.paypal.hybris.data.Level2Level3CardData;
import com.paypal.hybris.data.OrderItemData;
import com.paypal.hybris.data.PayPalAddressDetailsData;
import com.paypal.hybris.data.PayPalHostedFieldsCreateOrderData;
import com.paypal.hybris.data.PayPalHostedFieldsOrderRequestData;
import com.paypal.hybris.data.PayPalOrderRequestData;
import com.paypal.hybris.data.PayPalOrderResponseData;
import com.paypal.hybris.data.PayPalVaultOrderRequestData;
import com.paypal.hybris.data.StoredCredentialData;
import de.hybris.bootstrap.annotations.UnitTest;
import de.hybris.platform.acceleratorservices.urlresolver.SiteBaseUrlResolutionService;
import de.hybris.platform.basecommerce.model.site.BaseSiteModel;
import de.hybris.platform.commercefacades.order.data.OrderEntryData;
import de.hybris.platform.commercefacades.product.data.PriceData;
import de.hybris.platform.commercefacades.product.data.ProductData;
import de.hybris.platform.commerceservices.url.impl.AbstractUrlResolver;
import de.hybris.platform.core.model.c2l.CurrencyModel;
import de.hybris.platform.core.model.media.MediaModel;
import de.hybris.platform.core.model.order.AbstractOrderEntryModel;
import de.hybris.platform.core.model.order.CartEntryModel;
import de.hybris.platform.core.model.order.CartModel;
import de.hybris.platform.core.model.product.ProductModel;
import de.hybris.platform.order.CartService;
import de.hybris.platform.payment.AdapterException;
import de.hybris.platform.servicelayer.dto.converter.Converter;
import de.hybris.platform.site.BaseSiteService;
import org.junit.Before;
import org.junit.Test;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;
import org.mockito.Spy;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.web.client.RestTemplate;

import java.math.BigDecimal;

import static com.paypal.hybris.core.constants.PaypalcoreConstants.NO_SHIPPING;
import static com.paypal.hybris.core.constants.PaypalcoreConstants.PAYPAL_INTENT_AUTHORIZE;
import static org.junit.Assert.assertEquals;
import static org.mockito.Matchers.any;
import static org.mockito.Matchers.eq;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

@UnitTest
public class DefaultPayPalCreateOrderCommandTest {

    private static final String ORDER_ID = "orderId";
    private static final String USD = "USD";
    private static final String AMOUNT = "123.45";
    private static final String SANDBOX_ORDERS_URL = "https://api.sandbox.paypal.com/v2/checkout/orders/";
    private static final String USAGE = "usage";
    private static final String PAYMENT_INITIATOR = "paymentInitiator";
    private static final String PAYMENT_TYPE = "paymentType";
    private static final String CUSTOMER_ID = "customerId";
    private static final String FIRST_NAME = "firstName";
    private static final String LAST_NAME = "lastName";
    private static final String POSTAL_CODE = "postalCode";
    private static final String VAULT_ID = "vaultId";
    private static final String NAME_ON_CARD = "nameOnCard";
    private static final String USAGE_TYPE = "usageType";
    private static final String CUSTOMER_TYPE = "customerType";
    private static final String PAY_PAL_PARTNER_ATTRIBUTION_ID = "payPalPartnerAttributionId";
    private static final String EMAIL_ADDRESS = "emailAddress";
    private static final String PHONE_NUMBER = "phoneNumber";
    private static final String PAYPAL_PRODUCT_ITEM_TYPE = "PHYSICAL_GOODS";
    private static final int MAX_DESCRIPTION_LENGTH = 100;
    private static final String PRODUCT_DESCRIPTION = "This is a mock product description";
    private static final String SITE_URL = "https://example.com";
    private static final String PRODUCT_NAME = "Test Product";
    private static final String CODE = "TEST_PRODUCT_CODE";
    private static final String IMAGE = "test/thumbnail.jpg";
    private static final String PRODUCT_URL = "https://example.com/product/test-product";

    @InjectMocks
    @Spy
    private DefaultPayPalCreateOrderCommand unit;
    @Mock
    private RestTemplate restTemplate;
    @Mock
    private PayPalConfigurationService defaultPayPalConfigurationService;
    @Mock
    private CartEntryModel cartEntry1;
    @Mock
    private CartEntryModel cartEntry2;
    @Mock
    private CartModel currentCart;
    @Mock
    private AbstractUrlResolver<ProductModel> urlResolver;
    @Mock
    private CurrencyModel currencyModel;
    @Mock
    private ProductModel productModel;
    @Mock
    private MediaModel mediaModel;
    @Mock
    private BaseSiteModel baseSiteModel;
    @Mock
    private BaseSiteService baseSiteService;
    @Mock
    private SiteBaseUrlResolutionService siteBaseUrlResolutionService;
    @Mock
    private CartService cartService;
    @Mock
    private Converter<AbstractOrderEntryModel, OrderItemData> orderItemDataConverter;
    @Mock
    private Converter<CartModel, Level2Level3CardData> payPalL2L3DataConverter;

    private PayPalHostedFieldsOrderRequestData hostedFieldsOrderRequestData;
    private ApplePayOrderRequestData applePayOrderRequestData;
    private PayPalVaultOrderRequestData vaultOrderRequestData;
    private PayPalOrderResponseData responseData;
    private PayPalOrderRequestData orderRequestData;
    private OrderEntryData orderEntryData;
    private Level2Level3CardData level2Level3CardData;

    @Before
    public void setUp() throws Exception {
        MockitoAnnotations.initMocks(this);
        PayPalAbstractCommandTest.mockCreatePayPalEnvironmentMethodToReturnSandboxEnv(unit);
        PayPalAbstractCommandTest.mockCreateApiContextMethodToReturnSandboxApiContext(unit);

        when(defaultPayPalConfigurationService.getPayPalPartnerAttributionId()).thenReturn(PAY_PAL_PARTNER_ATTRIBUTION_ID);

        level2Level3CardData = new Level2Level3CardData();
        orderEntryData = new OrderEntryData();
        PriceData price = new PriceData();
        price.setCurrencyIso(USD);
        price.setValue(BigDecimal.TEN);
        orderEntryData.setBasePrice(price);
        orderEntryData.setQuantity(1L);
        ProductData productData = new ProductData();
        productData.setImageUrl(IMAGE);
        productData.setUrl(PRODUCT_URL);
        productData.setName(PRODUCT_NAME);
        productData.setDescription(PRODUCT_DESCRIPTION);
        productData.setCode(CODE);
        orderEntryData.setProduct(productData);

        hostedFieldsOrderRequestData = getHostedFieldsOrderRequestData();
        vaultOrderRequestData = getVaultOrderRequestData();
        applePayOrderRequestData = getApplePayOrderRequestData();

        responseData = new PayPalOrderResponseData();
        responseData.setId(ORDER_ID);

        when(currentCart.getSubtotal()).thenReturn(10d);
        when(currentCart.getCurrency()).thenReturn(currencyModel);
        when(currencyModel.getIsocode()).thenReturn(USD);
        when(cartService.getSessionCart()).thenReturn(currentCart);
        orderRequestData = new PayPalOrderRequestData();
        when(baseSiteService.getCurrentBaseSite()).thenReturn(baseSiteModel);
        when(siteBaseUrlResolutionService.getWebsiteUrlForSite(baseSiteModel, true, IMAGE))
                .thenReturn(SITE_URL + IMAGE);
    }

    @Test
    public void shouldReturnOrderIdWhenHostedFieldsDataPresent() {
        when(restTemplate.postForEntity(eq(SANDBOX_ORDERS_URL), any(HttpEntity.class), eq(PayPalOrderResponseData.class)))
                .thenReturn(new ResponseEntity<>(responseData, HttpStatus.OK));

        String id = unit.perform(hostedFieldsOrderRequestData);

        assertEquals(ORDER_ID, id);
        verify(restTemplate).postForEntity(eq(SANDBOX_ORDERS_URL), any(HttpEntity.class), eq(PayPalOrderResponseData.class));
    }

    @Test
    public void shouldReturnOrderIdWithL2L3Data() {
        when(defaultPayPalConfigurationService.isL2L3DataEnabled()).thenReturn(Boolean.TRUE);
        when(cartService.getSessionCart()).thenReturn(currentCart);
        when(payPalL2L3DataConverter.convert(currentCart)).thenReturn(level2Level3CardData);
        when(restTemplate.postForEntity(eq(SANDBOX_ORDERS_URL), any(HttpEntity.class), eq(PayPalOrderResponseData.class)))
                .thenReturn(new ResponseEntity<>(responseData, HttpStatus.OK));

        String id = unit.perform(hostedFieldsOrderRequestData);

        assertEquals(ORDER_ID, id);
        verify(restTemplate).postForEntity(eq(SANDBOX_ORDERS_URL), any(HttpEntity.class), eq(PayPalOrderResponseData.class));
    }

    @Test
    public void shouldReturnOrderIdWhenApplePayDataPresent() {
        when(restTemplate.postForEntity(eq(SANDBOX_ORDERS_URL), any(HttpEntity.class), eq(PayPalOrderResponseData.class)))
                .thenReturn(new ResponseEntity<>(responseData, HttpStatus.OK));

        String id = unit.perform(applePayOrderRequestData);

        assertEquals(ORDER_ID, id);
        verify(restTemplate).postForEntity(eq(SANDBOX_ORDERS_URL), any(HttpEntity.class), eq(PayPalOrderResponseData.class));
    }

    @Test
    public void shouldReturnOrderIdWhenHostedFieldsDataIsNull() {
        hostedFieldsOrderRequestData.setHostedFieldsData(null);

        when(restTemplate.postForEntity(eq(SANDBOX_ORDERS_URL), any(HttpEntity.class), eq(PayPalOrderResponseData.class)))
                .thenReturn(new ResponseEntity<>(responseData, HttpStatus.OK));

        String id = unit.perform(hostedFieldsOrderRequestData);

        assertEquals(ORDER_ID, id);
        verify(restTemplate).postForEntity(eq(SANDBOX_ORDERS_URL), any(HttpEntity.class), eq(PayPalOrderResponseData.class));
    }

    @Test
    public void shouldReturnOrderIdWhenShippingAddressIsNullAndShippingPreferencePresent() {
        hostedFieldsOrderRequestData.setShippingAddress(null);

        when(restTemplate.postForEntity(eq(SANDBOX_ORDERS_URL), any(HttpEntity.class), eq(PayPalOrderResponseData.class)))
                .thenReturn(new ResponseEntity<>(responseData, HttpStatus.OK));

        String id = unit.perform(hostedFieldsOrderRequestData);

        assertEquals(ORDER_ID, id);
        verify(restTemplate).postForEntity(eq(SANDBOX_ORDERS_URL), any(HttpEntity.class), eq(PayPalOrderResponseData.class));
    }

    @Test
    public void shouldReturnOrderIdWhenShippingAddressIsNullAndShippingPreferenceIsNull() {
        hostedFieldsOrderRequestData.setShippingAddress(null);
        hostedFieldsOrderRequestData.setShippingPreference(null);

        when(restTemplate.postForEntity(eq(SANDBOX_ORDERS_URL), any(HttpEntity.class), eq(PayPalOrderResponseData.class)))
                .thenReturn(new ResponseEntity<>(responseData, HttpStatus.OK));

        String id = unit.perform(hostedFieldsOrderRequestData);

        assertEquals(ORDER_ID, id);
        verify(restTemplate).postForEntity(eq(SANDBOX_ORDERS_URL), any(HttpEntity.class), eq(PayPalOrderResponseData.class));
    }

    @Test
    public void shouldReturnOrderIdWhenPickUpInStoreIsTrue() {
        hostedFieldsOrderRequestData.setPickUpInStore(true);

        when(restTemplate.postForEntity(eq(SANDBOX_ORDERS_URL), any(HttpEntity.class), eq(PayPalOrderResponseData.class)))
                .thenReturn(new ResponseEntity<>(responseData, HttpStatus.OK));

        String id = unit.perform(hostedFieldsOrderRequestData);

        assertEquals(ORDER_ID, id);
        verify(restTemplate).postForEntity(eq(SANDBOX_ORDERS_URL), any(HttpEntity.class), eq(PayPalOrderResponseData.class));
    }

    @Test
    public void shouldReturnOrderIdWhenPassingPayPalVaultOrderRequestData() {
        when(restTemplate.postForEntity(eq(SANDBOX_ORDERS_URL), any(HttpEntity.class), eq(PayPalOrderResponseData.class)))
                .thenReturn(new ResponseEntity<>(responseData, HttpStatus.OK));

        String id = unit.perform(vaultOrderRequestData);

        assertEquals(ORDER_ID, id);
        verify(restTemplate).postForEntity(eq(SANDBOX_ORDERS_URL), any(HttpEntity.class), eq(PayPalOrderResponseData.class));
    }

    @Test
    public void shouldReturnOrderIdWhenSaveOrderFlowIsActive() {
        vaultOrderRequestData.setSaveOrderFlowActive(true);

        when(restTemplate.postForEntity(eq(SANDBOX_ORDERS_URL), any(HttpEntity.class), eq(PayPalOrderResponseData.class)))
                .thenReturn(new ResponseEntity<>(responseData, HttpStatus.OK));

        String id = unit.perform(vaultOrderRequestData);

        assertEquals(ORDER_ID, id);
        verify(restTemplate).postForEntity(eq(SANDBOX_ORDERS_URL), any(HttpEntity.class), eq(PayPalOrderResponseData.class));
    }

    @Test(expected = AdapterException.class)
    public void shouldThrowAdapterExceptionWhenRestTemplateThrowsAnyException() {
        PayPalAbstractCommandTest.mockCreatePayPalEnvironmentMethodToReturnSandboxEnv(unit);
        PayPalAbstractCommandTest.mockCreateApiContextMethodToReturnSandboxApiContext(unit);

        when(restTemplate.postForEntity(eq(SANDBOX_ORDERS_URL), any(HttpEntity.class), eq(PayPalOrderResponseData.class)))
                .thenThrow(RuntimeException.class);

        unit.perform(hostedFieldsOrderRequestData);
    }

    private PayPalHostedFieldsOrderRequestData getHostedFieldsOrderRequestData() {
        PayPalHostedFieldsOrderRequestData payPalHostedFieldsOrderRequestData = new PayPalHostedFieldsOrderRequestData();
        PayPalAddressDetailsData shippingAddress = new PayPalAddressDetailsData();
        payPalHostedFieldsOrderRequestData.setOrderId(ORDER_ID);
        payPalHostedFieldsOrderRequestData.setAmount(AMOUNT);
        payPalHostedFieldsOrderRequestData.setIntent(PAYPAL_INTENT_AUTHORIZE);
        payPalHostedFieldsOrderRequestData.setCurrency(USD);
        payPalHostedFieldsOrderRequestData.setShippingPreference(NO_SHIPPING);
        payPalHostedFieldsOrderRequestData.setCustomerId(CUSTOMER_ID);
        payPalHostedFieldsOrderRequestData.setPickUpInStore(false);
        payPalHostedFieldsOrderRequestData.setSaveOrderFlowActive(false);
        payPalHostedFieldsOrderRequestData.setShippingAddress(shippingAddress);
        shippingAddress.setFirstName(FIRST_NAME);
        shippingAddress.setLastName(LAST_NAME);
        shippingAddress.setPostalCode(POSTAL_CODE);
        StoredCredentialData storedCredential = new StoredCredentialData();
        storedCredential.setPaymentInitiator(PAYMENT_INITIATOR);
        storedCredential.setUsage(USAGE);
        storedCredential.setPaymentType(PAYMENT_TYPE);
        payPalHostedFieldsOrderRequestData.setStoredCredential(storedCredential);
        payPalHostedFieldsOrderRequestData.setVaultId(VAULT_ID);
        PayPalHostedFieldsCreateOrderData hostedFieldsData = new PayPalHostedFieldsCreateOrderData();
        payPalHostedFieldsOrderRequestData.setHostedFieldsData(hostedFieldsData);
        hostedFieldsData.setIsShouldBeSaved(Boolean.FALSE);
        hostedFieldsData.setNameOnCard(NAME_ON_CARD);
        hostedFieldsData.setBillingAddress(shippingAddress);

        return payPalHostedFieldsOrderRequestData;
    }

    private ApplePayOrderRequestData getApplePayOrderRequestData() {
        ApplePayOrderRequestData applePayOrderRequestData = new ApplePayOrderRequestData();
        applePayOrderRequestData.setOrderId(ORDER_ID);
        applePayOrderRequestData.setAmount(AMOUNT);
        applePayOrderRequestData.setIntent(PAYPAL_INTENT_AUTHORIZE);
        applePayOrderRequestData.setCurrency(USD);
        applePayOrderRequestData.setShippingPreference(NO_SHIPPING);
        applePayOrderRequestData.setCustomerId(CUSTOMER_ID);
        applePayOrderRequestData.setPickUpInStore(false);
        applePayOrderRequestData.setSaveOrderFlowActive(false);

        StoredCredentialData storedCredential = new StoredCredentialData();
        storedCredential.setPaymentInitiator(PAYMENT_INITIATOR);
        storedCredential.setUsage(USAGE);
        storedCredential.setPaymentType(PAYMENT_TYPE);
        applePayOrderRequestData.setStoredCredential(storedCredential);

        applePayOrderRequestData.setName(NAME_ON_CARD);
        applePayOrderRequestData.setEmailAddress(EMAIL_ADDRESS);
        applePayOrderRequestData.setPhoneNumber(PHONE_NUMBER);

        return applePayOrderRequestData;
    }

    private PayPalVaultOrderRequestData getVaultOrderRequestData() {
        PayPalVaultOrderRequestData payPalVaultOrderRequestData = new PayPalVaultOrderRequestData();
        PayPalAddressDetailsData shippingAddress = new PayPalAddressDetailsData();
        payPalVaultOrderRequestData.setOrderId(ORDER_ID);
        payPalVaultOrderRequestData.setAmount(AMOUNT);
        payPalVaultOrderRequestData.setIntent(PAYPAL_INTENT_AUTHORIZE);
        payPalVaultOrderRequestData.setCurrency(USD);
        payPalVaultOrderRequestData.setShippingPreference(NO_SHIPPING);
        payPalVaultOrderRequestData.setCustomerId(CUSTOMER_ID);
        payPalVaultOrderRequestData.setPickUpInStore(false);
        payPalVaultOrderRequestData.setSaveOrderFlowActive(false);
        payPalVaultOrderRequestData.setShippingAddress(shippingAddress);
        shippingAddress.setFirstName(FIRST_NAME);
        shippingAddress.setLastName(LAST_NAME);
        shippingAddress.setPostalCode(POSTAL_CODE);
        StoredCredentialData storedCredential = new StoredCredentialData();
        storedCredential.setPaymentInitiator(PAYMENT_INITIATOR);
        storedCredential.setUsage(USAGE);
        storedCredential.setPaymentType(PAYMENT_TYPE);
        payPalVaultOrderRequestData.setVaultId(VAULT_ID);
        payPalVaultOrderRequestData.setUsageType(USAGE_TYPE);
        payPalVaultOrderRequestData.setCustomerType(CUSTOMER_TYPE);

        return payPalVaultOrderRequestData;
    }
}
