package com.paypal.hybris.core.commands.impl;

import com.paypal.hybris.core.commands.PayPalAbstractCommandTest;
import com.paypal.hybris.core.constants.PaypalcoreConstants;
import com.paypal.hybris.core.service.PayPalConfigurationService;
import com.paypal.hybris.data.PayPalConnectAccessTokenData;
import com.paypal.hybris.data.PayPalConnectAccessTokenRequest;
import de.hybris.bootstrap.annotations.UnitTest;
import de.hybris.platform.core.model.user.CustomerModel;
import de.hybris.platform.payment.AdapterException;
import de.hybris.platform.servicelayer.session.SessionService;
import org.junit.Before;
import org.junit.Test;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;
import org.mockito.Spy;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.util.LinkedMultiValueMap;
import org.springframework.util.MultiValueMap;
import org.springframework.web.client.RestTemplate;

import static org.junit.Assert.assertEquals;
import static org.mockito.Matchers.any;
import static org.mockito.Matchers.eq;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

@UnitTest
public class DefaultPayPalConnectExchangeCodeToAccessTokenCommandTest {
    private static final String API_URL = "https://api-m.sandbox.paypal.com/";
    private static final String API_ENDPOINT = "https://api-m.sandbox.paypal.com/oauth2/token";
    private static final String AUTHORIZATION_CODE = "authorizationCode";
    private static final String USER_SESSION_ATTRIBUTE = "user";
    private static final String VAULT_CUSTOMER_ID = "vaultCustomerId";
    private static final String TOKEN_ID = "tokenId";
    private static final String GRANT_TYPE = "grant_type";
    private static final String GRANT_TYPE_VALUE = "authorization_code";
    private static final String CLIENT_CREDENTIALS = "client_credentials";
    private static final String CODE = "code";
    private static final String RESPONSE_TYPE = "response_type";
    private static final String ID_TOKEN = "id_token";
    private static final String TARGET_CUSTOMER_ID = "target_customer_id";
    private static final String ENCODED_AUTH_HEADER = "Basic bnVsbDpudWxs";


    @InjectMocks
    @Spy
    private DefaultPayPalConnectExchangeCodeToAccessTokenCommand unit;
    @Mock
    private RestTemplate restTemplate;
    @Mock
    private PayPalConfigurationService defaultPayPalConfigurationService;
    @Mock
    private CustomerModel customerModel;
    @Mock
    private SessionService sessionService;

    private PayPalConnectAccessTokenData response;

    private PayPalConnectAccessTokenRequest request;

    @Before
    public void setUp() throws Exception {
        MockitoAnnotations.initMocks(this);
        PayPalAbstractCommandTest.mockCreatePayPalEnvironmentMethodToReturnSandboxEnv(unit);
        PayPalAbstractCommandTest.mockCreateApiContextMethodToReturnSandboxApiContext(unit);

        response = new PayPalConnectAccessTokenData();
        request = new PayPalConnectAccessTokenRequest();

        response.setId_token(TOKEN_ID);
        request.setAuthorizationCode(AUTHORIZATION_CODE);
        request.setChangePaymentButtonAvailable(Boolean.TRUE);

        when(defaultPayPalConfigurationService.getPayPalApiUrl()).thenReturn(API_URL);
        when(sessionService.getAttribute(USER_SESSION_ATTRIBUTE)).thenReturn(customerModel);
        when(customerModel.getVaultCustomerId()).thenReturn(VAULT_CUSTOMER_ID);
    }

    @Test(expected = AdapterException.class)
    public void shouldThrowAdapterExceptionWhenRestTemplateThrowsAnyException() {
        request.setChangePaymentButtonAvailable(Boolean.FALSE);
        when(restTemplate.postForEntity(eq(API_ENDPOINT), any(HttpEntity.class), eq(PayPalConnectAccessTokenData.class)))
                .thenThrow(RuntimeException.class);

        unit.perform(request);
    }

    @Test
    public void verifyHttpEntityWhenAuthCodePresent() {
        when(restTemplate.postForEntity(eq(API_ENDPOINT), any(HttpEntity.class), eq(PayPalConnectAccessTokenData.class)))
                .thenReturn(new ResponseEntity<>(response, HttpStatus.OK));

        MultiValueMap<String, String> map = new LinkedMultiValueMap<>();
        HttpHeaders headers = new HttpHeaders();

        map.add(CODE, AUTHORIZATION_CODE);
        map.add(GRANT_TYPE, GRANT_TYPE_VALUE);
        map.add(TARGET_CUSTOMER_ID, VAULT_CUSTOMER_ID);
        headers.setContentType(MediaType.APPLICATION_FORM_URLENCODED);
        headers.add(PaypalcoreConstants.AUTHORIZATION_HEADER, ENCODED_AUTH_HEADER);

        PayPalConnectAccessTokenData result = unit.perform(request);
        assertEquals(TOKEN_ID, result.getId_token());
        verify(restTemplate).postForEntity(
                eq(API_ENDPOINT),
                eq(new HttpEntity<>(map, headers)),
                eq(PayPalConnectAccessTokenData.class));
    }

    @Test
    public void verifyHttpEntityWhenAuthCodeNotPresent() {
        request.setAuthorizationCode(null);
        request.setChangePaymentButtonAvailable(Boolean.TRUE);
        when(restTemplate.postForEntity(eq(API_ENDPOINT), any(HttpEntity.class), eq(PayPalConnectAccessTokenData.class)))
                .thenReturn(new ResponseEntity<>(response, HttpStatus.OK));

        MultiValueMap<String, String> map = new LinkedMultiValueMap<>();
        HttpHeaders headers = new HttpHeaders();

        map.add(RESPONSE_TYPE, ID_TOKEN);
        map.add(GRANT_TYPE, CLIENT_CREDENTIALS);
        map.add(TARGET_CUSTOMER_ID, VAULT_CUSTOMER_ID);
        headers.setContentType(MediaType.APPLICATION_FORM_URLENCODED);
        headers.add(PaypalcoreConstants.AUTHORIZATION_HEADER, ENCODED_AUTH_HEADER);

        PayPalConnectAccessTokenData result = unit.perform(request);
        assertEquals(TOKEN_ID, result.getId_token());
        verify(restTemplate).postForEntity(
                eq(API_ENDPOINT),
                eq(new HttpEntity<>(map, headers)),
                eq(PayPalConnectAccessTokenData.class));
    }

}
