package com.paypal.hybris.core.commands.impl;

import com.paypal.hybris.core.commands.PayPalAbstractCommandTest;
import com.paypal.hybris.core.exception.PayPalHttpClientErrorException;
import com.paypal.hybris.core.exception.PayPalProcessPaymentException;
import com.paypal.hybris.core.results.PayPalCaptureResult;
import com.paypal.hybris.core.service.PayPalConfigurationService;
import com.paypal.hybris.core.util.PayPalCommandsUtil;
import com.paypal.hybris.data.AmountWithBreakdownData;
import com.paypal.hybris.data.PayPalOrderResponseData;
import com.paypal.hybris.data.PayPalPaymentCollectionData;
import com.paypal.hybris.data.PayPalPaymentData;
import com.paypal.hybris.data.PurchaseUnitData;
import de.hybris.bootstrap.annotations.UnitTest;
import de.hybris.platform.core.model.order.CartModel;
import de.hybris.platform.order.CartService;
import de.hybris.platform.payment.commands.request.CaptureRequest;
import de.hybris.platform.payment.dto.TransactionStatus;
import de.hybris.platform.payment.dto.TransactionStatusDetails;
import de.hybris.platform.payment.model.PaymentTransactionEntryModel;
import de.hybris.platform.payment.model.PaymentTransactionModel;
import de.hybris.platform.servicelayer.model.ModelService;
import org.apache.commons.io.IOUtils;
import org.junit.Before;
import org.junit.Test;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;
import org.mockito.Spy;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.HttpStatus;
import org.springframework.http.ResponseEntity;
import org.springframework.util.MultiValueMap;
import org.springframework.web.client.HttpClientErrorException;
import org.springframework.web.client.RestTemplate;

import java.math.BigDecimal;
import java.nio.charset.StandardCharsets;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Currency;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import static com.paypal.hybris.core.constants.PaypalcoreConstants.PAYPAL_INTENT_CAPTURE;
import static org.junit.Assert.assertEquals;
import static org.mockito.Matchers.any;
import static org.mockito.Matchers.eq;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

@UnitTest
public class DefaultPayPalCaptureIntentCommandTest {
    private static final String TEST_RESPONSE_ID = "testResponseId";
    private static final String MERCHANT_TRANSACTION_CODE = "merchantTransactionCode";
    private static final String REQUEST_ID = "requestId";
    private static final String REQUEST_TOKEN = "requestToken";
    private static final String USD = "USD";
    private static final String AMOUNT = "123.45";
    private static final String PAYMENT_PROVIDER = "paymentProvider";
    private static final String SUBSCRIPTION_ID = "subscriptionId";
    private static final String API_URL = "https://api.sandbox.paypal.com/v2/checkout/orders/" + REQUEST_ID + "/capture";
    private static final String COMPLETED = "COMPLETED";
    private static final String SAVED = "SAVED";
    private static final String PENDING = "PENDING";
    private static final String DECLINED = "DECLINED";
    private static final String PAYMENT_ID = "paymentId";
    private static final String CLIENT_META_DATA_ID = "clientMetaDataId";
    private static final String PAYPAL_DEBUG_ID = "Paypal-Debug-Id";
    private static final String PAYPAL_DEBUG_ID_VALUE = "PAYPAL_DEBUG_ID_VALUE";
    private String expectedCaptureResponse;
    private String expectedCaptureRequest;

    @InjectMocks
    @Spy
    private DefaultPayPalCaptureIntentCommand unit;
    @Mock
    private RestTemplate restTemplate;
    @Mock
    private PayPalConfigurationService defaultPayPalConfigurationService;
    @Mock
    private CartService cartService;
    @Mock
    private CartModel cartModel;
    @Mock
    private ModelService modelService;

    private CaptureRequest request;
    private PayPalOrderResponseData response;

    @Before
    public void setUp() throws Exception {
        MockitoAnnotations.initMocks(this);
        PayPalAbstractCommandTest.mockCreatePayPalEnvironmentMethodToReturnSandboxEnv(unit);
        PayPalAbstractCommandTest.mockCreateApiContextMethodToReturnSandboxApiContext(unit);

        Map<String, TransactionStatusDetails> transactionStatusDetailsMap = new HashMap<>();
        transactionStatusDetailsMap.put(COMPLETED, TransactionStatusDetails.SUCCESFULL);
        transactionStatusDetailsMap.put(SAVED, TransactionStatusDetails.SUCCESFULL);
        transactionStatusDetailsMap.put(PENDING, TransactionStatusDetails.REVIEW_NEEDED);
        transactionStatusDetailsMap.put(DECLINED, TransactionStatusDetails.PROCESSOR_DECLINE);

        Map<String, TransactionStatus> transactionStatusMap = new HashMap<>();
        transactionStatusMap.put(COMPLETED, TransactionStatus.ACCEPTED);
        transactionStatusMap.put(SAVED, TransactionStatus.ACCEPTED);
        transactionStatusMap.put(PENDING, TransactionStatus.REVIEW);
        transactionStatusMap.put(DECLINED, TransactionStatus.REJECTED);

        unit.setTransactionStatusDetailsMap(transactionStatusDetailsMap);
        unit.setTransactionStatusMap(transactionStatusMap);

        request = new CaptureRequest(MERCHANT_TRANSACTION_CODE,
                REQUEST_ID,
                REQUEST_TOKEN,
                Currency.getInstance(USD),
                new BigDecimal(AMOUNT),
                PAYMENT_PROVIDER,
                SUBSCRIPTION_ID);

        response = new PayPalOrderResponseData();
        response.setId(TEST_RESPONSE_ID);
        response.setStatus(COMPLETED);
        ArrayList<PurchaseUnitData> purchaseUnits = new ArrayList<>();
        PurchaseUnitData purchaseUnitData = new PurchaseUnitData();
        purchaseUnits.add(purchaseUnitData);
        PayPalPaymentCollectionData payments = populateCaptures();
        purchaseUnitData.setPayments(payments);
        response.setPurchaseUnits(purchaseUnits);

        when(defaultPayPalConfigurationService.isFraudNetEnabled()).thenReturn(Boolean.FALSE);
        expectedCaptureResponse = IOUtils.resourceToString("/test/CaptureResponse.json", StandardCharsets.UTF_8);
        expectedCaptureRequest = IOUtils.resourceToString("/test/CaptureRequest.json", StandardCharsets.UTF_8);
        when(defaultPayPalConfigurationService.getPayPalIntent()).thenReturn(PAYPAL_INTENT_CAPTURE);
    }

    @Test(expected = PayPalProcessPaymentException.class)
    public void shouldThrowPayPalProcessPaymentExceptionWhenRestTemplateThrowsHttpClientErrorException() {
        when(restTemplate.postForEntity(eq(API_URL), any(HttpEntity.class), eq(PayPalOrderResponseData.class)))
                .thenThrow(HttpClientErrorException.class);

        unit.perform(request);
    }

    @Test(expected = PayPalProcessPaymentException.class)
    public void shouldThrowPayPalProcessPaymentExceptionWhenCompletedCaptureIsNotPresent() {
        response.getPurchaseUnits().stream()
                .findFirst()
                .map((PurchaseUnitData::getPayments))
                .map(PayPalPaymentCollectionData::getCaptures)
                .ifPresent(ArrayList::clear);

        when(restTemplate.postForEntity(eq(API_URL), any(HttpEntity.class), eq(PayPalOrderResponseData.class)))
                .thenReturn(new ResponseEntity<>(response, HttpStatus.OK));

        unit.perform(request);
    }

    @Test
    public void shouldReturnTransactionStatusAcceptedWhenFraudNetIsDisabled() {
        MultiValueMap<String, String> headers = new HttpHeaders();
        headers.put(PAYPAL_DEBUG_ID, List.of(PAYPAL_DEBUG_ID_VALUE));
        when(restTemplate.postForEntity(eq(API_URL), any(HttpEntity.class), eq(PayPalOrderResponseData.class)))
                .thenReturn(new ResponseEntity<>(response, headers, HttpStatus.OK));

        PayPalCaptureResult result = (PayPalCaptureResult) unit.perform(request);

        assertEquals(request.getTotalAmount(), result.getTotalAmount());
        assertCaptureResultHasCorrectData(result);
    }

    @Test
    public void shouldReturnTransactionStatusAcceptedWhenFraudNetIsEnabled() {
        when(defaultPayPalConfigurationService.isFraudNetEnabled()).thenReturn(Boolean.TRUE);
        when(cartService.getSessionCart()).thenReturn(cartModel);
        when(cartModel.getPayPalClientMetadataId()).thenReturn(CLIENT_META_DATA_ID);

        MultiValueMap<String, String> headers = new HttpHeaders();
        headers.put(PAYPAL_DEBUG_ID, List.of(PAYPAL_DEBUG_ID_VALUE));
        when(restTemplate.postForEntity(eq(API_URL), any(HttpEntity.class), eq(PayPalOrderResponseData.class)))
                .thenReturn(new ResponseEntity<>(response, headers, HttpStatus.OK));

        PayPalCaptureResult result = (PayPalCaptureResult) unit.perform(request);

        verify(cartModel).getPayPalClientMetadataId();
        assertCaptureResultHasCorrectData(result);
    }

    @Test(expected = PayPalHttpClientErrorException.class)
    public void shouldThrowPayPalHttpClientErrorExceptionWhenHttpClientErrorExceptionOccurs() {
        HttpHeaders headers = new HttpHeaders();
        headers.put(PAYPAL_DEBUG_ID, List.of(PAYPAL_DEBUG_ID_VALUE));
        HttpClientErrorException parentException = new HttpClientErrorException(HttpStatus.BAD_REQUEST,
                HttpStatus.BAD_REQUEST.toString(), headers, new byte[]{}, StandardCharsets.UTF_8);

        when(restTemplate.postForEntity(eq(API_URL), any(HttpEntity.class), eq(PayPalOrderResponseData.class)))
                .thenThrow(parentException);

        unit.performCaptureRequest(headers, request);
    }

    @Test
    public void shouldSaveEntryToTransactionWhenTransactionHasNoEntries() {
        PaymentTransactionModel transaction = new PaymentTransactionModel();
        PaymentTransactionEntryModel entry = new PaymentTransactionEntryModel();

        PayPalCommandsUtil.saveTransactionEntryToTransaction(modelService, transaction, entry);

        assertEquals(entry, transaction.getEntries().get(0));
        verify(modelService).save(entry);
        verify(modelService).refresh(transaction);
    }

    @Test
    public void shouldSaveEntryToTransactionWhenTransactionHasEntries() {
        PaymentTransactionModel transaction = new PaymentTransactionModel();
        PaymentTransactionEntryModel entry = new PaymentTransactionEntryModel();
        PaymentTransactionEntryModel entry1 = new PaymentTransactionEntryModel();
        transaction.setEntries(List.of(entry1));

        PayPalCommandsUtil.saveTransactionEntryToTransaction(modelService, transaction, entry);

        assertEquals(2, transaction.getEntries().size());
        verify(modelService).save(entry);
        verify(modelService).save(transaction);
        verify(modelService).refresh(transaction);
    }

    private void assertCaptureResultHasCorrectData(PayPalCaptureResult result) {
        assertEquals(request.getTotalAmount(), result.getTotalAmount());
        assertEquals(TransactionStatus.ACCEPTED, result.getTransactionStatus());
        assertEquals(TransactionStatusDetails.SUCCESFULL, result.getTransactionStatusDetails());
        assertEquals(MERCHANT_TRANSACTION_CODE, result.getMerchantTransactionCode());
        assertEquals(SUBSCRIPTION_ID, result.getReconciliationId());
        assertEquals(expectedCaptureResponse, result.getResponseField());
        assertEquals(expectedCaptureRequest, result.getRequestField());
        assertEquals(PAYPAL_DEBUG_ID_VALUE, result.getDebugId());
        assertEquals(Date.from(Instant.parse("2021-10-08T23:37:39Z")), result.getCreateTime());
        assertEquals(Date.from(Instant.parse("2021-10-08T23:37:39Z")), result.getUpdateTime());
    }
    private static PayPalPaymentCollectionData populateCaptures() {
        PayPalPaymentCollectionData payments = new PayPalPaymentCollectionData();
        ArrayList<PayPalPaymentData> captures = new ArrayList<>();
        PayPalPaymentData palPaymentData = new PayPalPaymentData();
        palPaymentData.setId(PAYMENT_ID);
        palPaymentData.setStatus(COMPLETED);
        palPaymentData.setCreateTime(Date.from(Instant.parse("2021-10-08T23:37:39Z")));
        palPaymentData.setUpdateTime(Date.from(Instant.parse("2021-10-08T23:37:39Z")));
        AmountWithBreakdownData amountWithBreakdownData = new AmountWithBreakdownData();
        amountWithBreakdownData.setCurrencyCode(USD);
        amountWithBreakdownData.setValue(AMOUNT);
        palPaymentData.setAmount(amountWithBreakdownData);
        captures.add(palPaymentData);
        payments.setCaptures(captures);
        return payments;
    }

}
