package com.paypal.hybris.core.commands.impl;

import com.paypal.http.Headers;
import com.paypal.http.HttpResponse;
import com.paypal.hybris.core.commands.PayPalAbstractCommandTest;
import com.paypal.hybris.core.exception.PayPalAuthorizeAdapterException;
import com.paypal.hybris.core.request.PayPalSavedOrderSubscriptionAuthorizationRequest;
import com.paypal.hybris.core.results.PayPalAuthorizationResult;
import com.paypal.orders.Authorization;
import com.paypal.orders.Money;
import com.paypal.orders.Order;
import com.paypal.orders.PaymentCollection;
import com.paypal.orders.PurchaseUnit;
import de.hybris.bootstrap.annotations.UnitTest;
import de.hybris.platform.payment.commands.request.SubscriptionAuthorizationRequest;
import de.hybris.platform.payment.dto.BillingInfo;
import de.hybris.platform.payment.dto.TransactionStatus;
import de.hybris.platform.payment.dto.TransactionStatusDetails;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.StringUtils;
import org.junit.Before;
import org.junit.Test;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;
import org.mockito.Spy;

import java.io.IOException;
import java.math.BigDecimal;
import java.nio.charset.StandardCharsets;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Currency;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertThrows;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

@UnitTest
public class DefaultPayPalAuthorizeSavedOrderCommandTest {

    private static final String MERCHANT_TRANSACTION_CODE = "merchantTransactionCode";
    private static final String USD = "USD";
    private static final String AMOUNT = "123.45";
    private static final String ERROR_MESSAGE_FROM_IO_EXCEPTION = "testErrorMessageFromIOException";
    private static final String ORDER_NUMBER = "orderNumber";
    private static final String CREATED = "CREATED";
    private static final String NOT_CREATED = "NOT_CREATED";
    private static final String PAYMENT_PROVIDER = "paymentProvider";
    private static final BigDecimal TOTAL_AMOUNT_BIG_DECIMAL = new BigDecimal(AMOUNT);
    private static final String SUBSCRIPTION_ID = "subscriptionId";
    private static final String CV_2 = "cv2";
    private static final String PURCHASE_UNIT_ID = "purchaseUnitId";
    private static final String COMPLETED = "COMPLETED";
    private static final String SAVED = "SAVED";
    private static final String PENDING = "PENDING";
    private static final String DECLINED = "DECLINED";

    private static final String PAYPAL_DEBUG_ID = "Paypal-Debug-Id";
    private static final String PAYPAL_DEBUG_ID_VALUE = "PAYPAL_DEBUG_ID_VALUE";
    private String expectedAuthorizationResponseWithAmount;
    private String expectedAuthorizationRequest;

    @InjectMocks
    @Spy
    private DefaultPayPalAuthorizeSavedOrderCommand unit;
    @Mock
    private HttpResponse<Order> httpResponse;
    private Order order;
    private PayPalSavedOrderSubscriptionAuthorizationRequest request;
    private SubscriptionAuthorizationRequest subscriptionAuthorizationRequest;

    @Before
    public void setUp() throws Exception {
        MockitoAnnotations.initMocks(this);
        Map<String, TransactionStatusDetails> transactionStatusDetailsMap = new HashMap<>();
        transactionStatusDetailsMap.put(COMPLETED, TransactionStatusDetails.SUCCESFULL);
        transactionStatusDetailsMap.put(SAVED, TransactionStatusDetails.SUCCESFULL);
        transactionStatusDetailsMap.put(PENDING, TransactionStatusDetails.REVIEW_NEEDED);
        transactionStatusDetailsMap.put(DECLINED, TransactionStatusDetails.PROCESSOR_DECLINE);

        Map<String, TransactionStatus> transactionStatusMap = new HashMap<>();
        transactionStatusMap.put(COMPLETED, TransactionStatus.ACCEPTED);
        transactionStatusMap.put(SAVED, TransactionStatus.ACCEPTED);
        transactionStatusMap.put(PENDING, TransactionStatus.REVIEW);
        transactionStatusMap.put(DECLINED, TransactionStatus.REJECTED);

        unit.setTransactionStatusDetailsMap(transactionStatusDetailsMap);
        unit.setTransactionStatusMap(transactionStatusMap);

        subscriptionAuthorizationRequest = new SubscriptionAuthorizationRequest(
                MERCHANT_TRANSACTION_CODE, SUBSCRIPTION_ID,
                Currency.getInstance(USD), TOTAL_AMOUNT_BIG_DECIMAL,
                new BillingInfo(), CV_2, PAYMENT_PROVIDER);
        request = new PayPalSavedOrderSubscriptionAuthorizationRequest(subscriptionAuthorizationRequest, ORDER_NUMBER);

        order = new Order();
        ArrayList<PurchaseUnit> purchaseUnits = new ArrayList<>();
        PurchaseUnit purchaseUnit = new PurchaseUnit();
        purchaseUnit.id(StringUtils.EMPTY);
        purchaseUnits.add(purchaseUnit);

        purchaseUnit = new PurchaseUnit();
        purchaseUnit.id(ORDER_NUMBER);

        ArrayList<Authorization> authorizations = new ArrayList<>();
        Authorization authorization = new Authorization();
        authorization.status(CREATED);
        authorizations.add(authorization);

        PaymentCollection paymentCollection = new PaymentCollection();
        paymentCollection.authorizations(authorizations);
        purchaseUnit.payments(paymentCollection);
        purchaseUnits.add(purchaseUnit);

        order.purchaseUnits(purchaseUnits);

        Headers headers = new Headers();
        headers.header(PAYPAL_DEBUG_ID, PAYPAL_DEBUG_ID_VALUE);

        when(httpResponse.result()).thenReturn(order);
        when(httpResponse.headers()).thenReturn(headers);

        expectedAuthorizationResponseWithAmount = IOUtils.resourceToString("/test/AuthorizationResponseWithAmount.json", StandardCharsets.UTF_8);
        expectedAuthorizationRequest = IOUtils.resourceToString("/test/SubscriptionAuthorizationRequest.json", StandardCharsets.UTF_8);
    }

    @Test
    public void shouldThrowAdapterExceptionWhenHttpClientThrowsIOException() throws IOException {
        PayPalAbstractCommandTest.mockHttpClientToThrowIOException(unit);

        assertThrows(ERROR_MESSAGE_FROM_IO_EXCEPTION,
                PayPalAuthorizeAdapterException.class, () -> unit.perform(request));
    }

    @Test
    public void shouldReturnSameAmountWhenAuthorizationHasBeenCreated() throws IOException {
        PayPalAbstractCommandTest.mockHttpClientToReturnResponseWith204StatusCode(unit, httpResponse);

        List<Authorization> authorizations = new ArrayList<>();
        authorizations.add(getAuthorization(NOT_CREATED));
        authorizations.add(getAuthorization(CREATED));
        order.expirationTime("2021-10-08T23:37:39Z");

        order.purchaseUnits().stream().filter(purchaseUnit -> ORDER_NUMBER.equals(purchaseUnit.id()))
                .findAny().ifPresent(purchaseUnit -> purchaseUnit.payments().authorizations(authorizations));

        PayPalAuthorizationResult result = (PayPalAuthorizationResult) unit.perform(request);

        assertEquals(new BigDecimal(AMOUNT), result.getTotalAmount());
        assertEquals(PAYPAL_DEBUG_ID_VALUE, result.getDebugId());
        assertEquals(Date.from(Instant.parse("2021-10-08T23:37:39Z")), result.getExpirationTime());
        assertEquals(expectedAuthorizationResponseWithAmount, result.getResponseField());
        assertEquals(expectedAuthorizationRequest, result.getRequestField());
        verify(httpResponse, times(4)).result();
    }

    @Test(expected = IllegalArgumentException.class)
    public void shouldThrowIllegalArgumentExceptionWhenAuthorizationHasNotBeenCreated() throws IOException {
        PayPalAbstractCommandTest.mockHttpClientToReturnResponseWith204StatusCode(unit, httpResponse);

        List<Authorization> authorizations = new ArrayList<>();
        authorizations.add(getAuthorization(NOT_CREATED));

        order.purchaseUnits().stream().filter(purchaseUnit -> ORDER_NUMBER.equals(purchaseUnit.id()))
                .findFirst().ifPresent(purchaseUnit -> purchaseUnit.payments().authorizations(authorizations));

        unit.perform(request);
    }

    @Test(expected = IllegalArgumentException.class)
    public void shouldThrowIllegalArgumentExceptionWhenSubscriptionIdIsNull() throws IOException {
        PayPalAbstractCommandTest.mockHttpClientToReturnResponseWith204StatusCode(unit, httpResponse);

        subscriptionAuthorizationRequest = new SubscriptionAuthorizationRequest(
                MERCHANT_TRANSACTION_CODE, null,
                Currency.getInstance(USD), TOTAL_AMOUNT_BIG_DECIMAL,
                new BillingInfo(), CV_2, PAYMENT_PROVIDER);
        request = new PayPalSavedOrderSubscriptionAuthorizationRequest(subscriptionAuthorizationRequest, PURCHASE_UNIT_ID);

        unit.perform(request);
    }

    private Authorization getAuthorization(String status) {
        Authorization authorization = new Authorization();
        authorization.status(status);
        Money amount = new Money();
        amount.currencyCode(USD);
        amount.value(AMOUNT);
        authorization.amount(amount);
        authorization.expirationTime("2021-10-08T23:37:39Z");
        return authorization;
    }
}
