package com.paypal.hybris.core.commands.impl;

import com.paypal.http.Headers;
import com.paypal.http.HttpResponse;
import com.paypal.hybris.core.commands.PayPalAbstractCommandTest;
import com.paypal.hybris.core.request.PayPalAuthorizationRequest;
import com.paypal.hybris.core.results.PayPalAuthorizationResult;
import com.paypal.orders.Authorization;
import com.paypal.orders.Money;
import com.paypal.orders.Order;
import com.paypal.orders.PaymentCollection;
import com.paypal.orders.PurchaseUnit;
import de.hybris.bootstrap.annotations.UnitTest;
import de.hybris.platform.payment.AdapterException;
import de.hybris.platform.payment.dto.BillingInfo;
import de.hybris.platform.payment.dto.CardInfo;
import org.apache.commons.io.IOUtils;
import org.apache.commons.lang.StringUtils;
import org.junit.Before;
import org.junit.Test;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;
import org.mockito.Spy;

import java.io.IOException;
import java.math.BigDecimal;
import java.nio.charset.StandardCharsets;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Currency;
import java.util.Date;
import java.util.List;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertThrows;
import static org.mockito.Mockito.atLeast;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

@UnitTest
public class DefaultPayPalAuthorizationCommandTest {

    private static final String MERCHANT_TRANSACTION_CODE = "merchantTransactionCode";
    private static final String USD = "USD";
    private static final String AMOUNT = "123.45";
    private static final String ERROR_MESSAGE_FROM_IO_EXCEPTION = "testErrorMessageFromIOException";
    private static final String ORDER_NUMBER = "orderNumber";
    private static final String CREATED = "CREATED";
    private static final String NOT_CREATED = "NOT_CREATED";
    private static final String PAYPAL_DEBUG_ID = "Paypal-Debug-Id";
    private static final String PAYPAL_DEBUG_ID_VALUE = "PAYPAL_DEBUG_ID_VALUE";

    private String expectedAuthorizationResponse;
    private String expectedAuthorizationResponseWithAmount;
    private String expectedAuthorizationRequest;

    @InjectMocks
    @Spy
    private DefaultPayPalAuthorizationCommand unit;
    @Mock
    private HttpResponse<Order> httpResponse;
    private Order order;
    private PayPalAuthorizationRequest authorizationRequest;

    @Before
    public void setUp() throws Exception {
        MockitoAnnotations.initMocks(this);

        authorizationRequest = new PayPalAuthorizationRequest(MERCHANT_TRANSACTION_CODE,
                new CardInfo(),
                Currency.getInstance(USD),
                new BigDecimal(AMOUNT),
                new BillingInfo());
        authorizationRequest.setOrderNumber(ORDER_NUMBER);

        order = new Order();
        ArrayList<PurchaseUnit> purchaseUnits = new ArrayList<>();

        PurchaseUnit purchaseUnit = new PurchaseUnit();
        purchaseUnit.id(StringUtils.EMPTY);
        purchaseUnits.add(purchaseUnit);

        purchaseUnit = new PurchaseUnit();
        PaymentCollection paymentCollection = new PaymentCollection();
        purchaseUnit.id(ORDER_NUMBER);
        purchaseUnits.add(purchaseUnit);

        ArrayList<Authorization> authorizations = new ArrayList<>();
        paymentCollection.authorizations(authorizations);
        purchaseUnit.payments(paymentCollection);
        order.purchaseUnits(purchaseUnits);

        Headers headers = new Headers();
        headers.header(PAYPAL_DEBUG_ID, PAYPAL_DEBUG_ID_VALUE);

        when(httpResponse.result()).thenReturn(order);
        when(httpResponse.headers()).thenReturn(headers);

        expectedAuthorizationResponseWithAmount = IOUtils.resourceToString("/test/AuthorizationResponseWithAmount.json", StandardCharsets.UTF_8);
        expectedAuthorizationResponse = IOUtils.resourceToString("/test/AuthorizationResponse.json", StandardCharsets.UTF_8);
        expectedAuthorizationRequest = IOUtils.resourceToString("/test/AuthorizationRequest.json", StandardCharsets.UTF_8);
    }

    @Test
    public void shouldThrowAdapterExceptionWhenHttpClientThrowsIOException() throws IOException {
        PayPalAbstractCommandTest.mockHttpClientToThrowIOException(unit);

        AdapterException adapterException = assertThrows(AdapterException.class, () -> unit.perform(authorizationRequest));
        assertEquals(ERROR_MESSAGE_FROM_IO_EXCEPTION, adapterException.getMessage());
    }

    @Test
    public void shouldReturnSameAmountWhenAuthorizationHasBeenCreated() throws IOException {
        PayPalAbstractCommandTest.mockHttpClientToReturnResponseWith204StatusCode(unit, httpResponse);

        List<Authorization> authorizations = new ArrayList<>();
        authorizations.add(getAuthorization(NOT_CREATED));
        authorizations.add(getAuthorization(CREATED));

        order.purchaseUnits().stream().filter(purchaseUnit -> ORDER_NUMBER.equals(purchaseUnit.id()))
                .findFirst().ifPresent(purchaseUnit -> purchaseUnit.payments().authorizations(authorizations));

        order.expirationTime("2021-10-08T23:37:39Z");
        PayPalAuthorizationResult result = (PayPalAuthorizationResult) unit.perform(authorizationRequest);

        assertEquals(new BigDecimal(AMOUNT), result.getTotalAmount());
        assertEquals(PAYPAL_DEBUG_ID_VALUE, result.getDebugId());
        assertEquals(Date.from(Instant.parse("2021-10-08T23:37:39Z")), result.getExpirationTime());
        assertEquals(expectedAuthorizationResponseWithAmount, result.getResponseField());
        assertEquals(expectedAuthorizationRequest, result.getRequestField());
        verify(httpResponse, atLeast(2)).result();
    }

    @Test
    public void shouldReturnZeroAmountWhenAuthorizationHasNotBeenCreated() throws IOException {
        PayPalAbstractCommandTest.mockHttpClientToReturnResponseWith204StatusCode(unit, httpResponse);
        when(httpResponse.result()).thenReturn(order);

        List<Authorization> authorizations = new ArrayList<>();
        authorizations.add(getAuthorization(NOT_CREATED));

        order.purchaseUnits().stream().filter(purchaseUnit -> ORDER_NUMBER.equals(purchaseUnit.id()))
                .findFirst().ifPresent(purchaseUnit -> purchaseUnit.payments().authorizations(authorizations));

        PayPalAuthorizationResult result = (PayPalAuthorizationResult) unit.perform(authorizationRequest);

        assertEquals(BigDecimal.ZERO, result.getTotalAmount());
        assertEquals(PAYPAL_DEBUG_ID_VALUE, result.getDebugId());
        assertEquals(Date.from(Instant.parse("2021-10-08T23:37:39Z")), result.getExpirationTime());
        assertEquals(expectedAuthorizationResponse, result.getResponseField());
        assertEquals(expectedAuthorizationRequest, result.getRequestField());
        verify(httpResponse, atLeast(2)).result();
    }

    private Authorization getAuthorization(String status) {
        Authorization authorization = new Authorization();
        authorization.status(status);
        Money amount = new Money();
        amount.currencyCode(USD);
        amount.value(AMOUNT);
        authorization.amount(amount);
        authorization.expirationTime("2021-10-08T23:37:39Z");
        return authorization;
    }
}
