package com.paypal.hybris.core.commands;

import com.paypal.base.rest.APIContext;
import com.paypal.base.rest.PayPalRESTException;
import com.paypal.core.PayPalEnvironment;
import com.paypal.core.PayPalHttpClient;
import com.paypal.http.Headers;
import com.paypal.http.HttpResponse;
import com.paypal.hybris.core.commands.impl.DefaultPayPalVoidCommand;
import com.paypal.hybris.core.service.PayPalConfigurationService;
import de.hybris.bootstrap.annotations.UnitTest;
import org.junit.Before;
import org.junit.Test;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;
import org.mockito.Spy;

import java.io.IOException;

import static org.junit.Assert.assertNotNull;
import static org.mockito.Matchers.any;
import static org.mockito.Mockito.doReturn;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.when;

@UnitTest
public class PayPalAbstractCommandTest {

    private static final String API_ERROR_MESSAGE_JSON = "{details:[{description:testErrorMessageFromIOException, issue:testIssue}]}";
    private static final String API_ERROR_MESSAGE_JSON_FORMAT = "{details:[{description:testErrorMessageFromIOException, issue:%s}]}";
    private static final String CLIENT_ID = "clientId";
    private static final String SECRET_KEY = "secretKey";
    private static final String SANDBOX = "sandbox";
    private static final String LIVE = "live";
    private static final String PAYPAL_DEBUG_ID = "Paypal-Debug-Id";
    private static final String PAYPAL_DEBUG_ID_VALUE = "PAYPAL_DEBUG_ID_VALUE";

    @InjectMocks
    @Spy
    private DefaultPayPalVoidCommand unit;
    @Mock
    private PayPalConfigurationService defaultPayPalConfigurationService;
    @Mock
    private HttpResponse httpResponse;
    @Mock
    private Headers headers;

    @Before
    public void setUp() throws Exception {
        MockitoAnnotations.initMocks(this);
        when(httpResponse.headers()).thenReturn(headers);
        when(headers.header(PAYPAL_DEBUG_ID)).thenReturn(PAYPAL_DEBUG_ID_VALUE);
        when(httpResponse.headers()).thenReturn(headers);
    }

    @Test
    public void shouldCreateSandBoxClient() {
        when(defaultPayPalConfigurationService.getClientID()).thenReturn(CLIENT_ID);
        when(defaultPayPalConfigurationService.getSecretKey()).thenReturn(SECRET_KEY);

        PayPalHttpClient client = unit.createClient();

        assertNotNull(client);
    }

    @Test
    public void shouldCreateLiveClient() {
        when(defaultPayPalConfigurationService.getClientID()).thenReturn(CLIENT_ID);
        when(defaultPayPalConfigurationService.getSecretKey()).thenReturn(SECRET_KEY);
        when(defaultPayPalConfigurationService.getEnvironmentType()).thenReturn(LIVE);

        PayPalHttpClient client = unit.createClient();

        assertNotNull(client);
    }

    @Test
    public void shouldCreateApiContext() {
        when(defaultPayPalConfigurationService.getClientID()).thenReturn(CLIENT_ID);
        when(defaultPayPalConfigurationService.getSecretKey()).thenReturn(SECRET_KEY);
        when(defaultPayPalConfigurationService.getEnvironmentType()).thenReturn(SANDBOX);

        APIContext apiContext = unit.createAPIContext();

        assertNotNull(apiContext);
    }

    /**
     * Static method for usage in non-abstract command's tests
     *
     * @param command Spy object which is being tested
     */
    public static void mockHttpClientToThrowIOException(PayPalAbstractCommand command) throws IOException {
        PayPalHttpClient httpClient = mock(PayPalHttpClient.class);
        IOException ioException = new IOException(API_ERROR_MESSAGE_JSON);
        when(httpClient.execute(any())).thenThrow(ioException);

        doReturn(httpClient).when(command).createClient();
    }

    /**
     * Static method for usage in non-abstract command's tests
     *
     * @param command Spy object which is being tested
     * @param issueMessage message that will be in the 'issue' property of exception's Json message
     */
    public static void mockHttpClientToThrowIOExceptionWithIssueMessage(PayPalAbstractCommand command, String issueMessage) throws IOException {
        PayPalHttpClient httpClient = mock(PayPalHttpClient.class);
        IOException ioException = new IOException(String.format(API_ERROR_MESSAGE_JSON_FORMAT, issueMessage));
        when(httpClient.execute(any())).thenThrow(ioException);

        doReturn(httpClient).when(command).createClient();
    }

    /**
     * Static method for usage in non-abstract command's tests
     *
     * @param command Spy object which is being tested
     */
    public static void mockHttpClientToReturnResponseWith200StatusCode(PayPalAbstractCommand command) throws IOException {
        PayPalHttpClient httpClient = mock(PayPalHttpClient.class);
        HttpResponse httpResponse = mock(HttpResponse.class);
        Headers headers = mock(Headers.class);
        when(httpResponse.headers()).thenReturn(headers);
        when(headers.header(PAYPAL_DEBUG_ID)).thenReturn(PAYPAL_DEBUG_ID_VALUE);
        when(httpResponse.headers()).thenReturn(headers);
        when(httpResponse.statusCode()).thenReturn(200);
        when(httpClient.execute(any())).thenReturn(httpResponse);

        doReturn(httpClient).when(command).createClient();
    }

    /**
     * Static method for usage in non-abstract command's tests
     *
     * @param command Spy object which is being tested
     */
    public static void mockHttpClientToReturnResponseWith204StatusCode(PayPalAbstractCommand command) throws IOException {
        PayPalHttpClient httpClient = mock(PayPalHttpClient.class);
        HttpResponse httpResponse = mock(HttpResponse.class);
        Headers headers = mock(Headers.class);
        when(httpResponse.headers()).thenReturn(headers);
        when(headers.header(PAYPAL_DEBUG_ID)).thenReturn(PAYPAL_DEBUG_ID_VALUE);
        when(httpResponse.headers()).thenReturn(headers);
        when(httpResponse.statusCode()).thenReturn(204);
        when(httpClient.execute(any())).thenReturn(httpResponse);

        doReturn(httpClient).when(command).createClient();
    }

    /**
     * Static method for usage in non-abstract command's tests
     *
     * @param command Spy object which is being tested
     * @param response Mock HttpResponse
     */
    public static void mockHttpClientToReturnResponseWith204StatusCode(PayPalAbstractCommand command, HttpResponse<? extends Object> response) throws IOException {
        PayPalHttpClient httpClient = mock(PayPalHttpClient.class);

        Headers headers = mock(Headers.class);
        when(response.headers()).thenReturn(headers);
        when(headers.header(PAYPAL_DEBUG_ID)).thenReturn(PAYPAL_DEBUG_ID_VALUE);
        when(response.headers()).thenReturn(headers);
        when(response.statusCode()).thenReturn(204);
        when(httpClient.execute(any())).thenReturn((HttpResponse<Object>) response);

        doReturn(httpClient).when(command).createClient();
    }

    /**
     * Static method for usage in non-abstract command's tests
     *
     * @param command Spy object which is being tested
     */
    public static void mockCreatePayPalEnvironmentMethodToReturnSandboxEnv(PayPalAbstractCommand command) {
        PayPalEnvironment payPalEnvironment = new PayPalEnvironment.Sandbox(CLIENT_ID, SECRET_KEY);
        doReturn(payPalEnvironment).when(command).createPayPalEnvironment();
    }

    /**
     * Static method for usage in non-abstract command's tests
     *
     * @param command Spy object which is being tested
     */
    public static void mockCreateApiContextMethodToReturnSandboxApiContext(PayPalAbstractCommand command) {
        APIContext apiContext = new APIContext(CLIENT_ID, SECRET_KEY, SANDBOX);
        doReturn(apiContext).when(command).createAPIContext();
    }

    /**
     * Static method for usage in non-abstract command's tests
     *
     * @param command Spy object which is being tested
     */
    public static void mockApiContextFetchAccessTokenMethodToThrowPayPalRESTException(PayPalAbstractCommand command) throws PayPalRESTException {
        APIContext apiContext = mock(APIContext.class);
        when(apiContext.fetchAccessToken()).thenThrow(PayPalRESTException.class);
        doReturn(apiContext).when(command).createAPIContext();
    }

}
