package com.paypal.hybris.core.auth;

import de.hybris.bootstrap.annotations.UnitTest;
import de.hybris.platform.core.model.user.CustomerModel;
import de.hybris.platform.jalo.user.User;
import de.hybris.platform.servicelayer.user.UserService;
import org.junit.Before;
import org.junit.Test;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;
import org.mockito.Spy;
import org.springframework.security.authentication.AbstractAuthenticationToken;
import org.springframework.security.authentication.BadCredentialsException;
import org.springframework.security.authentication.CredentialsExpiredException;
import org.springframework.security.core.userdetails.UserDetails;
import org.springframework.security.core.userdetails.UserDetailsChecker;
import org.springframework.security.core.userdetails.UserDetailsService;
import org.springframework.security.core.userdetails.UsernameNotFoundException;

import java.util.Collections;

import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotEquals;
import static org.mockito.Mockito.any;
import static org.mockito.Mockito.doNothing;
import static org.mockito.Mockito.doReturn;
import static org.mockito.Mockito.when;


@UnitTest
public class PayPalAuthProviderTest {

    private static final String USERNAME = "username";
    private static final String CREDENTIAL = "credential";

    @Mock
    private UserDetailsChecker preAuthenticationChecks;

    @Mock
    private UserService userService;

    @Mock
    private UserDetailsService userDetailsService;

    @Mock
    private AbstractAuthenticationToken authentication;

    @Mock
    private UserDetails userDetails;

    @Mock
    private CustomerModel customerModel;

    @Mock
    private User user;

    @InjectMocks
    @Spy
    private PayPalAuthProvider unit;

    private Object details;
    private ConnectWithPayPalAuthenticationToken expectedResult;

    @Before
    public void setUp() throws Exception {
        MockitoAnnotations.initMocks(this);

        doReturn(Boolean.TRUE).when(unit).isSystemHasCurrentTenant();
        doReturn(Boolean.TRUE).when(unit).isSystemInitialized();
        doReturn(user).when(unit).getUser(any());
        doNothing().when(unit).setUserToCurrentJaloSession(user);

        details = new Object();
        expectedResult = new ConnectWithPayPalAuthenticationToken(USERNAME, Collections.emptyList());
        expectedResult.setDetails(details);
    }

    @Test
    public void shouldCreateSuccessAuthentication() {
        when(userDetails.getUsername()).thenReturn(USERNAME);
        when(userDetails.getAuthorities()).thenReturn(Collections.emptyList());
        when(authentication.getDetails()).thenReturn(details);

        assertEquals(expectedResult, unit.createSuccessAuthentication(authentication, userDetails));
    }

    @Test
    public void shouldNotAuthenticateWhenCurrentTenantNotExist() {
        doReturn(Boolean.FALSE).when(unit).isSystemHasCurrentTenant();
        doReturn(Boolean.FALSE).when(unit).isSystemInitialized();

        assertNotEquals(expectedResult, unit.authenticate(authentication));
    }

    @Test
    public void shouldAuthenticateWhenCurrentTenantExistAndSystemInitialized() {

        when(authentication.getPrincipal()).thenReturn(USERNAME);
        when(authentication.getName()).thenReturn(USERNAME);
        when(userService.getUserForUID(USERNAME)).thenReturn(customerModel);

        when(customerModel.getPayPalAccessToken()).thenReturn(USERNAME);
        when(authentication.getCredentials()).thenReturn(USERNAME);
        when(userDetailsService.loadUserByUsername(USERNAME)).thenReturn(userDetails);
        doReturn(preAuthenticationChecks).when(unit).getPreAuthenticationChecks();

        when(userDetails.isCredentialsNonExpired()).thenReturn(true);

        when(userDetails.getUsername()).thenReturn(USERNAME);
        when(userDetails.getAuthorities()).thenReturn(Collections.emptyList());
        when(authentication.getDetails()).thenReturn(details);

        assertEquals(expectedResult, unit.authenticate(authentication));
    }

    @Test(expected = CredentialsExpiredException.class)
    public void shouldThrowCredentialsExpiredExceptionWhenCredentialExpired() {

        when(authentication.getPrincipal()).thenReturn(USERNAME);
        when(authentication.getName()).thenReturn(USERNAME);
        when(userService.getUserForUID(USERNAME)).thenReturn(customerModel);

        when(customerModel.getPayPalAccessToken()).thenReturn(USERNAME);
        when(authentication.getCredentials()).thenReturn(USERNAME);
        when(userDetailsService.loadUserByUsername(USERNAME)).thenReturn(userDetails);
        doReturn(preAuthenticationChecks).when(unit).getPreAuthenticationChecks();

        when(userDetails.isCredentialsNonExpired()).thenReturn(Boolean.TRUE);

        when(userDetails.getUsername()).thenReturn(USERNAME);
        when(userDetails.getAuthorities()).thenReturn(Collections.emptyList());
        when(userDetails.isCredentialsNonExpired()).thenReturn(Boolean.FALSE);
        when(authentication.getDetails()).thenReturn(details);

        unit.authenticate(authentication);
    }

    @Test(expected = BadCredentialsException.class)
    public void shouldThrowBadCredentialExceptionWhenTokenAndCredentionalAreNotEquals() {

        when(authentication.getPrincipal()).thenReturn(USERNAME);
        when(authentication.getName()).thenReturn(USERNAME);
        when(userService.getUserForUID(USERNAME)).thenReturn(customerModel);
        when(userDetailsService.loadUserByUsername(USERNAME)).thenReturn(userDetails);

        when(customerModel.getPayPalAccessToken()).thenReturn(USERNAME);
        when(authentication.getCredentials()).thenReturn(CREDENTIAL);

        unit.authenticate(authentication);

    }

    @Test(expected = BadCredentialsException.class)
    public void shouldThrowBadCredentialExceptionWhenCredentialNotAString() {

        when(authentication.getPrincipal()).thenReturn(USERNAME);
        when(authentication.getName()).thenReturn(USERNAME);
        when(userService.getUserForUID(USERNAME)).thenReturn(customerModel);
        when(userDetails.isCredentialsNonExpired()).thenReturn(true);
        when(userDetailsService.loadUserByUsername(USERNAME)).thenReturn(userDetails);

        when(customerModel.getPayPalAccessToken()).thenReturn(USERNAME);
        when(authentication.getCredentials()).thenReturn(new Object());

        unit.authenticate(authentication);
    }

    @Test(expected = BadCredentialsException.class)
    public void shouldThrowBadCredentialsExceptionWhenUserNotFoundByUID() {

        when(authentication.getPrincipal()).thenReturn(USERNAME);
        when(authentication.getName()).thenReturn(USERNAME);
        when(userService.getUserForUID(USERNAME)).thenThrow(UsernameNotFoundException.class);

        unit.authenticate(authentication);
    }

}