package com.paypal.hybris.core.util;

import com.fasterxml.jackson.annotation.JsonAutoDetect;
import com.fasterxml.jackson.annotation.JsonInclude;
import com.fasterxml.jackson.annotation.PropertyAccessor;
import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import com.fasterxml.jackson.databind.SerializationFeature;
import com.paypal.hybris.core.util.builder.GenericBuilder;
import com.paypal.hybris.data.CardData;
import com.paypal.hybris.data.PayPalAttributesData;
import com.paypal.hybris.data.PayPalCustomerData;
import com.paypal.hybris.data.PayPalData;
import com.paypal.hybris.data.PayPalOrderResponseData;
import com.paypal.hybris.data.PayPalVaultData;
import com.paypal.hybris.data.PaymentSourceData;
import com.paypal.hybris.data.SimpleAmount;
import de.hybris.platform.commercefacades.product.data.PriceData;
import de.hybris.platform.payment.enums.PaymentTransactionType;
import de.hybris.platform.payment.model.PaymentTransactionEntryModel;
import de.hybris.platform.payment.model.PaymentTransactionModel;
import de.hybris.platform.servicelayer.model.ModelService;
import org.apache.commons.lang3.StringUtils;
import org.apache.log4j.Logger;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.time.Instant;
import java.util.ArrayList;
import java.util.Collections;
import java.util.Date;
import java.util.List;
import java.util.Optional;

public class PayPalCommandsUtil {

    private static final String ZERO_VALUE_STRING = BigDecimal.valueOf(0.0).toString();
    private static final String PARSING_EXCEPTION_MESSAGE = "Could not parse value of type '%s'";
    private static final String ITEMS_TOTAL_MISSING_MESSAGE = "Items total amount can not be calculated for cart/order since cart is null or entries are missing";

    private static final Logger LOG = Logger.getLogger(PayPalCommandsUtil.class);

    private PayPalCommandsUtil() {
    }

    public static String getCustomerIdFromPayPalOrderResponseData(PayPalOrderResponseData orderResponseData) {
        return Optional.ofNullable(orderResponseData.getPaymentSource())
                .map(PaymentSourceData::getCard)
                .map(CardData::getAttributes)
                .map(PayPalAttributesData::getVault)
                .map(PayPalVaultData::getCustomer)
                .map(PayPalCustomerData::getId)
                .orElseGet(() -> Optional.ofNullable(orderResponseData.getPaymentSource())
                        .map(PaymentSourceData::getPaypal)
                        .map(PayPalData::getAttributes)
                        .map(PayPalAttributesData::getVault)
                        .map(PayPalVaultData::getCustomer)
                        .map(PayPalCustomerData::getId).orElse(null));
    }

    public static String getVaultIdFromPayPalOrderResponseData(PayPalOrderResponseData orderResponseData) {
        return Optional.ofNullable(orderResponseData.getPaymentSource())
                .map(PaymentSourceData::getCard)
                .map(CardData::getAttributes)
                .map(PayPalAttributesData::getVault)
                .map(PayPalVaultData::getId)
                .orElseGet(() -> Optional.ofNullable(orderResponseData.getPaymentSource())
                        .map(PaymentSourceData::getPaypal)
                        .map(PayPalData::getAttributes)
                        .map(PayPalAttributesData::getVault)
                        .map(PayPalVaultData::getId).orElse(null));
    }

    public static String getValueAsString(Object value) {
        ObjectMapper objectMapper = new ObjectMapper();

        objectMapper.configure(SerializationFeature.FAIL_ON_EMPTY_BEANS, false);
        objectMapper.configure(SerializationFeature.INDENT_OUTPUT, true);
        objectMapper.setDefaultPropertyInclusion(JsonInclude.Include.ALWAYS);
        objectMapper.setVisibility(PropertyAccessor.FIELD, JsonAutoDetect.Visibility.ANY);

        try {
            return objectMapper.writeValueAsString(value);
        } catch (JsonProcessingException e) {
            LOG.warn(PARSING_EXCEPTION_MESSAGE.formatted(value.getClass()), e);
            return StringUtils.EMPTY;
        }
    }

    public static void saveTransactionEntryToTransaction(ModelService service, PaymentTransactionModel transaction,
                                                         PaymentTransactionEntryModel entry) {
        List<PaymentTransactionEntryModel> existingTransactions
                = transaction.getEntries() == null ? Collections.emptyList() : transaction.getEntries();
        List<PaymentTransactionEntryModel> newTransactionEntries = new ArrayList<>(existingTransactions);

        newTransactionEntries.add(entry);
        transaction.setEntries(newTransactionEntries);
        service.save(transaction);
        service.refresh(transaction);
        service.save(entry);
    }

    public static String getNewPaymentTransactionEntryCode(PaymentTransactionModel transaction,
                                                    PaymentTransactionType paymentTransactionType) {
        return transaction.getEntries() == null ?
                transaction.getCode() + "-" + paymentTransactionType.getCode() + "-1" :
                transaction.getCode() + "-" + paymentTransactionType.getCode() + "-" + (transaction.getEntries().size()
                        + 1);
    }

    public static String getPriceValue(SimpleAmount simpleAmount) {
        return Optional.ofNullable(simpleAmount).map(SimpleAmount::getValue).orElse(ZERO_VALUE_STRING);
    }

    public static String getPriceValue(PriceData priceData) {
        return Optional.ofNullable(priceData).map(PriceData::getValue).map(String::valueOf).orElse(ZERO_VALUE_STRING);
    }

    public static SimpleAmount createAmount(Double value, String currency) {
        String price = String.valueOf(BigDecimal.valueOf(value).setScale(2, RoundingMode.HALF_UP));
        return GenericBuilder.of(SimpleAmount::new)
                .with(SimpleAmount::setValue, price)
                .with(SimpleAmount::setCurrencyCode, currency)
                .build();
    }

    public static SimpleAmount createAmount(BigDecimal value, String currency) {
		String price = String.valueOf(value.setScale(2, RoundingMode.HALF_UP));
		return GenericBuilder.of(SimpleAmount::new)
				.with(SimpleAmount::setValue, price)
				.with(SimpleAmount::setCurrencyCode, currency)
				.build();
	}

    public static Date convertDate(String date) {
        if(date != null) {
            return Date.from(Instant.parse(date));
        }
        return new Date();
    }
}
