package com.paypal.hybris.core.strategy.impl;

import static com.paypal.hybris.core.constants.PaypalcoreConstants.PAYPAL_INTENT_CAPTURE;
import static com.paypal.hybris.core.constants.PaypalcoreConstants.SAVED_PAYMENT;

import com.paypal.hybris.core.enums.PaymentProvider;
import com.paypal.hybris.core.enums.PaymentStatusType;
import com.paypal.hybris.core.model.PayPalCreditCardPaymentInfoModel;
import com.paypal.hybris.core.service.PayPalPaymentService;
import com.paypal.hybris.data.PayPalCommerceCheckoutParameter;
import de.hybris.platform.commerceservices.order.impl.DefaultCommercePaymentAuthorizationStrategy;
import de.hybris.platform.commerceservices.service.data.CommerceCheckoutParameter;
import de.hybris.platform.core.model.order.CartModel;
import de.hybris.platform.payment.dto.TransactionStatus;
import de.hybris.platform.payment.model.PaymentTransactionEntryModel;
import de.hybris.platform.payment.model.PaymentTransactionModel;
import java.math.BigDecimal;
import java.util.Currency;
import java.util.Optional;


public class DefaultPayPalVaultedPaymentStrategy extends DefaultCommercePaymentAuthorizationStrategy {

    private PayPalPaymentService payPalPaymentService;

    @Override
    public PaymentTransactionEntryModel authorizePaymentAmount(final CommerceCheckoutParameter parameter) {
        final CartModel cartModel = parameter.getCart();

        PaymentTransactionEntryModel transactionEntryModel = null;
        try {
            beforeAuthorizePaymentAmount(parameter);
            if (cartModel.getPaymentInfo() instanceof PayPalCreditCardPaymentInfoModel) {
                transactionEntryModel = processPayPalPaymentUsingHostedFields(parameter);
            }
            return transactionEntryModel;
        } finally {
            afterAuthorizePaymentAmount(parameter, transactionEntryModel);
        }
    }

    private PaymentTransactionEntryModel processPayPalPaymentUsingHostedFields(CommerceCheckoutParameter parameter) {
        PaymentTransactionEntryModel transactionEntryModel;

        final CartModel cartModel = parameter.getCart();
        final BigDecimal amount = parameter.getAuthorizationAmount();
        final String paymentProvider = parameter.getPaymentProvider();
        String scaFlow = Optional.ofNullable(parameter)
                .filter(PayPalCommerceCheckoutParameter.class::isInstance)
                .map(PayPalCommerceCheckoutParameter.class::cast)
                .map(PayPalCommerceCheckoutParameter::getStoredCredentialFlow).orElse(SAVED_PAYMENT);
        final PayPalCreditCardPaymentInfoModel paymentInfo = (PayPalCreditCardPaymentInfoModel) cartModel
                .getPaymentInfo();
        final PaymentProvider payPalPaymentProvider = paymentInfo.getPaymentProvider();
        final Currency currency = getI18nService()
                .getBestMatchingJavaCurrency(cartModel.getCurrency().getIsocode());
        final String merchantTransactionCode = getGenerateMerchantTransactionCodeStrategy()
                .generateCode(cartModel);
        if(PAYPAL_INTENT_CAPTURE.equals(paymentInfo.getIntent())) {
            transactionEntryModel = getPayPalPaymentService()
                    .createStubAuthorize(merchantTransactionCode, amount, currency, paymentProvider, paymentInfo);
            final PaymentTransactionModel paymentTransaction = transactionEntryModel.getPaymentTransaction();
            paymentTransaction.setOrder(cartModel);
            paymentTransaction.setInfo(paymentInfo);
            getPayPalPaymentService().captureVaultedPayment(paymentTransaction, scaFlow, payPalPaymentProvider);
            cartModel.setPaymentStatusType(PaymentStatusType.COMPLETED);
            getModelService().saveAll(cartModel, paymentTransaction);
        } else {
            transactionEntryModel = getPayPalPaymentService().authorizeVaultedPayment(merchantTransactionCode, amount,
                    currency, cartModel, paymentProvider, scaFlow, payPalPaymentProvider);
            if (transactionEntryModel != null) {
                final PaymentTransactionModel paymentTransaction = transactionEntryModel.getPaymentTransaction();
                if (TransactionStatus.ACCEPTED.name().equals(transactionEntryModel.getTransactionStatus())
                        || TransactionStatus.REVIEW.name().equals(transactionEntryModel.getTransactionStatus())) {
                    cartModel.setPaymentStatusType(PaymentStatusType.PENDING);
                    paymentTransaction.setOrder(cartModel);
                    paymentTransaction.setInfo(paymentInfo);
                    getModelService().saveAll(cartModel, paymentTransaction);
                }
            }
        }
        return transactionEntryModel;
    }

    public PayPalPaymentService getPayPalPaymentService() {
        return payPalPaymentService;
    }

    public void setPayPalPaymentService(PayPalPaymentService payPalPaymentService) {
        this.payPalPaymentService = payPalPaymentService;
    }

}
