package com.paypal.hybris.core.service.impl;

import com.paypal.hybris.core.dao.PaymentTransactionsDao;
import com.paypal.hybris.core.enums.PaymentStatusType;
import com.paypal.hybris.core.service.PaymentTransactionsService;
import de.hybris.backoffice.PaypalWebhookData;
import de.hybris.platform.core.enums.OrderStatus;
import de.hybris.platform.core.model.order.AbstractOrderEntryModel;
import de.hybris.platform.core.model.order.AbstractOrderModel;
import de.hybris.platform.core.model.order.OrderModel;
import de.hybris.platform.core.model.security.PrincipalModel;
import de.hybris.platform.ordercancel.OrderCancelEntry;
import de.hybris.platform.ordercancel.OrderCancelRecordsHandler;
import de.hybris.platform.ordercancel.OrderCancelRequest;
import de.hybris.platform.ordercancel.OrderCancelResponse;
import de.hybris.platform.ordercancel.OrderStatusChangeStrategy;
import de.hybris.platform.ordercancel.OrderUtils;
import de.hybris.platform.ordercancel.exceptions.OrderCancelRecordsHandlerException;
import de.hybris.platform.ordercancel.model.OrderCancelRecordEntryModel;
import de.hybris.platform.payment.dto.TransactionStatus;
import de.hybris.platform.payment.dto.TransactionStatusDetails;
import de.hybris.platform.payment.enums.PaymentTransactionType;
import de.hybris.platform.payment.model.PaymentTransactionEntryModel;
import de.hybris.platform.payment.model.PaymentTransactionModel;
import de.hybris.platform.processengine.BusinessProcessService;
import de.hybris.platform.servicelayer.model.ModelService;
import org.apache.log4j.Logger;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;
import java.util.Optional;
import java.util.function.Predicate;

/**
 * This class is a default implementation of the PaymentTransactionsService interface
 */
public class DefaultPaymentTransactionsService implements PaymentTransactionsService {

    private static final String REVIEW_DECISION = "_ReviewDecision";
    private static final String TRANSACTION_STATUS_VOIDED = "CANCELED";
    private static final String SUCCESSFUL_TRANSACTION_STATUS_DETAILS = "SUCCESFULL";
    private static final Logger LOG = Logger.getLogger(DefaultPaymentTransactionsService.class);
    private PaymentTransactionsDao transactionsDao;
    private ModelService modelService;
    private BusinessProcessService businessProcessService;
    private DefaultPayPalPaymentService paypalPaymentService;
    private OrderCancelRecordsHandler orderCancelRecordsHandler;
    private OrderStatusChangeStrategy completeCancelStatusChangeStrategy;


    public void doOnVoidPerformed(PaypalWebhookData paypalWebhookData) {
        final PaymentTransactionEntryModel authorizationEntry = getTransactionEntryById(paypalWebhookData.getResource().getId());
        if (authorizationEntry != null && PaymentTransactionType.AUTHORIZATION
                .equals(authorizationEntry.getType())) {
            PaymentTransactionModel paymentTransaction = authorizationEntry.getPaymentTransaction();

            try {
                OrderCancelRequest orderCancelRequest = buildCancelRequest((OrderModel) paymentTransaction.getOrder());
                OrderCancelRecordEntryModel result = this.orderCancelRecordsHandler.createRecordEntry(orderCancelRequest, new PrincipalModel());

                OrderModel order = orderCancelRequest.getOrder();
                this.doCancel(order);
                this.modelService.refresh(order);

                if (!OrderUtils.hasLivingEntries(order) && this.completeCancelStatusChangeStrategy != null) {
                    this.completeCancelStatusChangeStrategy
                            .changeOrderStatusAfterCancelOperation(result, true);
                }

                this.orderCancelRecordsHandler
                        .updateRecordEntry(this.makeInternalResponse(orderCancelRequest, true, null));
            } catch (OrderCancelRecordsHandlerException e) {
                LOG.info(e.getMessage(), e);
            }
        }
    }

    public void doOnCapturePerformed(PaypalWebhookData paypalWebhookData) {
        Optional<String> authorizationId = Optional.ofNullable(paypalWebhookData
                .getResource().getSupplementaryData().getRelatedIds().getAuthorizationId());

        Optional<PaymentTransactionEntryModel> captureTransaction = Optional
                .ofNullable(paypalWebhookData.getResource().getId())
                .map(this::getTransactionEntryById);

        if (authorizationId.isPresent() && captureTransaction.isEmpty()) {
            doCapture(paypalWebhookData);
        } else if (authorizationId.isEmpty()) {
            LOG.info("Authorization ID is missing");
        } else {
            LOG.info("Order already captured. Capture id:" + paypalWebhookData.getResource().getId());
        }
    }

    private void doCapture(PaypalWebhookData paypalWebhookData) {
        final PaymentTransactionEntryModel authorizationEntry = getTransactionEntryById(paypalWebhookData.getResource()
                .getSupplementaryData().getRelatedIds().getAuthorizationId());
        final PaymentTransactionModel paymentTransaction = authorizationEntry.getPaymentTransaction();
        final AbstractOrderModel currentOrder = paymentTransaction.getOrder();
        final BigDecimal amountToCapture = new BigDecimal(paypalWebhookData.getResource().getAmount().getValue());
        final PaymentTransactionEntryModel newCaptureTransactionEntry = createCaptureTransactionEntry(paypalWebhookData, paymentTransaction);
        if (newCaptureTransactionEntry.getType().equals(PaymentTransactionType.PARTIAL_CAPTURE)) {
            currentOrder.setPaymentStatusType(PaymentStatusType.PARTIAL_CAPTURE);
        }
        modelService.saveAll(newCaptureTransactionEntry, currentOrder);
        if (isOrderFullyCaptured(newCaptureTransactionEntry, amountToCapture)) {
            executeManualPaymentCaptureOperation((OrderModel) currentOrder);
        }
    }

    public void doOnRefundPerformed(PaypalWebhookData paypalWebhookData, String captureId) {
        final PaymentTransactionEntryModel transactionEntryToRefund = getTransactionEntryById(captureId);
        final PaymentTransactionModel paymentTransaction = transactionEntryToRefund.getPaymentTransaction();
        final AbstractOrderModel currentOrder = paymentTransaction.getOrder();
        final BigDecimal amountToRefund = new BigDecimal(paypalWebhookData.getResource().getAmount().getValue());
        final BigDecimal availableAmount = transactionEntryToRefund.getAmount().subtract(calculateRefundedAmount(transactionEntryToRefund));
        final PaymentTransactionType paymentTransactionType = (availableAmount.compareTo(amountToRefund) > 0) ?
                PaymentTransactionType.REFUND_PARTIAL : PaymentTransactionType.REFUND_STANDALONE;
        final PaymentStatusType paymentStatusType = paymentTransactionType.equals(PaymentTransactionType.REFUND_PARTIAL) ?
                PaymentStatusType.PARTIAL_REFUND : PaymentStatusType.REFUNDED;
        PaymentTransactionEntryModel refundTransactionEntryModel = createRefundTransaction(paypalWebhookData, amountToRefund, paymentTransactionType, paymentTransaction, captureId);
        currentOrder.setPaymentStatusType(paymentStatusType);
        modelService.saveAll(currentOrder, refundTransactionEntryModel);
    }

    private PaymentTransactionEntryModel createCaptureTransactionEntry(PaypalWebhookData paypalWebhookData, PaymentTransactionModel paymentTransaction) {
        final PaymentTransactionEntryModel entry = this.modelService.create(PaymentTransactionEntryModel.class);
        final BigDecimal amount = new BigDecimal(paypalWebhookData.getResource().getAmount().getValue());
        final boolean isFullCapture = paymentTransaction.getPlannedAmount().compareTo(amount) <= 0;
        final PaymentTransactionType paymentTransactionType = isFullCapture ?
                PaymentTransactionType.CAPTURE : PaymentTransactionType.PARTIAL_CAPTURE;
        entry.setAmount(amount);
        entry.setCurrency(paymentTransaction.getCurrency());
        entry.setType(paymentTransactionType);
        entry.setRequestId(paypalWebhookData.getResource().getId());
        entry.setTime(new Date());
        entry.setPaymentTransaction(paymentTransaction);
        entry.setTransactionStatus(TransactionStatus.ACCEPTED.toString());
        entry.setTransactionStatusDetails(TransactionStatusDetails.SUCCESFULL.toString());
        entry.setCode(getNewPaymentTransactionEntryCode(paymentTransaction, paymentTransactionType));
        return entry;
    }

    private PaymentTransactionEntryModel createRefundTransaction(final PaypalWebhookData paypalWebhookData, final BigDecimal amount,
                                                                 final PaymentTransactionType transactionType, final PaymentTransactionModel paymentTransaction, final String captureId) {
        PaymentTransactionEntryModel entry = this.modelService.create(PaymentTransactionEntryModel.class);
        entry.setCurrency(paymentTransaction.getCurrency());
        entry.setType(transactionType);
        entry.setTime(new Date());
        entry.setPaymentTransaction(paymentTransaction);
        entry.setAmount(amount);
        entry.setRequestId(paypalWebhookData.getResource().getId());
        entry.setRequestToken(paymentTransaction.getRequestToken());
        entry.setSubscriptionID(captureId);
        entry.setTransactionStatus(TransactionStatus.ACCEPTED.toString());
        entry.setTransactionStatusDetails(TransactionStatusDetails.SUCCESFULL.toString());
        entry.setCode(getNewPaymentTransactionEntryCode(paymentTransaction, transactionType));
        return entry;
    }

    private void cancel(PaymentTransactionEntryModel transactionEntry) {
        PaymentTransactionType transactionType = PaymentTransactionType.CANCEL;
        String newEntryCode = this.paypalPaymentService.getNewPaymentTransactionEntryCode(transactionEntry.getPaymentTransaction(), transactionType);
        PaymentTransactionEntryModel entry = this.modelService.create(PaymentTransactionEntryModel.class);
        if (transactionEntry.getCurrency() != null) {
            entry.setCurrency(transactionEntry.getCurrency());
        }
        entry.setType(transactionType);
        entry.setTime(new Date());
        entry.setPaymentTransaction(transactionEntry.getPaymentTransaction());
        entry.setRequestId(transactionEntry.getRequestId());
        entry.setAmount(transactionEntry.getAmount());
        entry.setRequestToken(transactionEntry.getRequestToken());
        entry.setTransactionStatus(TransactionStatus.ACCEPTED.toString());
        entry.setTransactionStatusDetails(TransactionStatusDetails.SUCCESFULL.toString());
        entry.setCode(newEntryCode);

        transactionEntry.setTransactionStatus(TRANSACTION_STATUS_VOIDED);
        this.modelService.save(transactionEntry);
        this.modelService.save(entry);
    }

    @Override
    public BigDecimal calculateRefundedAmount(PaymentTransactionEntryModel transaction) {
        final String captureRequestId = transaction.getRequestId();
        return transaction.getPaymentTransaction().getOrder().getPaymentTransactions().stream()
                .flatMap(transactionModel -> transactionModel.getEntries().stream())
                .filter(entry -> isRefundForThisCapture(captureRequestId, entry))
                .map(PaymentTransactionEntryModel::getAmount)
                .reduce(BigDecimal.ZERO, BigDecimal::add);
    }

    private boolean isRefundForThisCapture(String captureRequestId, PaymentTransactionEntryModel transactionEntry) {
        return (PaymentTransactionType.REFUND_PARTIAL.equals(transactionEntry.getType()) ||
                PaymentTransactionType.REFUND_STANDALONE.equals(transactionEntry.getType()))
                && TransactionStatus.ACCEPTED.name().equals(transactionEntry.getTransactionStatus())
                && captureRequestId.equals(transactionEntry.getSubscriptionID());
    }

    @Override
    public List<PaymentTransactionEntryModel> getCaptureTransactionEntries(OrderModel orderModel) {
        return orderModel.getPaymentTransactions().stream()
                .flatMap(transactionEntity -> transactionEntity.getEntries().stream())
                .filter(this::isSuccessfulCaptureTransactionEntry)
                .toList();
    }

    private boolean isSuccessfulCaptureTransactionEntry(PaymentTransactionEntryModel transactionEntry) {
        return (PaymentTransactionType.CAPTURE.equals(transactionEntry.getType())
                || PaymentTransactionType.PARTIAL_CAPTURE.equals(transactionEntry.getType()))
                && SUCCESSFUL_TRANSACTION_STATUS_DETAILS.equals(transactionEntry.getTransactionStatusDetails());
    }

    @Override
    public List<PaymentTransactionEntryModel> getTransactionsToRefund(OrderModel orderModel) {
        return this.getCaptureTransactionEntries(orderModel).stream()
                .filter(this::isTransactionRefundable)
                .toList();
    }

    private boolean isTransactionRefundable(PaymentTransactionEntryModel transactionEntryModel) {
        BigDecimal capturedAmount = transactionEntryModel.getAmount();
        BigDecimal refundedAmount = this.calculateRefundedAmount(transactionEntryModel);
        return !capturedAmount.equals(refundedAmount);
    }

    protected void executeManualPaymentCaptureOperation(OrderModel order) {
        order.getOrderProcess().stream()
                .filter(process -> process.getCode().startsWith(order.getStore().getSubmitOrderProcessCode()))
                .forEach(filteredProcess -> businessProcessService
                        .triggerEvent(filteredProcess.getOrder().getCode() + REVIEW_DECISION));
        LOG.info(String.format("Payment Capture Manual Release completed. %s triggered.", "ReviewDecision"));
        order.setStatus(OrderStatus.PAYMENT_CAPTURED);
        order.setPaymentStatusType(PaymentStatusType.COMPLETED);
        modelService.save(order);
    }

    protected OrderCancelResponse makeInternalResponse(OrderCancelRequest request, boolean success, String message) {
        if (success) {
            return request.isPartialCancel()
                    ? new OrderCancelResponse(request.getOrder(), request.getEntriesToCancel())
                    : new OrderCancelResponse(request.getOrder());
        }
        return new OrderCancelResponse(request.getOrder(), request.getEntriesToCancel(),
                OrderCancelResponse.ResponseStatus.error, message);
    }

    protected OrderCancelRequest buildCancelRequest(OrderModel order) {
        if (order != null) {
            List<OrderCancelEntry> orderCancelEntries = new ArrayList<>();
            order.getEntries()
                    .forEach(entry -> createOrderCancelEntry(orderCancelEntries, entry));
            return new OrderCancelRequest(order, orderCancelEntries);

        }
        return null;
    }

    protected void createOrderCancelEntry(List<OrderCancelEntry> orderCancelEntries, Object entry) {
        AbstractOrderEntryModel orderEntryToCancel = (AbstractOrderEntryModel) entry;
        OrderCancelEntry orderCancelEntry = new OrderCancelEntry(orderEntryToCancel);
        orderCancelEntries.add(orderCancelEntry);
    }

    private void doCancel(OrderModel order) {
        List<PaymentTransactionEntryModel> cancelableTransactions = this.paypalPaymentService.getCancelableTransactions(order);
        for (PaymentTransactionEntryModel cancelableTransaction : cancelableTransactions) {
            cancel(cancelableTransaction);
        }
        if (this.paypalPaymentService.isOrderCanceled(order)) {
            order.setStatus(OrderStatus.CANCELLED);
            order.setPaymentStatusType(PaymentStatusType.CANCELLED);
            this.modelService.save(order);
            LOG.info("Order: " + order.getCode() + " has been cancelled by webhook!\n Order status: " + order.getStatus());
        }
    }

    private boolean isOrderFullyCaptured(PaymentTransactionEntryModel captureTransactionEntry, BigDecimal amount) {
        BigDecimal result = captureTransactionEntry.getPaymentTransaction().getEntries().stream()
                .filter(getCaptureEntriesPredicate()).map(PaymentTransactionEntryModel::getAmount)
                .reduce(BigDecimal.ZERO, BigDecimal::add);

        return result.add(amount).compareTo(getAuthorizedAmount(captureTransactionEntry)) >= 0;
    }

    private String getNewPaymentTransactionEntryCode(PaymentTransactionModel transaction,
                                                     PaymentTransactionType paymentTransactionType) {
        return transaction.getEntries() == null ?
                transaction.getCode() + "-" + paymentTransactionType.getCode() + "-1" :
                transaction.getCode() + "-" + paymentTransactionType.getCode() + "-" + (transaction.getEntries().size()
                        + 1);
    }

    private BigDecimal getAuthorizedAmount(PaymentTransactionEntryModel captureTransactionEntry) {
        Predicate<PaymentTransactionEntryModel> authorizationEntries = (entry -> (PaymentTransactionType.AUTHORIZATION
                .equals(entry.getType())) && TransactionStatus.ACCEPTED.name().equals(entry.getTransactionStatus()));

        return captureTransactionEntry.getPaymentTransaction().getOrder().getPaymentTransactions().stream().flatMap(transaction -> transaction.getEntries().stream())
                .filter(authorizationEntries).map(PaymentTransactionEntryModel::getAmount)
                .reduce(BigDecimal.ZERO, BigDecimal::add);
    }

    private Predicate<PaymentTransactionEntryModel> getCaptureEntriesPredicate() {
        return entry ->
                (PaymentTransactionType.PARTIAL_CAPTURE.equals(entry.getType()) || PaymentTransactionType.CAPTURE
                        .equals(entry.getType()))
                        && TransactionStatus.ACCEPTED.name().equals(entry.getTransactionStatus());
    }

    @Override
    public PaymentTransactionEntryModel getTransactionEntryById(String transactionId) {
        return transactionsDao.getTransactionEntryById(transactionId);
    }

    public void setTransactionsDao(PaymentTransactionsDao transactionsDao) {
        this.transactionsDao = transactionsDao;
    }

    public void setModelService(ModelService modelService) {
        this.modelService = modelService;
    }

    public void setBusinessProcessService(BusinessProcessService businessProcessService) {
        this.businessProcessService = businessProcessService;
    }

    public void setPaypalPaymentService(DefaultPayPalPaymentService paypalPaymentService) {
        this.paypalPaymentService = paypalPaymentService;
    }

    public void setCompleteCancelStatusChangeStrategy(OrderStatusChangeStrategy completeCancelStatusChangeStrategy) {
        this.completeCancelStatusChangeStrategy = completeCancelStatusChangeStrategy;
    }

    public void setOrderCancelRecordsHandler(OrderCancelRecordsHandler orderCancelRecordsHandler) {
        this.orderCancelRecordsHandler = orderCancelRecordsHandler;
    }

}