package com.paypal.hybris.core.service.impl;

import com.paypal.hybris.core.commands.impl.DefaultPayPalSendTrackingInfoCommand;
import com.paypal.hybris.core.service.PayPalTrackingInfoService;
import com.paypal.hybris.data.OrderItemData;
import com.paypal.hybris.data.PayPalTrackingInfoData;
import com.paypal.hybris.data.PayPalTrackingInfoRequestData;
import de.hybris.platform.core.model.order.AbstractOrderEntryModel;
import de.hybris.platform.ordersplitting.model.ConsignmentEntryModel;
import de.hybris.platform.ordersplitting.model.ConsignmentModel;
import de.hybris.platform.payment.commands.factory.CommandFactoryRegistry;
import de.hybris.platform.payment.model.PaymentTransactionEntryModel;
import de.hybris.platform.servicelayer.dto.converter.Converter;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.util.List;

import static com.paypal.hybris.core.constants.PaypalcoreConstants.PAYPAL_PROVIDER_NAME;

public class DefaultPayPalTrackingInfoService implements PayPalTrackingInfoService {

    private static final Logger LOG = LoggerFactory.getLogger(DefaultPayPalTrackingInfoService.class);

    private static final String TRACKING_INFO_ERROR_MESSAGE = "Error while sending tracking info for consignment: %s. %s";
    private Converter<AbstractOrderEntryModel, OrderItemData> orderItemDataConverter;
    private CommandFactoryRegistry commandFactoryRegistry;

    @Override
    public String sendTrackingInfoToPayPalForConsignment(PaymentTransactionEntryModel transactionEntryModel,
                                                         ConsignmentModel consignmentModel,
                                                         String carrier, String trackingNumber) {
        PayPalTrackingInfoData trackingInfoData = new PayPalTrackingInfoData();
        PayPalTrackingInfoRequestData trackingInfoRequest = new PayPalTrackingInfoRequestData();

        String captureId = transactionEntryModel.getRequestId();
        String orderId = transactionEntryModel.getPaymentTransaction().getOrder().getPayPalOrderId();

        trackingInfoRequest.setCaptureId(captureId);
        trackingInfoRequest.setTrackingNumber(trackingNumber);
        trackingInfoRequest.setCarrier(carrier);
        trackingInfoRequest.setItems(getOrderItemsFromConsignment(consignmentModel));

        trackingInfoData.setTrackingInfo(trackingInfoRequest);
        trackingInfoData.setOrderId(orderId);

        String resultStatus = null;
        try {
            resultStatus = commandFactoryRegistry.getFactory(PAYPAL_PROVIDER_NAME)
                    .createCommand(DefaultPayPalSendTrackingInfoCommand.class)
                    .perform(trackingInfoData);
        } catch (Exception e) {
            LOG.error(TRACKING_INFO_ERROR_MESSAGE.formatted(consignmentModel.getCode(), e));
        }

        return resultStatus;
    }

    @Override
    public List<OrderItemData> getOrderItemsFromConsignment(ConsignmentModel consignmentModel) {
        return consignmentModel.getConsignmentEntries()
                .stream()
                .map(ConsignmentEntryModel::getOrderEntry)
                .map(entry -> orderItemDataConverter.convert(entry))
                .toList();
    }

    public void setOrderItemDataConverter(Converter<AbstractOrderEntryModel, OrderItemData> orderItemDataConverter) {
        this.orderItemDataConverter = orderItemDataConverter;
    }

    public void setCommandFactoryRegistry(CommandFactoryRegistry commandFactoryRegistry) {
        this.commandFactoryRegistry = commandFactoryRegistry;
    }

}
