package com.paypal.hybris.core.service.impl;

import com.paypal.api.payments.Event;
import com.paypal.hybris.core.commands.impl.DefaultPayPalCaptureIntentCommand;
import com.paypal.hybris.core.commands.impl.DefaultPayPalConvertBAToPaymentTokensCommand;
import com.paypal.hybris.core.commands.impl.DefaultPayPalCreateOrderCommand;
import com.paypal.hybris.core.commands.impl.DefaultPayPalGetCardDetailsCommand;
import com.paypal.hybris.core.commands.impl.DefaultPayPalGetEventCommand;
import com.paypal.hybris.core.commands.impl.DefaultPayPalGetOrderDetailsCommand;
import com.paypal.hybris.core.commands.impl.DefaultPayPalProcessVaultedPaymentCommand;
import com.paypal.hybris.core.commands.impl.DefaultPayPalReauthorizationRequestCommand;
import com.paypal.hybris.core.commands.impl.DefaultPayPalSaveOrderCommand;
import com.paypal.hybris.core.commands.impl.DefaultPayPalUpdateOrderAmountCommand;
import com.paypal.hybris.core.enums.PaymentProvider;
import com.paypal.hybris.core.enums.PaymentStatusType;
import com.paypal.hybris.core.exception.PayPalHttpClientErrorException;
import com.paypal.hybris.core.exception.PayPalProcessPaymentException;
import com.paypal.hybris.core.exception.PayPalProcessVaultedPaymentException;
import com.paypal.hybris.core.exception.PayPalRefundAdapterException;
import com.paypal.hybris.core.exception.PayPalVoidAdapterException;
import com.paypal.hybris.core.model.PayPalCreditCardPaymentInfoModel;
import com.paypal.hybris.core.results.PayPalAuthorizationResult;
import com.paypal.hybris.core.results.PayPalCaptureResult;
import com.paypal.hybris.core.results.PayPalRefundResult;
import com.paypal.hybris.core.results.PayPalVaultedPaymentResult;
import com.paypal.hybris.core.results.PayPalVoidResult;
import com.paypal.hybris.core.service.PayPalConfigurationService;
import com.paypal.hybris.core.service.PayPalCustomerAccountService;
import com.paypal.hybris.core.service.PayPalPaymentService;
import com.paypal.hybris.core.strategy.storedcredential.StoredCredentialStrategy;
import com.paypal.hybris.core.util.PayPalCommandsUtil;
import com.paypal.hybris.core.util.builder.GenericBuilder;
import com.paypal.hybris.data.CardData;
import com.paypal.hybris.data.PayPalConvertBAToPaymentTokensRequestData;
import com.paypal.hybris.data.PayPalGetCardDetailsResponseData;
import com.paypal.hybris.data.PayPalHostedFieldsProcessRequestData;
import com.paypal.hybris.data.PayPalOrderDetailsData;
import com.paypal.hybris.data.PayPalOrderRequestData;
import com.paypal.hybris.data.PayPalVaultedPaymentProcessRequestData;
import com.paypal.hybris.data.StoredCredentialData;
import de.hybris.platform.core.enums.OrderStatus;
import de.hybris.platform.core.model.c2l.CurrencyModel;
import de.hybris.platform.core.model.order.AbstractOrderModel;
import de.hybris.platform.core.model.order.CartModel;
import de.hybris.platform.core.model.order.OrderModel;
import de.hybris.platform.core.model.user.CustomerModel;
import de.hybris.platform.payment.AdapterException;
import de.hybris.platform.payment.commands.factory.CommandFactory;
import de.hybris.platform.payment.commands.factory.CommandFactoryRegistry;
import de.hybris.platform.payment.commands.factory.CommandNotSupportedException;
import de.hybris.platform.payment.commands.request.AuthorizationRequest;
import de.hybris.platform.payment.commands.request.CaptureRequest;
import de.hybris.platform.payment.commands.request.FollowOnRefundRequest;
import de.hybris.platform.payment.commands.request.SubscriptionAuthorizationRequest;
import de.hybris.platform.payment.commands.request.VoidRequest;
import de.hybris.platform.payment.commands.result.AuthorizationResult;
import de.hybris.platform.payment.commands.result.CaptureResult;
import de.hybris.platform.payment.dto.BillingInfo;
import de.hybris.platform.payment.dto.TransactionStatus;
import de.hybris.platform.payment.dto.TransactionStatusDetails;
import de.hybris.platform.payment.enums.PaymentTransactionType;
import de.hybris.platform.payment.impl.DefaultPaymentServiceImpl;
import de.hybris.platform.payment.model.PaymentTransactionEntryModel;
import de.hybris.platform.payment.model.PaymentTransactionModel;
import de.hybris.platform.servicelayer.dto.converter.Converter;
import com.paypal.hybris.data.PayPalHostedFieldsOrderRequestData;
import de.hybris.platform.servicelayer.user.UserService;
import org.apache.commons.lang.StringUtils;
import org.apache.log4j.Logger;
import org.apache.logging.log4j.util.Strings;

import java.math.BigDecimal;
import java.util.Currency;
import java.util.Date;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Predicate;

import static com.paypal.hybris.core.constants.PaypalcoreConstants.PAYPAL_CHECKOUT_PAYMENT_ERROR_MSG;
import static com.paypal.hybris.core.constants.PaypalcoreConstants.PAYPAL_INTENT_CAPTURE;
import static com.paypal.hybris.core.constants.PaypalcoreConstants.PAYPAL_PROVIDER_NAME;

/**
 * This class is a default implementation of the PayPalPaymentService interface
 */
public class DefaultPayPalPaymentService extends DefaultPaymentServiceImpl implements PayPalPaymentService {

    private static final String TRANSACTION_STATUS_VOIDED = "CANCELED";

    private static final String ORDER_NOT_SAVED = "Order not saved";
    private static final String REFUND_ERROR = "Refund is not available";
    private static final String UPDATING_ORDER_ERROR = "[PayPal Payment Service] Errors during updating order details: ";
    private static final String EXPIRATION_DATE_SEPARATOR = "-";
    private static final String AUTHORIZATION_ENTRY_MISSING_EXCEPTION =
            "Could not capture payment without authorization entry. Transaction number: '%s', Order number: '%s' ";
    private static final String FAILURE_REASON_MESSAGE = "Status code: '%s', Status text: '%s', Message from PayPal: '%s'";

    private static final String PAYPAL_DEBUG_ID = "Paypal-Debug-Id";
    private static final String UPDATE_ORDER_DETAILS_MESSAGE = "UpdateOrderDetails orderId: '%s', orderAmount: '%s', currency: '%s'";

    private static final Logger LOG = Logger.getLogger(DefaultPayPalPaymentService.class);

    private CommandFactoryRegistry commandFactoryRegistry;

    private DefaultPayPalPaymentInfoService paymentInfoService;

    private PayPalConfigurationService defaultPayPalConfigurationService;

    private UserService userService;

    private List<StoredCredentialStrategy> storedCredentialStrategies;
    private StoredCredentialStrategy emptyStoredCredentialStrategy;
    private PayPalCustomerAccountService payPalCustomerAccountService;
    private DefaultPaymentTransactionsService paymentTransactionsService;
    private Converter<AbstractOrderModel, PayPalOrderRequestData> orderRequestDataConverter;


    @Override
    public PaymentTransactionEntryModel refund(final PaymentTransactionEntryModel selectedEntry,
                                               final BigDecimal amount) {
        final PaymentTransactionModel transaction = selectedEntry.getPaymentTransaction();
        if (!(transaction.getInfo() instanceof PayPalCreditCardPaymentInfoModel)) {
            return super.refundFollowOn(transaction, amount);
        }
        if (isAvailableRefund(selectedEntry)) {
            final AbstractOrderModel orderModel = transaction.getOrder();

            PaymentTransactionEntryModel entry = getRefundTransaction(selectedEntry, amount, transaction);

            this.getModelService().attach(entry);
            this.getModelService().saveAll(orderModel, entry);
            this.getModelService().refresh(transaction);
            return entry;
        }
        throw new AdapterException(REFUND_ERROR);

    }

    private PaymentTransactionEntryModel getRefundTransaction(PaymentTransactionEntryModel selectedEntry, BigDecimal amount, PaymentTransactionModel transaction) {
        final PaymentTransactionType transactionType = this.getTransactionRefundType(selectedEntry, amount);
        final String newEntryCode = this.getNewPaymentTransactionEntryCode(selectedEntry.getPaymentTransaction(), transactionType);
        final Currency currency = Currency.getInstance(selectedEntry.getCurrency().getIsocode());
        final FollowOnRefundRequest followOnRefundRequest = new FollowOnRefundRequest(newEntryCode,
                selectedEntry.getRequestId(),
                selectedEntry.getRequestToken(), currency, amount,
                selectedEntry.getPaymentTransaction().getPaymentProvider());

        try {
            final PayPalRefundResult refundResult = (PayPalRefundResult) this.getCardPaymentService()
                    .refundFollowOn(followOnRefundRequest);
            return createRefundTransaction(refundResult, newEntryCode, transactionType, transaction,
                    selectedEntry.getRequestId());
        } catch (Exception e) {
            String message = "Exception, message : " + e.getMessage();
            LOG.error(message, e);
            return createFailedRefundTransaction(selectedEntry, amount, transactionType, newEntryCode, e);
        }
    }

    private PaymentTransactionType getTransactionRefundType(PaymentTransactionEntryModel selectedEntry, BigDecimal amount) {
        final BigDecimal availableAmount = selectedEntry.getAmount().subtract(paymentTransactionsService.calculateRefundedAmount(selectedEntry));
        return (availableAmount.compareTo(amount) > 0)
                ? PaymentTransactionType.REFUND_PARTIAL
                : PaymentTransactionType.REFUND_STANDALONE;
    }

    @Override
    public boolean savePayPalOrder(PayPalCreditCardPaymentInfoModel paymentInfoModel, CartModel cart) {
        boolean isSaved = false;
        try {
            final DefaultPayPalSaveOrderCommand command = getCommandFactory()
                    .createCommand(DefaultPayPalSaveOrderCommand.class);
            final String saveOrderId = command.perform(paymentInfoModel.getPayPalOrderId());
            paymentInfoModel.setPurchaseUnitId(saveOrderId);
            cart.setPaymentStatusType(PaymentStatusType.O_ORDER_CREATED);
            getModelService().saveAll(paymentInfoModel, cart);
            isSaved = true;
        } catch (Exception ex) {
            LOG.error(ORDER_NOT_SAVED, ex);
        }
        return isSaved;
    }


    public PaymentTransactionEntryModel authorizePayment(String merchantTransactionCode, BigDecimal amount, Currency currency,
                                                         CartModel cartModel, String paymentProvider) {
        PaymentTransactionModel transaction = this.getModelService().create(PaymentTransactionModel.class);
        transaction.setCode(merchantTransactionCode);
        transaction.setPlannedAmount(amount);

        String newEntryCode = this.getNewPaymentTransactionEntryCode(transaction, PaymentTransactionType.AUTHORIZATION);

        PayPalAuthorizationResult result = getAuthorizationResult(cartModel, newEntryCode, currency, amount, paymentProvider);
        transaction.setRequestId(result.getRequestId());
        transaction.setRequestToken(result.getRequestToken());
        transaction.setPaymentProvider(result.getPaymentProvider());

        PaymentTransactionEntryModel entry = createPaymentTransactionEntry(result, cartModel, transaction, newEntryCode);
        PayPalCommandsUtil.saveTransactionEntryToTransaction(getModelService(), transaction, entry);
        this.getModelService().refresh(transaction);
        return entry;
    }

    private PaymentTransactionEntryModel createPaymentTransactionEntry(PayPalAuthorizationResult result, CartModel cartModel,
                                                                       PaymentTransactionModel transaction, String newEntryCode) {
        PayPalCreditCardPaymentInfoModel paymentInfo = (PayPalCreditCardPaymentInfoModel) cartModel.getPaymentInfo();
        PaymentTransactionEntryModel entry = this.getModelService().create(PaymentTransactionEntryModel.class);
        entry.setAmount(result.getTotalAmount());
        if (result.getCurrency() != null) {
            entry.setCurrency(this.getCommonI18NService().getCurrency(result.getCurrency().getCurrencyCode()));
        }
        entry.setType(PaymentTransactionType.AUTHORIZATION);
        entry.setTime(result.getAuthorizationTime() == null ? new Date() : result.getAuthorizationTime());
        entry.setPaymentTransaction(transaction);
        entry.setRequestId(result.getRequestId());
        entry.setRequestToken(result.getRequestToken());
        entry.setTransactionStatus(result.getTransactionStatus().toString());
        entry.setTransactionStatusDetails(result.getTransactionStatusDetails().toString());
        entry.setCode(newEntryCode);
        entry.setDebugId(result.getDebugId());
        entry.setRequest(result.getRequestField().replaceAll("\\s", StringUtils.EMPTY));
        entry.setResponse(result.getResponseField().replaceAll("\\s", StringUtils.EMPTY));
        entry.setExpirationTime(result.getExpirationTime());
        entry.setCreateTime(result.getCreateTime());
        entry.setUpdateTime(result.getUpdateTime());
        if (paymentInfo.getPayPalOrderId() != null) {
            entry.setSubscriptionID(paymentInfo.getPayPalOrderId());
        }
        return entry;
    }

    private PayPalAuthorizationResult getAuthorizationResult(CartModel cartModel, String newEntryCode, Currency currency,
                                                             BigDecimal amount, String paymentProvider) {
        BillingInfo shippingInfo = this.createBillingInfo(cartModel.getDeliveryAddress());
        PayPalCreditCardPaymentInfoModel paymentInfo = (PayPalCreditCardPaymentInfoModel) cartModel.getPaymentInfo();

        if (paymentInfo.getPayPalOrderId() == null) {
            return (PayPalAuthorizationResult) this.getCardPaymentService()
                    .authorize(new AuthorizationRequest(newEntryCode, null, currency, amount, shippingInfo));
        } else {
            PayPalAuthorizationResult authorizationResult = (PayPalAuthorizationResult) this.getCardPaymentService()
                    .authorize(new SubscriptionAuthorizationRequest(newEntryCode, paymentInfo.getPayPalOrderId(), currency,
                            amount, shippingInfo, null, paymentProvider));
            if (authorizationResult.getCustomerId() != null && authorizationResult.getPaymentToken() != null) {
                savePaymentInfo(paymentInfo, authorizationResult.getCustomerId(), authorizationResult.getPaymentToken());
            }
            return authorizationResult;
        }
    }

    @Override
    public PaymentTransactionEntryModel capture(PaymentTransactionModel transaction) {
        PaymentTransactionEntryModel transactionEntry = transaction.getEntries().stream()
                .filter(entry -> PaymentTransactionType.AUTHORIZATION.equals(entry.getType()))
                .findFirst()
                .orElseThrow(() -> new AdapterException(AUTHORIZATION_ENTRY_MISSING_EXCEPTION
                        .formatted(transaction.getCode(), transaction.getOrder().getCode())));
        String newEntryCode = this.getNewPaymentTransactionEntryCode(transaction, PaymentTransactionType.CAPTURE);

        try {
            CaptureResult captureResult = executeCapture(transaction, transactionEntry, newEntryCode);
            PaymentTransactionEntryModel entry = createPaymentTransactionEntry(captureResult, transaction, newEntryCode);

            this.getModelService().attach(entry);
            PayPalCommandsUtil.saveTransactionEntryToTransaction(getModelService(), transaction, entry);
            return entry;
        } catch (PayPalHttpClientErrorException e) {
            PaymentTransactionEntryModel entry = createFailedCapturePaymentTransactionEntry(e, transaction, newEntryCode);
            PayPalCommandsUtil.saveTransactionEntryToTransaction(getModelService(), transaction, entry);
            throw e;
        }
    }

    protected PaymentTransactionEntryModel createFailedCapturePaymentTransactionEntry(PayPalHttpClientErrorException e,
                                                                                      PaymentTransactionModel transaction,
                                                                                      String newEntryCode) {
        return GenericBuilder.of(PaymentTransactionEntryModel::new)
                .with(PaymentTransactionEntryModel::setType, PaymentTransactionType.CAPTURE)
                .with(PaymentTransactionEntryModel::setTransactionStatus, TransactionStatus.ERROR.name())
                .with(PaymentTransactionEntryModel::setTransactionStatusDetails,
                        TransactionStatusDetails.GENERAL_SYSTEM_ERROR.name())
                .with(PaymentTransactionEntryModel::setPaymentTransaction, transaction)
                .with(PaymentTransactionEntryModel::setCode, newEntryCode)
                .with(PaymentTransactionEntryModel::setRequest,
                        PayPalCommandsUtil.getValueAsString(e.getCaptureRequest()).replaceAll("\\s", StringUtils.EMPTY))
                .with(PaymentTransactionEntryModel::setResponse,
                        e.getParentException().getResponseBodyAsString().replaceAll("\\s", StringUtils.EMPTY))
                .with(PaymentTransactionEntryModel::setDebugId, getDebugIdFromException(e))
                .with(PaymentTransactionEntryModel::setFailureReason,
                        FAILURE_REASON_MESSAGE.formatted(e.getParentException().getStatusCode().value(),
                                e.getParentException().getStatusText(), e.getMessage()))
                .build();
    }

    private String getDebugIdFromException(PayPalHttpClientErrorException e) {
        String debugId = Objects.requireNonNull(e.getParentException().getResponseHeaders()).get(PAYPAL_DEBUG_ID).get(0);
        return debugId == null ? StringUtils.EMPTY : debugId;
    }

    private CaptureResult executeCapture(PaymentTransactionModel transaction, PaymentTransactionEntryModel transactionEntry,
                                         String newEntryCode) {

        PayPalCreditCardPaymentInfoModel paymentInfo = (PayPalCreditCardPaymentInfoModel) transaction.getInfo();

        if (PAYPAL_INTENT_CAPTURE.equalsIgnoreCase(paymentInfo.getIntent())) {
            return executeCaptureIntentRequestUsingPayPalAPI(newEntryCode, transaction, transactionEntry, paymentInfo);
        }
        return this.getCardPaymentService().capture(new CaptureRequest(newEntryCode, transaction.getRequestId(),
                null,
                Currency.getInstance(transactionEntry.getCurrency().getIsocode()),
                transactionEntry.getAmount(),
                transaction.getPaymentProvider(),
                transactionEntry.getSubscriptionID()));
    }

    private CurrencyModel getCurrency(CaptureResult captureResult) {
        return captureResult.getCurrency() != null
                ? this.getCommonI18NService().getCurrency(captureResult.getCurrency().getCurrencyCode()) : null;
    }

    protected PaymentTransactionEntryModel createPaymentTransactionEntry(CaptureResult captureResult,
                                                                         PaymentTransactionModel transaction,
                                                                         String newEntryCode) {
        CurrencyModel currency = getCurrency(captureResult);
        PayPalCaptureResult payPalCaptureResult = (PayPalCaptureResult) captureResult;

        return GenericBuilder.of(PaymentTransactionEntryModel::new)
                .with(PaymentTransactionEntryModel::setAmount, captureResult.getTotalAmount())
                .with(PaymentTransactionEntryModel::setCurrency, currency)
                .with(PaymentTransactionEntryModel::setType, PaymentTransactionType.CAPTURE)
                .with(PaymentTransactionEntryModel::setRequestId, captureResult.getRequestId())
                .with(PaymentTransactionEntryModel::setRequestToken, captureResult.getRequestToken())
                .with(PaymentTransactionEntryModel::setTime,
                        captureResult.getRequestTime() == null ? new Date() : captureResult.getRequestTime())
                .with(PaymentTransactionEntryModel::setPaymentTransaction, transaction)
                .with(PaymentTransactionEntryModel::setCreateTime, payPalCaptureResult.getCreateTime())
                .with(PaymentTransactionEntryModel::setUpdateTime, payPalCaptureResult.getUpdateTime())
                .with(PaymentTransactionEntryModel::setDebugId, payPalCaptureResult.getDebugId())
                .with(PaymentTransactionEntryModel::setRequest, payPalCaptureResult.getRequestField().replaceAll("\\s", StringUtils.EMPTY))
                .with(PaymentTransactionEntryModel::setResponse, payPalCaptureResult.getResponseField().replaceAll("\\s", StringUtils.EMPTY))
                .with(PaymentTransactionEntryModel::setTransactionStatus, captureResult.getTransactionStatus().toString())
                .with(PaymentTransactionEntryModel::setTransactionStatusDetails,
                        captureResult.getTransactionStatusDetails().toString())
                .with(PaymentTransactionEntryModel::setCode, newEntryCode)
                .build();
    }

    private CaptureResult executeCaptureIntentRequestUsingPayPalAPI(String newEntryCode, PaymentTransactionModel transaction,
                                                                    PaymentTransactionEntryModel auth, PayPalCreditCardPaymentInfoModel paymentInfo) {
        CaptureResult result = doCaptureIntentRequest(new CaptureRequest(newEntryCode, transaction.getRequestId(),
                null, Currency.getInstance(auth.getCurrency().getIsocode()),
                auth.getAmount(),
                transaction.getPaymentProvider(), auth.getSubscriptionID()));
        transaction.getOrder().setPaymentStatusType(PaymentStatusType.COMPLETED);
        PayPalCaptureResult captureResult = (PayPalCaptureResult) result;
        if (captureResult.getCustomerId() != null && captureResult.getPaymentToken() != null) {
            savePaymentInfo(paymentInfo, captureResult.getCustomerId(), captureResult.getPaymentToken());
        }
        return result;
    }

    private void savePaymentInfo(PayPalCreditCardPaymentInfoModel paymentInfo, String customerId, String paymentToken) {
        CustomerModel currentCustomer = (CustomerModel) userService.getCurrentUser();

        Optional.ofNullable(paymentInfo.getPayerEmail()).ifPresent(email ->
                payPalCustomerAccountService.removeDuplicatePaymentMethod(email));

        paymentInfo.setSubscriptionId(paymentToken);
        paymentInfo.setSaved(true);
        getCardData(paymentToken)
                .ifPresent(card -> {
                    final String[] expirationDate = card.getExpiry().split(EXPIRATION_DATE_SEPARATOR);
                    paymentInfo.setValidToMonth(expirationDate[1]);
                    paymentInfo.setValidToYear(expirationDate[0]);
                    paymentInfo.setCcOwner(card.getName());
                    if (defaultPayPalConfigurationService.isPayPalCreditCardOnAddingValidation()) {
                        paymentInfoService.updateExpirationStatus(paymentInfo);
                    }
                });
        boolean isPMProviderPaypal = Optional.ofNullable(paymentInfo.getPaymentProvider())
                .filter(paymentProvider -> paymentProvider.equals(PaymentProvider.PAYPAL))
                .isPresent();
        if (isPMProviderPaypal) {
            paymentInfo.setPMCustomerVaultId(customerId);
        }
        getModelService().save(paymentInfo);

        if (Strings.isEmpty(currentCustomer.getVaultCustomerId())) {
            currentCustomer.setVaultCustomerId(customerId);
            getModelService().save(currentCustomer);
        }
    }

    public void captureVaultedPayment(PaymentTransactionModel transaction, String scaFlow,
                                      PaymentProvider payPalPaymentProvider) {
        PaymentTransactionEntryModel auth = transaction.getEntries().stream()
                .filter(tx -> PaymentTransactionType.AUTHORIZATION.equals(tx.getType())).findFirst()
                .orElseThrow(() -> new AdapterException("Could not capture without authorization"));

        AbstractOrderModel cartModel = transaction.getOrder();
        PayPalCreditCardPaymentInfoModel paymentInfo = (PayPalCreditCardPaymentInfoModel) transaction.getInfo();
        PayPalOrderRequestData requestData = createVaultedRequestData(payPalPaymentProvider, scaFlow);

        requestData.setVaultId(paymentInfo.getSubscriptionId());
        requestData.setIntent(paymentInfo.getIntent());
        orderRequestDataConverter.convert(cartModel, requestData);

        try {
            PayPalVaultedPaymentResult result = doVaultedPaymentRequest(requestData);

            cartModel.setPayPalOrderId(result.getOrderId());
            auth.setRequestId(result.getOrderId());
            transaction.setRequestId(result.getOrderId());

            PaymentTransactionEntryModel entry = createTransactionEntryForCaptureVaultedPayment(transaction, auth, result);

            PayPalCommandsUtil.saveTransactionEntryToTransaction(getModelService(), transaction, entry);
            this.getModelService().saveAll(entry, auth, cartModel);
        } catch (PayPalProcessVaultedPaymentException e) {
            PaymentTransactionEntryModel entry = createFailedVaultedTransactionEntryModel(transaction, e);
            PayPalCommandsUtil.saveTransactionEntryToTransaction(getModelService(), transaction, entry);
            throw new PayPalProcessPaymentException(PAYPAL_CHECKOUT_PAYMENT_ERROR_MSG);
        }
    }

	private PayPalOrderRequestData createVaultedRequestData(PaymentProvider payPalPaymentProvider, String scaFlow) {
		if (PaymentProvider.PAYPAL_HOSTED_FIELDS.equals(payPalPaymentProvider)) {
			StoredCredentialData storedCredentialData = selectStoredCredentialStrategy(scaFlow).getStoredCredential();
			return GenericBuilder.of(PayPalHostedFieldsOrderRequestData::new)
					.with(PayPalHostedFieldsOrderRequestData::setStoredCredential, storedCredentialData)
					.build();
		} else {
			return new PayPalOrderRequestData();
		}
	}

    protected PaymentTransactionEntryModel createTransactionEntryForCaptureVaultedPayment(PaymentTransactionModel transaction,
						PaymentTransactionEntryModel authEntry, PayPalVaultedPaymentResult result) {
        PaymentTransactionEntryModel entry = this.getModelService().create(PaymentTransactionEntryModel.class);
        entry.setAmount(authEntry.getAmount());
        entry.setCurrency(authEntry.getCurrency());
        entry.setType(PaymentTransactionType.CAPTURE);
        entry.setRequestId(result.getRequestId());
        entry.setTime(new Date());
        entry.setPaymentTransaction(transaction);
        entry.setTransactionStatus(result.getTransactionStatus().toString());
        entry.setTransactionStatusDetails(result.getTransactionStatusDetails().toString());
        entry.setCode(this.getNewPaymentTransactionEntryCode(transaction, PaymentTransactionType.CAPTURE));
		entry.setDebugId(result.getDebugId());
        entry.setCreateTime(result.getCreateTime());
        entry.setUpdateTime(result.getUpdateTime());
        entry.setRequest(result.getRequest().replaceAll("\\s", StringUtils.EMPTY));
        entry.setResponse(result.getResponse().replaceAll("\\s", StringUtils.EMPTY));
        return entry;
    }

    protected PaymentTransactionEntryModel createFailedVaultedTransactionEntryModel(PaymentTransactionModel transaction,
                                                                                    PayPalProcessVaultedPaymentException e) {
        String newEntryCode = this.getNewPaymentTransactionEntryCode(transaction, PaymentTransactionType.CAPTURE);

        return GenericBuilder.of(PaymentTransactionEntryModel::new)
                .with(PaymentTransactionEntryModel::setType, PaymentTransactionType.CAPTURE)
                .with(PaymentTransactionEntryModel::setTime, new Date())
                .with(PaymentTransactionEntryModel::setPaymentTransaction, transaction)
                .with(PaymentTransactionEntryModel::setTransactionStatusDetails,
                        TransactionStatusDetails.GENERAL_SYSTEM_ERROR.name())
                .with(PaymentTransactionEntryModel::setDebugId, getDebugIdFromPayPalProcessVaultedPaymentException(e))
                .with(PaymentTransactionEntryModel::setCode, newEntryCode)
                .with(PaymentTransactionEntryModel::setRequest,
                        PayPalCommandsUtil.getValueAsString(e.getRequest()).replaceAll("\\s", StringUtils.EMPTY))
                .with(PaymentTransactionEntryModel::setResponse, e.getMessage().replaceAll("\\s", StringUtils.EMPTY))
                .with(PaymentTransactionEntryModel::setFailureReason, FAILURE_REASON_MESSAGE.formatted(
                        e.getParentException().getStatusCode().value(), e.getParentException().getStatusText(), e.getMessage()))
                .build();
    }

    private String getDebugIdFromPayPalProcessVaultedPaymentException(PayPalProcessVaultedPaymentException e) {
        if (e.getParentException() != null && e.getParentException().getResponseHeaders() != null) {
            Optional<List<String>> debugId = Optional.ofNullable(e.getParentException().getResponseHeaders().get(PAYPAL_DEBUG_ID));
            return debugId.isPresent() ? debugId.get().get(0) : StringUtils.EMPTY;
        }
        return StringUtils.EMPTY;
    }

    public PaymentTransactionEntryModel authorizeVaultedPayment(String merchantTransactionCode, BigDecimal amount, Currency currency,
                                                                CartModel cartModel, String paymentProvider, String scaFlow,
                                                                PaymentProvider payPalPaymentProvider) {
        PaymentTransactionModel transaction = this.getModelService().create(PaymentTransactionModel.class);
        transaction.setCode(merchantTransactionCode);
        transaction.setPlannedAmount(amount);
        PayPalCreditCardPaymentInfoModel paymentInfo = (PayPalCreditCardPaymentInfoModel) cartModel.getPaymentInfo();

        PaymentTransactionType paymentTransactionType = PaymentTransactionType.AUTHORIZATION;
        String newEntryCode = this.getNewPaymentTransactionEntryCode(transaction, paymentTransactionType);

        PayPalOrderRequestData requestData = createVaultedRequestData(payPalPaymentProvider, scaFlow);
        requestData.setVaultId(paymentInfo.getSubscriptionId());
        requestData.setIntent(paymentInfo.getIntent());
        orderRequestDataConverter.convert(cartModel, requestData);

        try {
            PayPalVaultedPaymentResult result = doVaultedPaymentRequest(requestData);

            cartModel.setPayPalOrderId(result.getOrderId());
            transaction.setRequestId(result.getRequestId());
            transaction.setPaymentProvider(paymentProvider);
            this.getModelService().save(transaction);
            PaymentTransactionEntryModel entry = this.getModelService().create(PaymentTransactionEntryModel.class);
            entry.setAmount(amount);
            if (currency != null) {
                entry.setCurrency(this.getCommonI18NService().getCurrency(currency.getCurrencyCode()));
            }
            entry.setType(paymentTransactionType);
            entry.setTime(new Date());
            entry.setPaymentTransaction(transaction);
            entry.setRequestId(result.getRequestId());
            entry.setTransactionStatus(result.getTransactionStatus().toString());
            entry.setTransactionStatusDetails(result.getTransactionStatusDetails().toString());
            entry.setCode(newEntryCode);
            entry.setSubscriptionID(result.getOrderId());
            entry.setUpdateTime(result.getUpdateTime());
            entry.setCreateTime(result.getCreateTime());
			entry.setDebugId(result.getDebugId());
			entry.setExpirationTime(result.getExpirationTime());
            entry.setRequest(result.getRequest().replaceAll("\\s", StringUtils.EMPTY));
            entry.setResponse(result.getResponse().replaceAll("\\s", StringUtils.EMPTY));

            this.getModelService().saveAll(entry, cartModel);
            this.getModelService().refresh(transaction);
            return entry;
        } catch (PayPalProcessVaultedPaymentException e) {
            PaymentTransactionEntryModel entry = createFailedVaultedTransactionEntryModel(transaction, e);
            PayPalCommandsUtil.saveTransactionEntryToTransaction(getModelService(), transaction, entry);
            throw new PayPalProcessPaymentException(PAYPAL_CHECKOUT_PAYMENT_ERROR_MSG);
        }
    }

    private StoredCredentialStrategy selectStoredCredentialStrategy(String flow) {
        return storedCredentialStrategies.stream()
                .filter(strategy -> strategy.test(flow))
                .findAny()
                .orElse(emptyStoredCredentialStrategy);
    }

    @Override
    public PaymentTransactionEntryModel createStubAuthorize(final String merchantTransactionCode,
                                                            final BigDecimal amount, final Currency currency, final String paymentProvider,
                                                            PayPalCreditCardPaymentInfoModel paymentInfo) {
        final PaymentTransactionModel transaction = this.getModelService().create(PaymentTransactionModel.class);
        transaction.setCode(merchantTransactionCode);
        transaction.setPlannedAmount(amount);
        transaction.setCurrency(this.getCommonI18NService().getCurrency(currency.getCurrencyCode()));
        transaction.setPaymentProvider(paymentProvider);
        transaction.setRequestId(paymentInfo.getPayPalOrderId());
        return stubAuthorize(transaction);
    }

    @Override
    public AuthorizationResult reauthorize(SubscriptionAuthorizationRequest request) {
        try {
            CommandFactory commandFactory;
            commandFactory = this.commandFactoryRegistry.getFactory(request.getPaymentProvider());
            DefaultPayPalReauthorizationRequestCommand command = commandFactory
                    .createCommand(DefaultPayPalReauthorizationRequestCommand.class);
            PayPalAuthorizationResult result = (PayPalAuthorizationResult) command.perform(request);
            result.setPaymentProvider(commandFactory.getPaymentProvider());
            return result;
        } catch (CommandNotSupportedException var5) {
            throw new AdapterException(var5.getMessage(), var5);
        }
    }

    @Override
    public Optional<PayPalGetCardDetailsResponseData> convertBillingAgreementTokenToVault(PayPalConvertBAToPaymentTokensRequestData convertBAToPaymentTokensRequestData) {
        try {
            final DefaultPayPalConvertBAToPaymentTokensCommand command = getCommandFactory()
                    .createCommand(DefaultPayPalConvertBAToPaymentTokensCommand.class);

            return command.perform(convertBAToPaymentTokensRequestData);
        } catch (final Exception exception) {
            LOG.error("[PayPal Payment Service] Errors during converting billing agreement: " + exception.getMessage(),
                    exception);
            throw new AdapterException(exception.getMessage());
        }
    }

    @Override
    public PayPalOrderDetailsData getOrderDetails(final String payPalOrderId) {
        try {
            LOG.debug("GetOrderDetails PayPal orderID: " + payPalOrderId);
            final DefaultPayPalGetOrderDetailsCommand command = getCommandFactory()
                    .createCommand(DefaultPayPalGetOrderDetailsCommand.class);

            return command.perform(payPalOrderId);
        } catch (final Exception exception) {
            LOG.error("[PayPal Payment Service] Errors during getting order details: " + exception.getMessage(),
                    exception);
            throw new AdapterException(exception.getMessage());
        }
    }

    public Optional<CardData> getCardData(final String paymentToken) {
        try {
            LOG.debug("GetCardData payment token: " + paymentToken);
            final DefaultPayPalGetCardDetailsCommand command = getCommandFactory()
                    .createCommand(DefaultPayPalGetCardDetailsCommand.class);

            return command.perform(paymentToken);
        } catch (final Exception exception) {
            LOG.error("[PayPal Payment Service] Errors during getting card data: " + exception.getMessage(),
                    exception);
            throw new AdapterException(exception.getMessage());
        }
    }

    @Override
    public Boolean updateOrderAmountDetails(PayPalOrderRequestData orderRequestData) {
        try {
            LOG.debug(UPDATE_ORDER_DETAILS_MESSAGE.formatted(orderRequestData.getOrderId(),
                    orderRequestData.getAmount(), orderRequestData.getCurrency()));
            final DefaultPayPalUpdateOrderAmountCommand command = getCommandFactory()
                    .createCommand(DefaultPayPalUpdateOrderAmountCommand.class);

            return command.perform(orderRequestData);
        } catch (final Exception exception) {
            LOG.error(UPDATING_ORDER_ERROR + exception.getMessage(),
                    exception);
            throw new AdapterException(exception.getMessage());
        }
    }

    @Override
    public String createOrder(PayPalOrderRequestData orderRequestData) {
        try {
            LOG.debug("CreateOrder with details: orderAmount: " + orderRequestData.getAmount() + ", currency: " + orderRequestData.getCurrency());
            final DefaultPayPalCreateOrderCommand command = getCommandFactory()
                    .createCommand(DefaultPayPalCreateOrderCommand.class);

            return command.perform(orderRequestData);
        } catch (final Exception exception) {
            LOG.error(UPDATING_ORDER_ERROR + exception.getMessage(),
                    exception);
            throw new AdapterException(exception.getMessage());
        }
    }

    @Override
    public CaptureResult doCaptureIntentRequest(CaptureRequest request) {
        LOG.debug("Do Capture intent request :" + request);
        try {
            final DefaultPayPalCaptureIntentCommand command = getCommandFactory()
                    .createCommand(DefaultPayPalCaptureIntentCommand.class);
            return command.perform(request);
        } catch (CommandNotSupportedException exception) {
            LOG.error("[PayPal Payment Service] Errors during capture intent request: " + exception.getMessage(),
                    exception);
            throw new AdapterException(exception.getMessage());
        }
    }

    public PayPalVaultedPaymentResult doVaultedPaymentRequest(PayPalOrderRequestData request) {
        LOG.debug("Do Hosted Fields request :" + request);
        try {
            final DefaultPayPalProcessVaultedPaymentCommand command = getCommandFactory()
                    .createCommand(DefaultPayPalProcessVaultedPaymentCommand.class);
            return (PayPalVaultedPaymentResult) command.perform(request);
        } catch (CommandNotSupportedException exception) {
            LOG.error("[PayPal Payment Service] Errors during processing request: " + exception.getMessage(),
                    exception);
            throw new AdapterException(exception.getMessage());
        }
    }

    /**
     * This method is used to get event by id
     *
     * @param eventId event id
     * @return event
     */
    public Event getEventById(String eventId) {
        try {
            CommandFactory commandFactory = getCommandFactoryRegistry().getFactory(PAYPAL_PROVIDER_NAME);
            DefaultPayPalGetEventCommand command = commandFactory.createCommand(DefaultPayPalGetEventCommand.class);
            return command.perform(eventId);
        } catch (CommandNotSupportedException e) {
            String message = "Exception, message : " + e.getMessage();
            LOG.error(message, e);
            throw new AdapterException();
        }
    }

    /**
     * This method is used to cancel order
     *
     * @param order order
     */
    public void doCancel(OrderModel order) {
        doCancelTransactions(order);
        if (isOrderCanceled(order)) {
            order.setPaymentStatusType(PaymentStatusType.CANCELLED);
            order.setStatus(OrderStatus.CANCELLED);
            getModelService().save(order);
        }
    }

    @Override
    public void doCancelTransactions(OrderModel order) {
        List<PaymentTransactionEntryModel> cancelableTransactions = getCancelableTransactions(order);
        for (PaymentTransactionEntryModel cancelableTransaction : cancelableTransactions) {
            PaymentTransactionEntryModel canceledTransaction = this.cancel(cancelableTransaction);
            if (TransactionStatus.ACCEPTED.name().equalsIgnoreCase(canceledTransaction.getTransactionStatus())) {
                cancelableTransaction.setTransactionStatus(TRANSACTION_STATUS_VOIDED);

                getModelService().save(cancelableTransaction);
            }
        }
    }

    @Override
    public PaymentTransactionEntryModel cancel(PaymentTransactionEntryModel transaction) {
        PaymentTransactionType transactionType = PaymentTransactionType.CANCEL;
        String newEntryCode = this.getNewPaymentTransactionEntryCode(transaction.getPaymentTransaction(), transactionType);
        try {
            PayPalVoidResult result = getPaypalVoidResult(newEntryCode, transaction);
            PaymentTransactionEntryModel entry = getPopulatedTransactionEntryModel(result, transaction);
            this.getModelService().save(entry);
            return entry;
        } catch (PayPalVoidAdapterException e) {
            PaymentTransactionEntryModel entryModel = getPopulatedFailedCancelTransactionEntry(e, transaction);
            PayPalCommandsUtil.saveTransactionEntryToTransaction(getModelService(), transaction.getPaymentTransaction(),
                    entryModel);
            throw new AdapterException(e.getMessage());
        }
    }

    protected PayPalVoidResult getPaypalVoidResult(String newEntryCode, PaymentTransactionEntryModel transaction) {
        Currency currency = transaction.getCurrency() == null ? null : Currency.getInstance(transaction.getCurrency().getIsocode());
        VoidRequest voidRequest = new VoidRequest(newEntryCode, transaction.getRequestId(), transaction.getRequestToken(),
                transaction.getPaymentTransaction().getPaymentProvider(), currency, transaction.getAmount());

        return (PayPalVoidResult) this.getCardPaymentService().voidCreditOrCapture(voidRequest);
    }

    private PaymentTransactionEntryModel getPopulatedTransactionEntryModel(PayPalVoidResult result,
                                                                           PaymentTransactionEntryModel transaction) {
        String newEntryCode = getNewPaymentTransactionEntryCode(transaction.getPaymentTransaction(),
                PaymentTransactionType.CANCEL);
        PaymentTransactionEntryModel entry = this.getModelService().create(PaymentTransactionEntryModel.class);

        if (result.getCurrency() != null) {
            entry.setCurrency(this.getCommonI18NService().getCurrency(result.getCurrency().getCurrencyCode()));
        }
        entry.setType(PaymentTransactionType.CANCEL);
        entry.setTime(result.getRequestTime() == null ? new Date() : result.getRequestTime());
        entry.setPaymentTransaction(transaction.getPaymentTransaction());
        entry.setRequestId(result.getRequestId());
        entry.setAmount(result.getAmount());
        entry.setRequestToken(result.getRequestToken());
        entry.setTransactionStatus(result.getTransactionStatus().toString());
        entry.setTransactionStatusDetails(result.getTransactionStatusDetails().toString());
        entry.setCode(newEntryCode);
        entry.setResponse(getResponse(result).replaceAll("\\s", StringUtils.EMPTY));
        entry.setCreateTime(new Date());
        entry.setUpdateTime(new Date());
        entry.setRequest(result.getRequestField().replaceAll("\\s", StringUtils.EMPTY));
        entry.setDebugId(result.getDebugId());
        return entry;
    }

    private String getResponse(PayPalVoidResult result) {
        return result.getResponseField() == null ? TRANSACTION_STATUS_VOIDED
                : result.getResponseField().replaceAll("\\s", StringUtils.EMPTY);
    }

    protected PaymentTransactionEntryModel getPopulatedFailedCancelTransactionEntry(PayPalVoidAdapterException e,
                                                                                    PaymentTransactionEntryModel entry) {
        String newEntryCode = getNewPaymentTransactionEntryCode(entry.getPaymentTransaction(),
                PaymentTransactionType.CANCEL);
        PaymentTransactionEntryModel failedEntry = new PaymentTransactionEntryModel();

        failedEntry.setDebugId(e.getDebugId());
        failedEntry.setCreateTime(e.getCreateDate());
        failedEntry.setResponse(e.getMessage().replaceAll("\\s", StringUtils.EMPTY));
        failedEntry.setType(PaymentTransactionType.CANCEL);
        failedEntry.setTime(new Date());
        failedEntry.setPaymentTransaction(entry.getPaymentTransaction());
        failedEntry.setAmount(entry.getAmount());
        failedEntry.setRequest(PayPalCommandsUtil.getValueAsString(e.getRequest()).replaceAll("\\s", StringUtils.EMPTY));
        failedEntry.setTransactionStatus(TransactionStatus.ERROR.name());
        failedEntry.setTransactionStatusDetails(TransactionStatusDetails.GENERAL_SYSTEM_ERROR.name());
        failedEntry.setCode(newEntryCode);
        failedEntry.setFailureReason(e.getParentException().getMessage());
        return failedEntry;
    }

    /**
     * This method is used to get CancelableTransactions
     *
     * @param order order
     * @return list of PaymentTransactionEntryModel
     */
    public List<PaymentTransactionEntryModel> getCancelableTransactions(OrderModel order) {
        return order.getPaymentTransactions().stream().flatMap(transaction -> transaction.getEntries().stream())
                .filter(authorizationEntries()).toList();

    }

    /**
     * This method is used to check if order is canceled
     *
     * @param order order
     * @return checking result
     */
    public boolean isOrderCanceled(OrderModel order) {
        getModelService().refresh(order);

        return order.getPaymentTransactions().stream().flatMap(transaction -> transaction.getEntries().stream())
                .noneMatch(authorizationEntries());
    }

    private boolean isAvailableRefund(final PaymentTransactionEntryModel selectedEntry) {
        return (PaymentTransactionType.CAPTURE.equals(selectedEntry.getType())
                || PaymentTransactionType.PARTIAL_CAPTURE.equals(selectedEntry.getType()))
                && TransactionStatus.ACCEPTED.name().equals(selectedEntry.getTransactionStatus());
    }

    private PaymentTransactionEntryModel createRefundTransaction(final PayPalRefundResult refundResult, final String entryCode,
                                                                 final PaymentTransactionType transactionType, final PaymentTransactionModel paymentTransaction,
                                                                 final String captureRequestId) {
        final CurrencyModel refundCurrency = refundResult.getCurrency() != null
                ? this.getCommonI18NService().getCurrency(refundResult.getCurrency().getCurrencyCode()) : null;

        return GenericBuilder.of(PaymentTransactionEntryModel::new)
                .with(PaymentTransactionEntryModel::setCurrency, refundCurrency)
                .with(PaymentTransactionEntryModel::setType, transactionType)
                .with(PaymentTransactionEntryModel::setTime,
                        refundResult.getRequestTime() == null ? new Date() : refundResult.getRequestTime())
                .with(PaymentTransactionEntryModel::setPaymentTransaction, paymentTransaction)
                .with(PaymentTransactionEntryModel::setAmount, refundResult.getTotalAmount())
                .with(PaymentTransactionEntryModel::setRequestId, refundResult.getRequestId())
                .with(PaymentTransactionEntryModel::setRequestToken, refundResult.getRequestToken())
                .with(PaymentTransactionEntryModel::setSubscriptionID, captureRequestId)
                .with(PaymentTransactionEntryModel::setTransactionStatus, refundResult.getTransactionStatus().toString())
                .with(PaymentTransactionEntryModel::setTransactionStatusDetails,
                        refundResult.getTransactionStatusDetails().toString())
                .with(PaymentTransactionEntryModel::setCode, entryCode)
                .with(PaymentTransactionEntryModel::setDebugId, refundResult.getDebugId())
                .with(PaymentTransactionEntryModel::setRequest, refundResult.getRequestField().replaceAll("\\s", StringUtils.EMPTY))
                .with(PaymentTransactionEntryModel::setResponse, refundResult.getResponseField().replaceAll("\\s", StringUtils.EMPTY))
                .with(PaymentTransactionEntryModel::setCreateTime, refundResult.getCreateTime())
                .with(PaymentTransactionEntryModel::setUpdateTime, refundResult.getUpdateTime())
                .build();
    }

    protected PaymentTransactionEntryModel createFailedRefundTransaction(final PaymentTransactionEntryModel selectedEntry,
                                                                         final BigDecimal amount,
                                                                         final PaymentTransactionType transactionType,
                                                                         final String entryCode, final Exception e) {
        PaymentTransactionEntryModel entry = GenericBuilder.of(PaymentTransactionEntryModel::new)
                .with(PaymentTransactionEntryModel::setCurrency, selectedEntry.getCurrency())
                .with(PaymentTransactionEntryModel::setType, transactionType)
                .with(PaymentTransactionEntryModel::setTime, new Date())
                .with(PaymentTransactionEntryModel::setPaymentTransaction, selectedEntry.getPaymentTransaction())
                .with(PaymentTransactionEntryModel::setAmount, amount)
                .with(PaymentTransactionEntryModel::setRequestId, selectedEntry.getRequestId())
                .with(PaymentTransactionEntryModel::setRequestToken, selectedEntry.getRequestToken())
                .with(PaymentTransactionEntryModel::setSubscriptionID, selectedEntry.getRequestId())
                .with(PaymentTransactionEntryModel::setTransactionStatus, TransactionStatus.REJECTED.name())
                .with(PaymentTransactionEntryModel::setTransactionStatusDetails, TransactionStatusDetails.GENERAL_SYSTEM_ERROR.name())
                .with(PaymentTransactionEntryModel::setCode, entryCode)
                .build();

        if (e instanceof PayPalRefundAdapterException exception) {
            entry.setRequest(PayPalCommandsUtil.getValueAsString(exception.getRequest()).replaceAll("\\s", StringUtils.EMPTY));
            entry.setFailureReason(exception.getParentException().getMessage());
            entry.setDebugId(exception.getDebugId());
            entry.setResponse(exception.getMessage().replaceAll("\\s", StringUtils.EMPTY));
            entry.setCreateTime(exception.getCreateDate());
        }
        return entry;
    }

    private PaymentTransactionEntryModel stubAuthorize(final PaymentTransactionModel transaction) {
        final PaymentTransactionType paymentTransactionType = PaymentTransactionType.AUTHORIZATION;
        final String newEntryCode = this.getNewPaymentTransactionEntryCode(transaction, paymentTransactionType);
        this.getModelService().save(transaction);

        final PaymentTransactionEntryModel entry = this.getModelService().create(PaymentTransactionEntryModel.class);

        entry.setAmount(transaction.getPlannedAmount());
        entry.setCurrency(transaction.getCurrency());
        entry.setType(paymentTransactionType);
        entry.setTime(new Date());
        entry.setPaymentTransaction(transaction);
        entry.setRequestId(transaction.getRequestId());
        entry.setRequestToken(transaction.getRequestToken());
        entry.setTransactionStatus(TransactionStatus.ACCEPTED.toString());
        entry.setTransactionStatusDetails(TransactionStatusDetails.SUCCESFULL.toString());
        entry.setCode(newEntryCode);

        this.getModelService().save(entry);
        this.getModelService().refresh(transaction);
        return entry;
    }

    private Predicate<PaymentTransactionEntryModel> authorizationEntries() {
        return entry -> (PaymentTransactionType.AUTHORIZATION.equals(entry.getType()))
                && TransactionStatus.ACCEPTED.name().equals(entry.getTransactionStatus());
    }

    private CommandFactory getCommandFactory() {
        return commandFactoryRegistry.getFactory(PAYPAL_PROVIDER_NAME);
    }

    public CommandFactoryRegistry getCommandFactoryRegistry() {
        return this.commandFactoryRegistry;
    }

    public void setCommandFactoryRegistry(CommandFactoryRegistry commandFactoryRegistry) {
        this.commandFactoryRegistry = commandFactoryRegistry;
    }

    public void setPaymentInfoService(DefaultPayPalPaymentInfoService paymentInfoService) {
        this.paymentInfoService = paymentInfoService;
    }

    public void setDefaultPayPalConfigurationService(final PayPalConfigurationService defaultPayPalConfigurationService) {
        this.defaultPayPalConfigurationService = defaultPayPalConfigurationService;
    }

    public void setUserService(UserService userService) {
        this.userService = userService;
    }

    public void setEmptyStoredCredentialStrategy(StoredCredentialStrategy emptyStoredCredentialStrategy) {
        this.emptyStoredCredentialStrategy = emptyStoredCredentialStrategy;
    }

    public void setStoredCredentialStrategies(List<StoredCredentialStrategy> storedCredentialStrategies) {
        this.storedCredentialStrategies = storedCredentialStrategies;
    }

    public void setPayPalCustomerAccountService(
            PayPalCustomerAccountService payPalCustomerAccountService) {
        this.payPalCustomerAccountService = payPalCustomerAccountService;
    }

    public void setPaymentTransactionsService(DefaultPaymentTransactionsService paymentTransactionsService) {
        this.paymentTransactionsService = paymentTransactionsService;
    }

    public void setOrderRequestDataConverter(Converter<AbstractOrderModel, PayPalOrderRequestData> orderRequestDataConverter) {
        this.orderRequestDataConverter = orderRequestDataConverter;
    }

}
