/**
 *
 */
package com.paypal.hybris.core.service.impl;

import com.paypal.hybris.core.commands.impl.DefaultPayPalDeletePaymentTokenCommand;
import com.paypal.hybris.core.commands.impl.DefaultPaypalCreatePaymentTokenCommand;
import com.paypal.hybris.core.commands.impl.DefaultPaypalCreateSetupTokenCommand;
import com.paypal.hybris.core.dao.PayPalPaymentInfoDao;
import com.paypal.hybris.core.enums.PaymentProvider;
import com.paypal.hybris.core.enums.SavedPaymentMethodsLimit;
import com.paypal.hybris.core.model.PayPalCreditCardPaymentInfoModel;
import com.paypal.hybris.core.service.PayPalConfigurationService;
import com.paypal.hybris.core.service.PayPalPaymentInfoService;
import com.paypal.hybris.core.enums.ExpirationStatus;
import com.paypal.hybris.core.util.ExpiredDateValidatorUtil;
import com.paypal.hybris.data.PayPalSavePaymentForPurchaseLaterRequest;
import com.paypal.hybris.data.PayPalSetupTokenResponse;
import de.hybris.platform.core.model.order.payment.PaymentInfoModel;
import de.hybris.platform.core.model.user.CustomerModel;
import de.hybris.platform.payment.commands.factory.CommandFactoryRegistry;
import de.hybris.platform.payment.commands.factory.CommandNotSupportedException;
import de.hybris.platform.servicelayer.model.ModelService;
import de.hybris.platform.servicelayer.user.UserService;
import org.apache.log4j.Logger;

import java.util.Map;
import java.util.Objects;
import java.util.Optional;


import static com.paypal.hybris.core.constants.PaypalcoreConstants.PAYPAL_PROVIDER_NAME;

/**
 * This class is a default implementation of PayPalPaymentInfoService interface
 */
public class DefaultPayPalPaymentInfoService implements PayPalPaymentInfoService {

    private static final Logger LOG = Logger.getLogger(DefaultPayPalPaymentInfoService.class);

    private PayPalPaymentInfoDao payPalPaymentInfoDao;
    private ModelService modelService;
    private PayPalConfigurationService defaultPayPalConfigurationService;
    private UserService userService;
    private Map<SavedPaymentMethodsLimit, Integer> savedPaymentMethodsLimitIntegerMap;
    private CommandFactoryRegistry commandFactoryRegistry;
    private ExpiredDateValidatorUtil expiredDateValidatorUtil;

    @Override
    public PayPalSetupTokenResponse getPaypalSetupToken(PayPalSavePaymentForPurchaseLaterRequest payPalSavePaymentRequest) {
        try {
            DefaultPaypalCreateSetupTokenCommand command = commandFactoryRegistry.getFactory(PAYPAL_PROVIDER_NAME)
                    .createCommand(DefaultPaypalCreateSetupTokenCommand.class);
            return command.perform(payPalSavePaymentRequest);
        } catch (CommandNotSupportedException e) {
            LOG.error(e.getMessage());
            throw new IllegalArgumentException(e);
        }
    }

    @Override
    public Optional<PayPalCreditCardPaymentInfoModel> getCreditCardBySubscriptionId(String subscriptionId) {
        return getPayPalPaymentInfoDao().findCardBySubscriptionId(subscriptionId);
    }

    @Override
    public Optional<PayPalCreditCardPaymentInfoModel> getPaymentInfoByOrderId(String orderId) {
        return getPayPalPaymentInfoDao().findPaymentInfoByPayPalOrderId(orderId);
    }

    @Override
    public Optional<PayPalCreditCardPaymentInfoModel> getPaymentInfoByPK(String pk) {
        return getPayPalPaymentInfoDao().findPaymentInfoByPK(pk);
    }

    @Override
    public void removePaymentInfoByPK(String pk) {
        getPayPalPaymentInfoDao().removePaymentInfoByPK(pk);
    }

    @Override
    public boolean isSavedPayPalAccountsAmountLimited() {
        SavedPaymentMethodsLimit savedPaymentMethodsLimit = defaultPayPalConfigurationService.getSavedPayPalAccountsLimit();
        return isSavedAmountLimited(savedPaymentMethodsLimit, PaymentProvider.PAYPAL);
    }

    @Override
    public boolean isSavedCreditCardsAmountLimited() {
        SavedPaymentMethodsLimit savedPaymentMethodsLimit = defaultPayPalConfigurationService.getSavedCreditCardsLimit();
        return isSavedAmountLimited(savedPaymentMethodsLimit, PaymentProvider.PAYPAL_HOSTED_FIELDS);
    }

    @Override
    public boolean isChangePaymentButtonActive() {
        CustomerModel customerModel = (CustomerModel) userService.getCurrentUser();
        String customerVaultId = customerModel.getVaultCustomerId();
        return defaultPayPalConfigurationService.isChangePaymentButtonEnabled() && customerModel.getPaymentInfos().stream()
                .filter(PayPalCreditCardPaymentInfoModel.class::isInstance)
                .filter(PaymentInfoModel::isSaved)
                .filter(paymentInfo -> !paymentInfo.getDuplicate())
                .map(PayPalCreditCardPaymentInfoModel.class::cast)
                .filter(paymentInfoModel -> paymentInfoModel.getPaymentProvider().equals(PaymentProvider.PAYPAL))
                .map(PayPalCreditCardPaymentInfoModel::getPMCustomerVaultId)
                .filter(Objects::nonNull)
                .anyMatch(pmCustomerVaultId -> pmCustomerVaultId.equals(customerVaultId));
    }

    private boolean isSavedAmountLimited(SavedPaymentMethodsLimit savedPaymentMethodsLimit, PaymentProvider paymentProvider) {
        boolean isSavedAmountLimited = false;
        if (!SavedPaymentMethodsLimit.UNLIMITED.equals(savedPaymentMethodsLimit)) {
            CustomerModel customerModel = (CustomerModel) userService.getCurrentUser();
            long payPalAccountsCount = customerModel.getPaymentInfos().stream()
                    .filter(PayPalCreditCardPaymentInfoModel.class::isInstance)
                    .filter(PaymentInfoModel::isSaved)
                    .filter(paymentInfoModel -> !paymentInfoModel.getDuplicate())
                    .map(PayPalCreditCardPaymentInfoModel.class::cast)
                    .filter(paymentInfoModel -> paymentInfoModel.getPaymentProvider().equals(paymentProvider))
                    .count();
            if (savedPaymentMethodsLimitIntegerMap.get(savedPaymentMethodsLimit) <= payPalAccountsCount) {
                isSavedAmountLimited = true;
            }
        }
        return isSavedAmountLimited;
    }

    @Override
    public PayPalSetupTokenResponse getPaypalPaymentToken(PayPalSavePaymentForPurchaseLaterRequest payPalSavePaymentRequest) {
        try {
            DefaultPaypalCreatePaymentTokenCommand command = commandFactoryRegistry.getFactory(PAYPAL_PROVIDER_NAME)
                    .createCommand(DefaultPaypalCreatePaymentTokenCommand.class);
            return command.perform(payPalSavePaymentRequest);
        } catch (CommandNotSupportedException e) {
            LOG.error(e.getMessage());
            throw new IllegalArgumentException(e);
        }
    }
    @Override
    public boolean deletePaypalPaymentToken(String subscriptionId) {
        try {
            DefaultPayPalDeletePaymentTokenCommand command = commandFactoryRegistry.getFactory(PAYPAL_PROVIDER_NAME)
                    .createCommand(DefaultPayPalDeletePaymentTokenCommand.class);
            return command.perform(subscriptionId);
        } catch (CommandNotSupportedException e) {
            LOG.error(e.getMessage());
            throw new IllegalArgumentException(e);
        }
    }

    @Override
    public void updateExpirationStatus(PayPalCreditCardPaymentInfoModel paymentInfoModel) {
        final int expireBetween = defaultPayPalConfigurationService.getPayPalExpireBetween();
        final long monthsToExpiration = expiredDateValidatorUtil.getMonthsToExpiration(paymentInfoModel.getValidToMonth(),
                paymentInfoModel.getValidToYear());

        if (monthsToExpiration < 0) {
            paymentInfoModel.setExpirationStatus(ExpirationStatus.EXPIRED);
        } else if (monthsToExpiration < expireBetween) {
            paymentInfoModel.setExpirationStatus(ExpirationStatus.EXPIRE_SOON);
            CustomerModel customer = (CustomerModel) paymentInfoModel.getUser();
            customer.setHasNewExpireSoonCard(true);
            modelService.save(customer);
        } else {
            paymentInfoModel.setExpirationStatus(ExpirationStatus.NOT_EXPIRED);
        }
    }

    @Override
    public boolean isPayPalSaveOrderFlowActive(PaymentInfoModel paymentInfo) {
        return paymentInfo instanceof PayPalCreditCardPaymentInfoModel payPalPaymentInfo
                && payPalPaymentInfo.isSaveOrderFlowActive();
    }

    public PayPalPaymentInfoDao getPayPalPaymentInfoDao() {
        return payPalPaymentInfoDao;
    }

    public void setPayPalPaymentInfoDao(final PayPalPaymentInfoDao payPalPaymentInfoDao) {
        this.payPalPaymentInfoDao = payPalPaymentInfoDao;
    }

    public void setModelService(final ModelService modelService) {
        this.modelService = modelService;
    }

    public void setDefaultPayPalConfigurationService(final PayPalConfigurationService defaultPayPalConfigurationService) {
        this.defaultPayPalConfigurationService = defaultPayPalConfigurationService;
    }

    public void setCommandFactoryRegistry(CommandFactoryRegistry commandFactoryRegistry) {
        this.commandFactoryRegistry = commandFactoryRegistry;
    }

    public void setExpiredDateValidatorUtil(ExpiredDateValidatorUtil expiredDateValidatorUtil) {
        this.expiredDateValidatorUtil = expiredDateValidatorUtil;
    }

    public void setUserService(UserService userService) {
        this.userService = userService;
    }

    public void setSavedPaymentMethodsLimitIntegerMap(Map<SavedPaymentMethodsLimit, Integer> savedPaymentMethodsLimitIntegerMap) {
        this.savedPaymentMethodsLimitIntegerMap = savedPaymentMethodsLimitIntegerMap;
    }

}
