package com.paypal.hybris.core.service.impl;

import com.paypal.hybris.core.enums.PaymentStatusType;
import com.paypal.hybris.core.results.PayPalCaptureResult;
import com.paypal.hybris.core.service.PayPalManualMultiCaptureService;
import com.paypal.hybris.core.util.builder.GenericBuilder;
import de.hybris.platform.core.enums.OrderStatus;
import de.hybris.platform.core.model.c2l.CurrencyModel;
import de.hybris.platform.core.model.order.AbstractOrderModel;
import de.hybris.platform.payment.AdapterException;
import de.hybris.platform.payment.commands.request.CaptureRequest;
import de.hybris.platform.payment.enums.PaymentTransactionType;
import de.hybris.platform.payment.methods.CardPaymentService;
import de.hybris.platform.payment.model.PaymentTransactionEntryModel;
import de.hybris.platform.payment.model.PaymentTransactionModel;
import de.hybris.platform.servicelayer.i18n.I18NService;
import de.hybris.platform.servicelayer.model.ModelService;

import java.math.BigDecimal;
import java.util.Currency;
import java.util.Date;

/**
 * This class is a default implementation of the PayPalManualMultiCaptureService interface
 */
public class DefaultPayPalManualMultiCaptureService implements PayPalManualMultiCaptureService {

    private ModelService modelService;
    private I18NService i18nService;
    private CardPaymentService cardPaymentService;

    /**
     * This method is used to do payment transaction multi capture
     * @param amount amount
     * @param selectedEntry selectedEntry
     * @param transactionType transactionType
     * @return PaymentTransactionEntryModel
     * @throws AdapterException if smth get wrong
     */
    public PaymentTransactionEntryModel doMultiCapture(BigDecimal amount, PaymentTransactionEntryModel selectedEntry,
        PaymentTransactionType transactionType) throws AdapterException {
        CaptureRequest request = prepareCaptureRequest(amount, selectedEntry, transactionType, selectedEntry.getSubscriptionID());
        PayPalCaptureResult captureResult = (PayPalCaptureResult) cardPaymentService.capture(request);
        return createCapturePaymentTransactionEntry(captureResult, selectedEntry, transactionType);
    }

    private CaptureRequest prepareCaptureRequest(BigDecimal amount, PaymentTransactionEntryModel selectedEntry,
        PaymentTransactionType transactionType, String subscriptionId) {
        final String merchantTransactionCode = getNewPaymentTransactionEntryCode(selectedEntry.getPaymentTransaction(),
            transactionType);
        final String requestId = selectedEntry.getRequestId();
        final String requestToken = selectedEntry.getRequestToken();
        final Currency currency = Currency.getInstance(selectedEntry.getCurrency().getIsocode());
        final String paymentProvider = selectedEntry.getPaymentTransaction().getPaymentProvider();

         return new CaptureRequest(merchantTransactionCode, requestId, requestToken,
            currency, amount, paymentProvider, subscriptionId);
    }

    private PaymentTransactionEntryModel createCapturePaymentTransactionEntry(PayPalCaptureResult captureResult,
        PaymentTransactionEntryModel selectedEntry, PaymentTransactionType transactionType) {
        final PaymentTransactionModel paymentTransaction = selectedEntry.getPaymentTransaction();
        final AbstractOrderModel orderModel = paymentTransaction.getOrder();
        final String newEntryCode = getNewPaymentTransactionEntryCode(selectedEntry.getPaymentTransaction(),
            transactionType);
        final CurrencyModel currency = captureResult.getCurrency() != null ? getI18nService()
            .getCurrency(captureResult.getCurrency().getCurrencyCode()) : null;

        final PaymentTransactionEntryModel entry = GenericBuilder.of(PaymentTransactionEntryModel::new)
                .with(PaymentTransactionEntryModel::setAmount, captureResult.getTotalAmount())
                .with(PaymentTransactionEntryModel::setCurrency, currency)
                .with(PaymentTransactionEntryModel::setType, transactionType)
                .with(PaymentTransactionEntryModel::setTime, captureResult.getRequestTime() == null ? new Date()
                        : captureResult.getRequestTime())
                .with(PaymentTransactionEntryModel::setPaymentTransaction, paymentTransaction)
                .with(PaymentTransactionEntryModel::setRequestId, captureResult.getRequestId())
                .with(PaymentTransactionEntryModel::setRequestToken, captureResult.getRequestToken())
                .with(PaymentTransactionEntryModel::setSubscriptionID, captureResult.getReconciliationId())
                .with(PaymentTransactionEntryModel::setTransactionStatus, captureResult.getTransactionStatus().toString())
                .with(PaymentTransactionEntryModel::setTransactionStatusDetails,
                        captureResult.getTransactionStatusDetails().toString())
                .with(PaymentTransactionEntryModel::setCode, newEntryCode)
                .with(PaymentTransactionEntryModel::setAuthorizationId, selectedEntry.getRequestId())
                .with(PaymentTransactionEntryModel::setDebugId, captureResult.getDebugId())
                .with(PaymentTransactionEntryModel::setCreateTime, captureResult.getCreateTime())
                .with(PaymentTransactionEntryModel::setUpdateTime, captureResult.getUpdateTime())
                .with(PaymentTransactionEntryModel::setRequest, captureResult.getRequestField())
                .with(PaymentTransactionEntryModel::setResponse, captureResult.getResponseField())
                .build();

        setPaymentStatusType(transactionType, orderModel);
        getModelService().attach(entry);
        getModelService().saveAll(orderModel, entry);
        getModelService().refresh(paymentTransaction);
        return entry;
    }

    private void setPaymentStatusType(PaymentTransactionType transactionType, AbstractOrderModel orderModel) {
        final PaymentStatusType paymentStatusType = transactionType == PaymentTransactionType.CAPTURE ?
                PaymentStatusType.COMPLETED : PaymentStatusType.PARTIAL_CAPTURE;
        if (paymentStatusType.getCode().equals(PaymentStatusType.COMPLETED.getCode())) {
            orderModel.setStatus(OrderStatus.COMPLETED);
            orderModel.setPaymentStatusType(PaymentStatusType.COMPLETED);
        } else {
            orderModel.setPaymentStatusType(PaymentStatusType.PARTIAL_CAPTURE);
        }
    }

    private String getNewPaymentTransactionEntryCode(PaymentTransactionModel transaction,
        PaymentTransactionType paymentTransactionType) {
        return transaction.getEntries() == null ?
            transaction.getCode() + "-" + paymentTransactionType.getCode() + "-1" :
            transaction.getCode() + "-" + paymentTransactionType.getCode() + "-" + (transaction.getEntries().size()
                + 1);
    }

    public ModelService getModelService() {
        return modelService;
    }

    public void setModelService(ModelService modelService) {
        this.modelService = modelService;
    }

    public I18NService getI18nService() {
        return i18nService;
    }

    public void setI18nService(I18NService i18nService) {
        this.i18nService = i18nService;
    }

    public void setCardPaymentService(CardPaymentService cardPaymentService) {
        this.cardPaymentService = cardPaymentService;
    }
}
