package com.paypal.hybris.core.service.impl;

import com.google.common.base.Predicate;
import com.paypal.hybris.core.commands.impl.DefaultPayPalAuthorizeSavedOrderCommand;
import com.paypal.hybris.core.enums.PaymentStatusType;
import com.paypal.hybris.core.exception.PayPalAuthorizeAdapterException;
import com.paypal.hybris.core.exception.PayPalProcessHttpClientErrorException;
import com.paypal.hybris.core.exception.PayPalProcessPaymentException;
import com.paypal.hybris.core.model.PayPalCreditCardPaymentInfoModel;
import com.paypal.hybris.core.request.PayPalSavedOrderSubscriptionAuthorizationRequest;
import com.paypal.hybris.core.service.PayPalConfigurationService;
import com.paypal.hybris.core.service.PayPalManualAuthorizationService;
import com.paypal.hybris.core.service.PayPalPaymentService;
import com.paypal.hybris.core.util.PayPalCommandsUtil;
import com.paypal.hybris.core.util.builder.GenericBuilder;
import com.paypal.hybris.data.PayPalOrderRequestData;
import de.hybris.platform.commercefacades.order.data.OrderData;
import de.hybris.platform.core.enums.OrderStatus;
import de.hybris.platform.core.model.c2l.CurrencyModel;
import de.hybris.platform.core.model.order.AbstractOrderModel;
import de.hybris.platform.core.model.order.OrderModel;
import de.hybris.platform.core.model.user.AddressModel;
import de.hybris.platform.payment.AdapterException;
import de.hybris.platform.payment.commands.SubscriptionAuthorizationCommand;
import de.hybris.platform.payment.commands.factory.CommandFactory;
import de.hybris.platform.payment.commands.factory.CommandFactoryRegistry;
import de.hybris.platform.payment.commands.factory.CommandNotSupportedException;
import de.hybris.platform.payment.commands.request.SubscriptionAuthorizationRequest;
import de.hybris.platform.payment.commands.result.AuthorizationResult;
import de.hybris.platform.payment.dto.BillingInfo;
import de.hybris.platform.payment.dto.TransactionStatus;
import de.hybris.platform.payment.dto.TransactionStatusDetails;
import de.hybris.platform.payment.enums.PaymentTransactionType;
import de.hybris.platform.payment.model.PaymentTransactionEntryModel;
import de.hybris.platform.payment.model.PaymentTransactionModel;
import de.hybris.platform.servicelayer.dto.converter.Converter;
import de.hybris.platform.servicelayer.i18n.I18NService;
import de.hybris.platform.servicelayer.model.ModelService;

import java.util.NoSuchElementException;

import org.apache.commons.lang3.StringUtils;
import org.apache.log4j.Logger;

import java.math.BigDecimal;
import java.util.Currency;
import java.util.Date;
import java.util.Objects;
import java.util.UUID;

import static com.paypal.hybris.core.constants.PaypalcoreConstants.PAYPAL_PROVIDER_NAME;

/**
 * This class is a default implementation of the PayPalManualAuthorizationService interface
 */
public class DefaultPayPalManualAuthorizationService implements PayPalManualAuthorizationService {

    private static final String PAYPAL_DEBUG_ID = "Paypal-Debug-Id";

    private static final String FAILURE_REASON_MESSAGE = "Status code: '%s', Status text: '%s', Message from PayPal: '%s'";

    private static final Logger LOG = Logger.getLogger(DefaultPayPalManualAuthorizationService.class);

    private PayPalPaymentService payPalPaymentService;
    private I18NService i18nService;
    private ModelService modelService;
    private CommandFactoryRegistry commandFactoryRegistry;
    private PayPalConfigurationService payPalConfigurationService;
    private Converter<OrderModel, OrderData> orderConverter;
    private Converter<AbstractOrderModel, PayPalOrderRequestData> orderRequestDataConverter;

    /**
     * This method is used to do authorization
     * @param order order
     * @param amount amount
     * @return payment transaction model
     */
    public PaymentTransactionEntryModel doAuthorization(final OrderModel order, final BigDecimal amount) {
        try {
            final PayPalCreditCardPaymentInfoModel paymentInfo = (PayPalCreditCardPaymentInfoModel)order.getPaymentInfo();
            final boolean isSaveOrderFlowActive = paymentInfo.isSaveOrderFlowActive();
            final String intent = order.getSite().getStores().stream()
                    .map(store -> payPalConfigurationService.getPayPalIntent(store))
                    .findFirst().orElseThrow();
            final String payPalOrderId = isSaveOrderFlowActive ? paymentInfo.getPayPalOrderId()
                : createOrderForGivenAmount(order, amount, intent);
            LOG.debug("New order Id --  " + payPalOrderId);
            final SubscriptionAuthorizationRequest request = prepareAuthorizationRequest(order, amount, payPalOrderId);
            final AuthorizationResult result =
                isSaveOrderFlowActive ? executeAuthorizeSavedOrderCommand(request, paymentInfo.getPurchaseUnitId())
                    : executeSubscriptionAuthorizeCommand(request);
            final PaymentTransactionEntryModel paymentTransactionEntry = createPaymentTransactionEntry(result, order,
                payPalOrderId);
            if(!isSaveOrderFlowActive){
                updatePlanetAmount(order);
            }
            getModelService().save(order);
            return paymentTransactionEntry;
        } catch (AdapterException exception) {
            String message = "Exception, message : " + exception.getMessage();
            LOG.error(message, exception);
            return handlePayPalSDKError(order, exception, amount);
        } catch (PayPalProcessPaymentException exception) {
            handlePayPalSDKError(order, exception, amount);
            throw new PayPalProcessPaymentException(exception.getMessage());
        }
    }

    private String createOrderForGivenAmount(final OrderModel orderModel, final BigDecimal amount, final String intent) {
        PayPalOrderRequestData payPalOrderRequestData = orderRequestDataConverter.convert(orderModel);
        payPalOrderRequestData.setSaveOrderFlowActive(false);
        payPalOrderRequestData.setIntent(intent);

        return getPayPalPaymentService().createOrder(payPalOrderRequestData);
    }

    private SubscriptionAuthorizationRequest prepareAuthorizationRequest(final OrderModel order,
        final BigDecimal amount,
        final String payPalOrderId) {
        final String merchantTransactionCode = generateCode(order);
        final Currency currency = getI18nService().getBestMatchingJavaCurrency(order.getCurrency().getIsocode());
        final BillingInfo shippingInfo = createBillingInfo(order.getDeliveryAddress());
        final String cv2 = null;

        return new SubscriptionAuthorizationRequest(merchantTransactionCode,
            payPalOrderId, currency, amount, shippingInfo, cv2, PAYPAL_PROVIDER_NAME);
    }

    private String generateCode(final OrderModel order) {
        return order.getUser().getUid() + "-" + UUID.randomUUID();
    }

    protected BillingInfo createBillingInfo(final AddressModel address) {
        if (address == null) {
            return null;
        } else {
            final String country = address.getCountry() != null ? address.getCountry().getIsocode() : null;
            final String region = address.getRegion() != null ? address.getRegion().getName() : null;

            return GenericBuilder.of(BillingInfo::new)
                .with(BillingInfo::setCity, address.getTown())
                .with(BillingInfo::setCountry, country)
                .with(BillingInfo::setEmail, address.getEmail())
                .with(BillingInfo::setFirstName, address.getFirstname())
                .with(BillingInfo::setLastName, address.getLastname())
                .with(BillingInfo::setPhoneNumber, address.getPhone1())
                .with(BillingInfo::setPostalCode, address.getPostalcode())
                .with(BillingInfo::setRegion, region)
                .with(BillingInfo::setStreet1, address.getStreetname())
                .with(BillingInfo::setStreet2, address.getStreetnumber())
                .build();
        }
    }

    protected AuthorizationResult executeSubscriptionAuthorizeCommand(final SubscriptionAuthorizationRequest request) {

        try {
            SubscriptionAuthorizationCommand command = getCommandFactory()
                .createCommand(SubscriptionAuthorizationCommand.class);

            return command.perform(request);
        } catch (CommandNotSupportedException var5) {
            throw new AdapterException(var5.getMessage(), var5);
        }
    }

    private PaymentTransactionEntryModel createPaymentTransactionEntry(final AuthorizationResult result,
        final OrderModel order, final String payPalOrderId) {
        final boolean isSaveOrderFlowActive = ((PayPalCreditCardPaymentInfoModel) order.getPaymentInfo()).isSaveOrderFlowActive();
        final PaymentTransactionType paymentTransactionType = PaymentTransactionType.AUTHORIZATION;
        order.setPaymentStatusType(PaymentStatusType.PENDING);
        PaymentTransactionModel transaction;
        if(isSaveOrderFlowActive){
            transaction = createPaymentTransactionForSavedOrder(order, result);
            order.setStatus(OrderStatus.PAYMENT_NOT_CAPTURED);
        }else{
            transaction = order.getPaymentTransactions().stream().findFirst()
                .orElseThrow(() -> new NoSuchElementException("No transactions found for this order"));
        }

        final String newEntryCode = getNewPaymentTransactionEntryCode(transaction, paymentTransactionType);
        final CurrencyModel currency = result.getCurrency() != null ? this.getI18nService()
            .getCurrency(result.getCurrency().getCurrencyCode()) : null;

        PaymentTransactionEntryModel entry = GenericBuilder.of(PaymentTransactionEntryModel::new)
            .with(PaymentTransactionEntryModel::setCurrency, currency)
            .with(PaymentTransactionEntryModel::setAmount, result.getTotalAmount())
            .with(PaymentTransactionEntryModel::setType, paymentTransactionType)
            .with(PaymentTransactionEntryModel::setTime, result.getAuthorizationTime() == null ? new Date()
                : result.getAuthorizationTime())
            .with(PaymentTransactionEntryModel::setPaymentTransaction, transaction)
            .with(PaymentTransactionEntryModel::setRequestId, result.getRequestId())
            .with(PaymentTransactionEntryModel::setRequestToken, result.getRequestToken())
            .with(PaymentTransactionEntryModel::setTransactionStatus, result.getTransactionStatus().toString())
            .with(PaymentTransactionEntryModel::setTransactionStatusDetails,
                result.getTransactionStatusDetails().toString())
            .with(PaymentTransactionEntryModel::setCode, newEntryCode)
            .with(PaymentTransactionEntryModel::setSubscriptionID, payPalOrderId)
            .build();
        getModelService().attach(entry);
        getModelService().saveAll(order, entry);
        getModelService().refresh(transaction);
        return entry;
    }

    protected PaymentTransactionEntryModel handlePayPalSDKError(final OrderModel order, Exception e,
        final BigDecimal amount) {
        PaymentTransactionType paymentTransactionType = PaymentTransactionType.AUTHORIZATION;
        PaymentTransactionModel transaction = order.getPaymentTransactions().stream().findFirst()
                .orElse(new PaymentTransactionModel());
        String newEntryCode = getNewPaymentTransactionEntryCode(transaction, paymentTransactionType);
        PaymentTransactionEntryModel entry = GenericBuilder.of(PaymentTransactionEntryModel::new)
            .with(PaymentTransactionEntryModel::setCode, newEntryCode)
            .with(PaymentTransactionEntryModel::setPaymentTransaction, transaction)
            .with(PaymentTransactionEntryModel::setType, paymentTransactionType)
            .with(PaymentTransactionEntryModel::setAmount, amount)
            .with(PaymentTransactionEntryModel::setCurrency, transaction.getCurrency())
            .with(PaymentTransactionEntryModel::setTime, new Date())
            .with(PaymentTransactionEntryModel::setTransactionStatus, TransactionStatus.ERROR.toString())
            .with(PaymentTransactionEntryModel::setTransactionStatusDetails, TransactionStatusDetails.GENERAL_SYSTEM_ERROR.name())
            .build();

        if (e instanceof PayPalProcessHttpClientErrorException exception) {
            entry.setRequest(PayPalCommandsUtil.getValueAsString(exception.getRequest()));
            entry.setResponse(exception.getParentException().getResponseBodyAsString());
            entry.setDebugId(getDebugIdFromException(exception));
            entry.setFailureReason(FAILURE_REASON_MESSAGE.formatted(exception.getParentException().getStatusCode().value(),
                            exception.getParentException().getStatusText(), exception.getParentException().getMessage()));
        }

        if (e instanceof PayPalAuthorizeAdapterException exception) {
            entry.setRequest(PayPalCommandsUtil.getValueAsString(exception.getRequest()));
            entry.setFailureReason(exception.getParentException().getMessage());
        }
        PayPalCommandsUtil.saveTransactionEntryToTransaction(getModelService(), transaction, entry);
        return entry;
    }

    private String getDebugIdFromException(PayPalProcessHttpClientErrorException e) {
        String debugId = Objects.requireNonNull(e.getParentException().getResponseHeaders()).get(PAYPAL_DEBUG_ID).get(0);
        return debugId == null ? StringUtils.EMPTY : debugId;
    }

    /**
     *This method is used to get new PaymentTransaction EntryCode
     * @param transaction transaction
     * @param paymentTransactionType transaction type
     * @return payment transaction entry code
     */
    public String getNewPaymentTransactionEntryCode(PaymentTransactionModel transaction,
        PaymentTransactionType paymentTransactionType) {
        return transaction.getEntries() == null ?
            transaction.getCode() + "-" + paymentTransactionType.getCode() + "-1" :
            transaction.getCode() + "-" + paymentTransactionType.getCode() + "-" + (transaction.getEntries().size()
                + 1);
    }

    private void updatePlanetAmount(final OrderModel order) {
        PaymentTransactionModel transaction = order.getPaymentTransactions().stream().findFirst().get();
        transaction.setPlannedAmount(getAuthorizedAmount(transaction));
        getModelService().save(transaction);
    }

    private BigDecimal getAuthorizedAmount(final PaymentTransactionModel transaction) {
        Predicate<PaymentTransactionEntryModel> authorizationEntries = (entry -> (PaymentTransactionType.AUTHORIZATION
            .equals(entry.getType())) && TransactionStatus.ACCEPTED.name().equals(entry.getTransactionStatus()));

        double authorizedAmount = transaction.getEntries().stream()
            .filter(authorizationEntries).mapToDouble(entry -> entry.getAmount().doubleValue()).sum();
        return BigDecimal.valueOf(authorizedAmount);
    }

    protected AuthorizationResult executeAuthorizeSavedOrderCommand(final SubscriptionAuthorizationRequest request,
        final String purchaseUnitID) {

        PayPalSavedOrderSubscriptionAuthorizationRequest savedOrderAuthRequest = new PayPalSavedOrderSubscriptionAuthorizationRequest(
            request, purchaseUnitID);

        try {
            DefaultPayPalAuthorizeSavedOrderCommand command = getCommandFactory()
                .createCommand(DefaultPayPalAuthorizeSavedOrderCommand.class);

            return command.perform(savedOrderAuthRequest);
        } catch (CommandNotSupportedException exception) {
            throw new AdapterException(exception.getMessage(), exception);
        }
    }

    private PaymentTransactionModel createPaymentTransactionForSavedOrder(final OrderModel orderModel, final AuthorizationResult result){
        final String merchantTransactionCode = generateMerchantTransactionCode(orderModel);
        PaymentTransactionModel transaction = this.getModelService().create(PaymentTransactionModel.class);
        transaction.setCode(merchantTransactionCode);
        transaction.setPlannedAmount(result.getTotalAmount());
        transaction.setRequestId(result.getRequestId());
        transaction.setRequestToken(result.getRequestToken());
        transaction.setPaymentProvider(result.getPaymentProvider());
        transaction.setOrder(orderModel);
        transaction.setInfo(orderModel.getPaymentInfo());
        this.getModelService().save(transaction);
        return transaction;
    }

    private String generateMerchantTransactionCode(OrderModel orderModel) {
        return orderModel.getUser().getUid() + "-" + UUID.randomUUID();
    }
    public PayPalPaymentService getPayPalPaymentService() {
        return payPalPaymentService;
    }

    public void setPayPalPaymentService(PayPalPaymentService payPalPaymentService) {
        this.payPalPaymentService = payPalPaymentService;
    }

    public I18NService getI18nService() {
        return i18nService;
    }

    public void setI18nService(I18NService i18nService) {
        this.i18nService = i18nService;
    }

    public ModelService getModelService() {
        return modelService;
    }

    public void setModelService(ModelService modelService) {
        this.modelService = modelService;
    }

    private CommandFactory getCommandFactory() {
        return commandFactoryRegistry.getFactory(PAYPAL_PROVIDER_NAME);
    }

    public void setCommandFactoryRegistry(CommandFactoryRegistry commandFactoryRegistry) {
        this.commandFactoryRegistry = commandFactoryRegistry;
    }

    public void setPayPalConfigurationService(
            PayPalConfigurationService payPalConfigurationService) {
        this.payPalConfigurationService = payPalConfigurationService;
    }

    public void setOrderConverter(Converter<OrderModel, OrderData> orderConverter) {
        this.orderConverter = orderConverter;
    }
}
