package com.paypal.hybris.core.service.impl;

import com.paypal.enums.PayPalPaymentProvider;
import com.paypal.hybris.addon.forms.CreditCardAddressData;
import com.paypal.hybris.core.dao.PayPalCustomerAccountDao;
import com.paypal.hybris.core.enums.PaymentProvider;
import com.paypal.hybris.core.model.PayPalCreditCardPaymentInfoModel;
import com.paypal.hybris.core.service.PayPalConfigurationService;
import com.paypal.hybris.core.service.PayPalCustomerAccountService;
import com.paypal.hybris.data.PayPalGetCardDetailsResponseData;
import com.paypal.hybris.data.PayPalSetupTokenResponse;
import com.paypal.hybris.data.SetupTokenRequestData;
import de.hybris.platform.commercefacades.order.data.CCPaymentInfoData;
import de.hybris.platform.commercefacades.user.data.CustomerData;
import de.hybris.platform.commerceservices.customer.impl.DefaultCustomerAccountService;
import de.hybris.platform.core.enums.CreditCardType;
import de.hybris.platform.core.model.order.payment.CreditCardPaymentInfoModel;
import de.hybris.platform.core.model.order.payment.PaymentInfoModel;
import de.hybris.platform.core.model.user.AddressModel;
import de.hybris.platform.core.model.user.CustomerModel;
import de.hybris.platform.servicelayer.dto.converter.Converter;
import de.hybris.platform.servicelayer.model.ModelService;
import de.hybris.platform.servicelayer.user.UserService;
import org.apache.commons.lang.BooleanUtils;
import org.apache.commons.lang.StringUtils;

import java.security.SecureRandom;
import java.util.Objects;
import java.util.Optional;
import java.util.UUID;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

import static com.paypal.hybris.core.constants.PaypalcoreConstants.PAYPAL_CHARACTERS_FOR_PASSWORD;
import static com.paypal.hybris.core.constants.PaypalcoreConstants.PAYPAL_EMPTY_STRING;
import static com.paypal.hybris.core.constants.PaypalcoreConstants.PAYPAL_TEMP_PASSWORD_LENGTH;
import static de.hybris.platform.servicelayer.util.ServicesUtil.validateParameterNotNull;

/**
 * This class is a default implementation of the PayPalCustomerAccountService interface
 */
public class DefaultPayPalCustomerAccountService extends DefaultCustomerAccountService implements
        PayPalCustomerAccountService {

    private static final int ZERO = 0;
    private static final String CARD_PLACEHOLDER = "************";

    private PayPalConfigurationService defaultPayPalConfigurationService;
    private PayPalCustomerAccountDao payPalCustomerAccountDao;
    private Converter<CustomerModel, CustomerData> customerConverter;
    private DefaultPayPalPaymentInfoService paymentInfoService;
    private ModelService modelService;
    private UserService userService;

    @Override
    public PayPalCreditCardPaymentInfoModel createPayPalPaymentSubscription(final CustomerModel customerModel,
                                                                            final CCPaymentInfoData ccPaymentInfoData, final AddressModel billingAddress, PayPalCreditCardPaymentInfoModel dumbPaymentInfo) {

        PayPalCreditCardPaymentInfoModel payPalPaymentInfoModel = getPayPalPaymentInfoModel(dumbPaymentInfo);

        populateDumbCCInfo(payPalPaymentInfoModel, customerModel);
        populatePayPalPaymentInfo(payPalPaymentInfoModel, ccPaymentInfoData);
        payPalPaymentInfoModel.setShouldBeSaved(ccPaymentInfoData.isShouldBeSaved());

        payPalPaymentInfoModel.setBillingAddress(billingAddress);
        payPalPaymentInfoModel.setIntent(defaultPayPalConfigurationService.getPayPalIntent());
        if (billingAddress.getOwner() == null) {
            billingAddress.setOwner(payPalPaymentInfoModel);
        }
        if (CreditCardType.PAYPAL_HOSTED_FIELDS_CARD.name().equalsIgnoreCase(ccPaymentInfoData.getCardType())) {
            payPalPaymentInfoModel.setPaymentProvider(PaymentProvider.PAYPAL_HOSTED_FIELDS);
        } else if (CreditCardType.APPLEPAY.name().equalsIgnoreCase(ccPaymentInfoData.getCardType())) {
            payPalPaymentInfoModel.setPaymentProvider(PaymentProvider.APPLEPAY);
        } else if (isPayPal(ccPaymentInfoData) || isVenmo(ccPaymentInfoData)) {
            payPalPaymentInfoModel.setPaymentProvider(PaymentProvider.PAYPAL);
            payPalPaymentInfoModel.setSaveOrderFlowActive(defaultPayPalConfigurationService.isSaveOrderFlow());
        }
        getModelService().saveAll(billingAddress, payPalPaymentInfoModel);
        getModelService().refresh(customerModel);

        addPaymentInfo(customerModel, payPalPaymentInfoModel);
        if (ccPaymentInfoData.isDefaultPaymentInfo()) {
            customerModel.setDefaultPaymentInfo(payPalPaymentInfoModel);
            getModelService().save(customerModel);
        }
        return payPalPaymentInfoModel;
    }

    private PayPalCreditCardPaymentInfoModel getPayPalPaymentInfoModel(PayPalCreditCardPaymentInfoModel dumbPaymentInfo) {
        if (Objects.nonNull(dumbPaymentInfo)) {
            return dumbPaymentInfo;
        } else {
            return getModelService().create(PayPalCreditCardPaymentInfoModel.class);
        }
    }

    @Override
    public PayPalCreditCardPaymentInfoModel updateStubCreditCardPaymentInfo(PayPalCreditCardPaymentInfoModel paymentInfo, PayPalSetupTokenResponse payPalPaymentTokenResponse) {
        paymentInfo.setSubscriptionId(payPalPaymentTokenResponse.getId());
        if (payPalPaymentTokenResponse.getPaymentSource().getCard() != null) {
            paymentInfo.setNumber(CARD_PLACEHOLDER + payPalPaymentTokenResponse.getPaymentSource().getCard().getLastDigits());
            paymentInfo.setType(CreditCardType.valueOf(payPalPaymentTokenResponse.getPaymentSource().getCard().getBrand()));
        }
        paymentInfo.setSaved(true);

        modelService.saveAll(paymentInfo);

        return paymentInfo;
    }

    @Override
    public PayPalCreditCardPaymentInfoModel createStubCreditCardPaymentInfo(SetupTokenRequestData tokenRequestData,
                                                                            CustomerModel customerModel) {
        PayPalCreditCardPaymentInfoModel paymentInfo = getModelService().create(PayPalCreditCardPaymentInfoModel.class);

        boolean shouldBeDefault = customerModel.getPaymentInfos().stream().noneMatch(PaymentInfoModel::isSaved);

        if (PayPalPaymentProvider.PAYPAL_HOSTED_FIELDS.equals(tokenRequestData.getPaymentType())) {
            paymentInfo.setUser(customerModel);
            paymentInfo.setCode(customerModel.getUid() + "_" + UUID.randomUUID());
            paymentInfo.setCcOwner(tokenRequestData.getCreditCardData().getNameOnCard());
            paymentInfo.setValidToYear(tokenRequestData.getCreditCardData().getExpiryYear());
            paymentInfo.setValidToMonth(tokenRequestData.getCreditCardData().getExpiryMonth());
            paymentInfo.setPaymentProvider(PaymentProvider.valueOf(tokenRequestData.getPaymentType().name()));
            paymentInfo.setPayerEmail(customerModel.getUid());
            paymentInfo.setNumber(CARD_PLACEHOLDER);
            paymentInfo.setType(CreditCardType.CARD);
            paymentInfo.setSaved(false);

            AddressModel address = getBillingAddress(tokenRequestData.getCreditCardData(), customerModel, paymentInfo);

            paymentInfo.setBillingAddress(address);
            if (address.getOwner() == null) {
                address.setOwner(paymentInfo);
            }
            updateExpirationStatus(paymentInfo);
            modelService.save(address);
        } else if (PayPalPaymentProvider.PAYPAL.equals(tokenRequestData.getPaymentType())) {
            populateDumbCCInfo(paymentInfo, customerModel);
            paymentInfo.setPaymentProvider(PaymentProvider.valueOf(tokenRequestData.getPaymentType().name()));
            paymentInfo.setPayerEmail(customerModel.getUid());
            paymentInfo.setNumber(CARD_PLACEHOLDER);
            paymentInfo.setType(CreditCardType.PAYPAL);
            paymentInfo.setSaved(false);
            paymentInfo.setCcOwner(StringUtils.EMPTY);
        }
        addPaymentInfo(customerModel, paymentInfo);

        if (shouldBeDefault) {
            customerModel.setDefaultPaymentInfo(paymentInfo);
        }

        modelService.saveAll(paymentInfo, customerModel);

        return paymentInfo;
    }

    @Override
    public void removeDuplicatePaymentMethod(String payerEmail) {
        CustomerModel customerModel = (CustomerModel) userService.getCurrentUser();
        getAlreadySavedAccountsPaymentInfo(payerEmail, customerModel)
                .ifPresent(paymentInfo -> this.deleteCCPaymentInfo(customerModel, paymentInfo));
    }

    private Optional<PayPalCreditCardPaymentInfoModel> getAlreadySavedAccountsPaymentInfo(
            final String payerEmail, final CustomerModel customerModel) {
        return customerModel.getPaymentInfos().stream()
                .filter(PayPalCreditCardPaymentInfoModel.class::isInstance)
                .map(PayPalCreditCardPaymentInfoModel.class::cast)
                .filter(paymentInfo -> !paymentInfo.getDuplicate())
                .filter(paymentInfo -> isPayPalAccountAlreadySaved(payerEmail, paymentInfo))
                .findFirst();
    }

    private static boolean isPayPalAccountAlreadySaved(final String payerEmail,
                                                       final PayPalCreditCardPaymentInfoModel payment) {
        return PaymentProvider.PAYPAL.equals(payment.getPaymentProvider())
                && payment.isSaved()
                && payment.getPayerEmail().equals(payerEmail);
    }

    private AddressModel getBillingAddress(CreditCardAddressData creditCardData,
                                           CustomerModel customerModel, PaymentInfoModel paymentInfoModel) {
        AddressModel address = getAddressForCode(customerModel, creditCardData.getSelectedAddressCode());
        if (BooleanUtils.isTrue(address.getVisibleInAddressBook())) {
            address = modelService.clone(address);
            address.setBillingAddress(Boolean.TRUE);
            address.setVisibleInAddressBook(Boolean.FALSE);
            address.setOwner(paymentInfoModel);
        }
        return address;
    }

    private void updateExpiryDate(PayPalCreditCardPaymentInfoModel paymentInfo, String expiry) {
        Matcher matcher = Pattern.compile("(\\d+)(-)(\\d+)").matcher(expiry);
        if (matcher.find()) {
            paymentInfo.setValidToMonth(matcher.group(3));
            paymentInfo.setValidToYear(matcher.group(1));
        }
    }

    @Override
    public void updateCreditCardInfo(PayPalGetCardDetailsResponseData payPalGetCardDetails,
                                     PayPalCreditCardPaymentInfoModel paymentInfo) {
        CustomerModel customerFromOrder = (CustomerModel) paymentInfo.getUser();
        if (customerFromOrder.getVaultCustomerId() == null) {
            String customerId = payPalGetCardDetails.getCustomer().getId();
            customerFromOrder.setVaultCustomerId(customerId);
        }

        updateExpiryDate(paymentInfo, payPalGetCardDetails.getPaymentSource().getCard().getExpiry());
        updateExpirationStatus(paymentInfo);

        paymentInfo.setSaved(true);
        paymentInfo.setSubscriptionId(payPalGetCardDetails.getId());
        paymentInfo.setCcOwner(payPalGetCardDetails.getPaymentSource().getCard().getName());
        paymentInfo.setNumber(CARD_PLACEHOLDER + payPalGetCardDetails.getPaymentSource().getCard().getLastDigits());

        modelService.saveAll(paymentInfo, customerFromOrder);
    }

    private void updateExpirationStatus(PayPalCreditCardPaymentInfoModel paymentInfo) {
        if (defaultPayPalConfigurationService.isPayPalCreditCardOnAddingValidation()) {
            paymentInfoService.updateExpirationStatus(paymentInfo);
        }
    }

    @Override
    public PayPalCreditCardPaymentInfoModel updatePayPalPaymentSubscription(final CustomerModel customerModel,
                                                                            final CCPaymentInfoData ccPaymentInfoData, final AddressModel billingAddress,
                                                                            PayPalCreditCardPaymentInfoModel paymentInfo) {
        populatePayPalPaymentInfo(paymentInfo, ccPaymentInfoData);

        paymentInfo.setBillingAddress(billingAddress);
        if (billingAddress.getOwner() == null) {
            billingAddress.setOwner(paymentInfo);
        }
        Optional.ofNullable(ccPaymentInfoData.getCardNumber()).ifPresent(paymentInfo::setNumber);
        getModelService().saveAll(billingAddress, paymentInfo);
        getModelService().refresh(customerModel);

        addPaymentInfo(customerModel, paymentInfo);
        return paymentInfo;
    }

    @Override
    public CustomerData getCustomerDataByPayerId(String payerId) {
        return customerConverter.convert(payPalCustomerAccountDao.findCustomerByPayPalPayerId(payerId));
    }

    @Override
    public boolean isCustomerWithPayerIdExist(String payerId) {
        return payPalCustomerAccountDao.findCustomerByPayPalPayerId(payerId) != null;
    }

    @Override
    public String setTempPassword(String payerId) {
        final CustomerModel customerModel = payPalCustomerAccountDao.findCustomerByPayPalPayerId(payerId);
        final String tempPassword = generateRandomPassword(PAYPAL_TEMP_PASSWORD_LENGTH);
        customerModel.setPassword(tempPassword);
        modelService.save(customerModel);
        return tempPassword;
    }

    @Override
    public void clearTempPassword(String payerId) {
        final CustomerModel customerModel = payPalCustomerAccountDao.findCustomerByPayPalPayerId(payerId);
        customerModel.setPassword(PAYPAL_EMPTY_STRING);
        modelService.save(customerModel);
    }

    @Override
    public CustomerModel getCustomerModelByPayerId(String payerId) {
        return payPalCustomerAccountDao.findCustomerByPayPalPayerId(payerId);
    }

    @Override
    public void setAccessTokenForCustomer(String accessTokenGuid, String payerId) {
        final CustomerModel customerModel = payPalCustomerAccountDao.findCustomerByPayPalPayerId(payerId);
        customerModel.setPayPalAccessToken(accessTokenGuid);
        modelService.save(customerModel);
    }

    @Override
    public void setDefaultAccessToken(String payerId) {
        final CustomerModel customerModel = payPalCustomerAccountDao.findCustomerByPayPalPayerId(payerId);
        customerModel.setPayPalAccessToken(StringUtils.EMPTY);
        modelService.save(customerModel);
    }

    @Override
    public CustomerData getCustomerDataByUid(final String uId) {
        return customerConverter.convert(userService.getUserForUID(uId, CustomerModel.class));
    }

    @Override
    public Optional<CustomerModel> getCustomerByVaultId(String vaultId) {
        return payPalCustomerAccountDao.findCustomerByVaultId(vaultId);
    }

    @Override
    public void setDefaultAddressEntry(final CustomerModel customerModel, final AddressModel addressModel)
    {
        validateParameterNotNull(customerModel, "Customer model cannot be null");
        validateParameterNotNull(addressModel, "Address model cannot be null");
        if (BooleanUtils.isTrue(addressModel.getVisibleInAddressBook()))
        {
            super.setDefaultAddressEntry(customerModel, addressModel);
        }
    }

    @Override
    public CustomerModel getCustomerModel(String payerId) {
        return Optional.ofNullable(payerId)
                .map(this::getCustomerModelByPayerId)
                .orElseGet(() -> (CustomerModel) userService.getCurrentUser());
    }

    private void populatePayPalPaymentInfo(final PayPalCreditCardPaymentInfoModel payPalPaymentInfoModel,
                                           final CCPaymentInfoData ccPaymentInfoData) {
        payPalPaymentInfoModel.setPayerEmail(ccPaymentInfoData.getPayerEmail());
        payPalPaymentInfoModel.setPayPalOrderId(ccPaymentInfoData.getPayPalOrderId());
        payPalPaymentInfoModel.setPayerId(ccPaymentInfoData.getPayerId());
        payPalPaymentInfoModel.setSaved(ccPaymentInfoData.isSaved());
        payPalPaymentInfoModel.setSubscriptionId(ccPaymentInfoData.getSubscriptionId());
        payPalPaymentInfoModel.setCcOwner(ccPaymentInfoData.getAccountHolderName());
        payPalPaymentInfoModel.setType(CreditCardType.valueOf(ccPaymentInfoData.getCardType().toUpperCase()));
        payPalPaymentInfoModel.setPMCustomerVaultId(ccPaymentInfoData.getPMCustomerVaultId());
    }

    private void populateDumbCCInfo(final CreditCardPaymentInfoModel creditCardPaymentInfoModel,
                                    final CustomerModel customerModel) {
        creditCardPaymentInfoModel.setCode(customerModel.getUid() + "_" + UUID.randomUUID());
        if (creditCardPaymentInfoModel.getUser() == null) {
            creditCardPaymentInfoModel.setUser(customerModel);
        }
        creditCardPaymentInfoModel.setValidToMonth(String.valueOf(ZERO));
        creditCardPaymentInfoModel.setValidToYear(String.valueOf(ZERO));
        creditCardPaymentInfoModel.setNumber(getMaskedCardNumber("****************"));
    }

    public void setDefaultPayPalConfigurationService(PayPalConfigurationService defaultPayPalConfigurationService) {
        this.defaultPayPalConfigurationService = defaultPayPalConfigurationService;
    }

    private static String generateRandomPassword(final int len) {
        final SecureRandom random = new SecureRandom();
        final StringBuilder sb = new StringBuilder();
        for (int i = 0; i < len; i++) {
            final int randomIndex = random.nextInt(PAYPAL_CHARACTERS_FOR_PASSWORD.length());
            sb.append(PAYPAL_CHARACTERS_FOR_PASSWORD.charAt(randomIndex));
        }
        return sb.toString();
    }

    public void setPayPalCustomerAccountDao(PayPalCustomerAccountDao payPalCustomerAccountDao) {
        this.payPalCustomerAccountDao = payPalCustomerAccountDao;
    }

    public void setCustomerConverter(
            Converter<CustomerModel, CustomerData> customerConverter) {
        this.customerConverter = customerConverter;
    }

    public void setPaymentInfoService(DefaultPayPalPaymentInfoService paymentInfoService) {
        this.paymentInfoService = paymentInfoService;
    }

    @Override
    public ModelService getModelService() {
        return modelService;
    }

    @Override
    public void setModelService(ModelService modelService) {
        this.modelService = modelService;
    }

    @Override
    public void setUserService(UserService userService) {
        this.userService = userService;
    }

    private boolean isPayPal(CCPaymentInfoData ccPaymentInfoData) {
        return CreditCardType.PAYPAL.name().equalsIgnoreCase(ccPaymentInfoData.getCardType());
    }

    private boolean isVenmo(CCPaymentInfoData ccPaymentInfoData) {
        return CreditCardType.VENMO.name().equalsIgnoreCase(ccPaymentInfoData.getCardType());
    }

}
