package com.paypal.hybris.core.service.impl;

import com.paypal.hybris.core.enums.PaymentProvider;
import com.paypal.hybris.core.model.PayPalCreditCardPaymentInfoModel;
import com.paypal.hybris.core.service.PayPalCommerceCheckoutService;
import com.paypal.hybris.core.service.PayPalConfigurationService;
import com.paypal.hybris.core.service.PayPalPaymentService;
import com.paypal.hybris.core.strategy.PayPalCommercePaymentProviderStrategy;
import com.paypal.hybris.data.PayPalCommerceCheckoutParameter;
import com.paypal.hybris.data.PayPalOrderRequestData;
import de.hybris.platform.commercefacades.order.data.CartData;
import de.hybris.platform.commerceservices.order.CommercePaymentAuthorizationStrategy;
import de.hybris.platform.commerceservices.order.impl.DefaultCommerceCheckoutService;
import de.hybris.platform.commerceservices.service.data.CommerceCheckoutParameter;
import de.hybris.platform.core.model.order.AbstractOrderModel;
import de.hybris.platform.core.model.order.CartModel;
import de.hybris.platform.core.model.order.payment.PaymentInfoModel;
import de.hybris.platform.payment.model.PaymentTransactionEntryModel;
import de.hybris.platform.servicelayer.dto.converter.Converter;
import org.apache.commons.lang3.BooleanUtils;

import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.Map;

import org.apache.log4j.Logger;

import static com.paypal.hybris.core.constants.PaypalcoreConstants.DEFAULT_FLOW;
import static com.paypal.hybris.core.constants.PaypalcoreConstants.PAYPAL_INTENT_CAPTURE;
import static com.paypal.hybris.core.constants.PaypalcoreConstants.PAYPAL_INTENT_CAPTURE_PROCESS_FLOW;
import static com.paypal.hybris.core.constants.PaypalcoreConstants.PAYPAL_ORDER_ID_PLACEHOLDER;
import static com.paypal.hybris.core.constants.PaypalcoreConstants.PAYPAL_ORDER_PROCESS_FLOW;
import static com.paypal.hybris.core.constants.PaypalcoreConstants.PAYPAL_REPLENISHMENT_FLOW;
import static com.paypal.hybris.core.constants.PaypalcoreConstants.PAYPAL_SUBSCRIPTION_ID_PLACEHOLDER;
import static com.paypal.hybris.core.constants.PaypalcoreConstants.PAYPAL_VAULT_SAVED_FLOW;
import static de.hybris.platform.servicelayer.util.ServicesUtil.validateParameterNotNull;


public class DefaultPayPalCommerceCheckoutService extends DefaultCommerceCheckoutService implements
    PayPalCommerceCheckoutService {

    private Map<String, CommercePaymentAuthorizationStrategy> strategyMap;
    private PayPalPaymentService payPalPaymentService;
    private PayPalConfigurationService defaultPayPalConfigurationService;
    private Converter<AbstractOrderModel, PayPalOrderRequestData> orderRequestDataConverter;
    private PayPalCommercePaymentProviderStrategy commercePaymentProviderStrategy;


    @Override
    public String getPaymentProvider(CartModel cart) {

        return getCommercePaymentProviderStrategy().getPaymentProvider(cart);
    }

    @Override
    public PaymentTransactionEntryModel authorizePayment(CommerceCheckoutParameter parameter) {
        final CartModel cartModel = parameter.getCart();
        validateParameterNotNull(cartModel, "Cart model cannot be null");
        validateParameterNotNull(cartModel.getPaymentInfo(), "Payment information on cart cannot be null");
        refreshOrderIntent(cartModel.getPaymentInfo());
        if (parameter.getAuthorizationAmount() == null) {
            parameter.setAuthorizationAmount(calculateAuthAmount(cartModel));
        }
        final String authorizationFlow = getAuthorizationFlow(parameter);
        if (isPayPalOrderShouldBeCreated(authorizationFlow, cartModel.getPaymentInfo())) {
            createPayPalOrder(cartModel);
        }
        return getAuthorizationStrategy(authorizationFlow).authorizePaymentAmount(parameter);
    }

    private void createPayPalOrder(final CartModel cartModel) {
        final PayPalCreditCardPaymentInfoModel paymentInfo = (PayPalCreditCardPaymentInfoModel) cartModel
            .getPaymentInfo();
        PayPalOrderRequestData requestData = orderRequestDataConverter.convert(cartModel);
        requestData.setIntent(getDefaultPayPalConfigurationService().getPayPalIntent());
        requestData.setSaveOrderFlowActive(false);

        String orderId = payPalPaymentService.createOrder(requestData);
        paymentInfo.setPayPalOrderId(orderId);
        paymentInfo.setSubscriptionId(PAYPAL_SUBSCRIPTION_ID_PLACEHOLDER);
        getModelService().save(paymentInfo);
        getModelService().refresh(cartModel);
    }

    private boolean isPayPalOrderShouldBeCreated(final String flow, final PaymentInfoModel paymentInfoModel) {
        if (paymentInfoModel instanceof PayPalCreditCardPaymentInfoModel) {
            return (PAYPAL_ORDER_PROCESS_FLOW.equals(flow) || PAYPAL_INTENT_CAPTURE_PROCESS_FLOW.equals(flow))
                && PAYPAL_ORDER_ID_PLACEHOLDER
                .equals(((PayPalCreditCardPaymentInfoModel) paymentInfoModel).getPayPalOrderId());
        }
        return false;
    }

    private String getAuthorizationFlow(CommerceCheckoutParameter parameter) {
        PaymentInfoModel paymentInfoModel = parameter.getCart().getPaymentInfo();
        if (paymentInfoModel instanceof PayPalCreditCardPaymentInfoModel) {
            if (isReplenishmentFlow(parameter)) {
                return PAYPAL_REPLENISHMENT_FLOW;
            }else if (isVaultedPaymentMethod((PayPalCreditCardPaymentInfoModel) paymentInfoModel)) {
                return PAYPAL_VAULT_SAVED_FLOW;
            } else if (((PayPalCreditCardPaymentInfoModel) paymentInfoModel).getIntent()
                .equalsIgnoreCase(PAYPAL_INTENT_CAPTURE)) {
                return PAYPAL_INTENT_CAPTURE_PROCESS_FLOW;
            } else {
                return PAYPAL_ORDER_PROCESS_FLOW;
            }
        }
        return DEFAULT_FLOW;
    }

    @Override
    public BigDecimal calculateCartTotalWithTax(AbstractOrderModel abstractOrderModel) {
        final Double totalPrice = abstractOrderModel.getTotalPrice();
        final Double totalTax = (abstractOrderModel.getNet() && abstractOrderModel.getStore() != null
                && abstractOrderModel.getStore().getExternalTaxEnabled()) ? abstractOrderModel.getTotalTax() : Double.valueOf(0d);
        final BigDecimal totalPriceWithoutTaxBD = BigDecimal.valueOf(totalPrice == null ? 0d : totalPrice).setScale(2,
                RoundingMode.HALF_EVEN);
        return BigDecimal.valueOf(totalTax == null ? 0d : totalTax)
                .setScale(2, RoundingMode.HALF_EVEN).add(totalPriceWithoutTaxBD);
    }

    private static boolean isVaultedPaymentMethod(PayPalCreditCardPaymentInfoModel paymentInfoModel) {
        return paymentInfoModel.isSaved()
                && (PaymentProvider.PAYPAL.equals(paymentInfoModel.getPaymentProvider())
                || PaymentProvider.PAYPAL_HOSTED_FIELDS.equals(paymentInfoModel.getPaymentProvider()));
    }

    private static boolean isReplenishmentFlow(CommerceCheckoutParameter parameter){
        return parameter instanceof PayPalCommerceCheckoutParameter && BooleanUtils.isTrue(
            ((PayPalCommerceCheckoutParameter) parameter).isReplenishment());
    }

    private CommercePaymentAuthorizationStrategy getAuthorizationStrategy(final String flow) {
        return strategyMap.getOrDefault(flow, strategyMap.get(DEFAULT_FLOW));
    }

    private void refreshOrderIntent(PaymentInfoModel paymentInfoModel) {
        if (paymentInfoModel instanceof PayPalCreditCardPaymentInfoModel) {
            ((PayPalCreditCardPaymentInfoModel) paymentInfoModel)
                .setIntent(getDefaultPayPalConfigurationService().getPayPalIntent());
        }
        getModelService().save(paymentInfoModel);
    }

    public void setStrategyMap(
        Map<String, CommercePaymentAuthorizationStrategy> strategyMap) {
        this.strategyMap = strategyMap;
    }

    public void setPayPalPaymentService(PayPalPaymentService payPalPaymentService) {
        this.payPalPaymentService = payPalPaymentService;
    }

    public PayPalConfigurationService getDefaultPayPalConfigurationService() {
        return defaultPayPalConfigurationService;
    }

    public void setDefaultPayPalConfigurationService(PayPalConfigurationService defaultPayPalConfigurationService) {
        this.defaultPayPalConfigurationService = defaultPayPalConfigurationService;
    }

    @Override
    public PayPalCommercePaymentProviderStrategy getCommercePaymentProviderStrategy() {
        return commercePaymentProviderStrategy;
    }

    public void setCommercePaymentProviderStrategy(PayPalCommercePaymentProviderStrategy commercePaymentProviderStrategy) {
        this.commercePaymentProviderStrategy = commercePaymentProviderStrategy;
    }

    public void setOrderRequestDataConverter(Converter<AbstractOrderModel, PayPalOrderRequestData> orderRequestDataConverter) {
        this.orderRequestDataConverter = orderRequestDataConverter;
    }
}
