package com.paypal.hybris.core.service.impl;

import com.paypal.base.rest.APIContext;
import com.paypal.core.PayPalEnvironment;
import com.paypal.hybris.core.constants.PaypalcoreConstants;
import com.paypal.hybris.core.service.PayPalClientTokenService;
import com.paypal.hybris.core.service.PayPalConfigurationService;
import com.paypal.hybris.data.PayPalGenerateClientTokenData;
import de.hybris.platform.core.model.user.CustomerModel;
import de.hybris.platform.payment.AdapterException;
import de.hybris.platform.servicelayer.user.UserService;
import org.apache.commons.lang3.StringUtils;
import org.apache.log4j.Logger;
import org.apache.logging.log4j.util.Strings;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.client.RestTemplate;
import java.util.HashMap;
import java.util.Map;

import static com.paypal.hybris.core.constants.PaypalcoreConstants.PAYPAL_LIVE_ENVIRONMENT;

public class DefaultPayPalClientTokenService implements PayPalClientTokenService {

    private static final Logger LOG = Logger.getLogger(DefaultPayPalClientTokenService.class);

    public static final String IDENTITY_GENERATE_TOKEN_URL = "identity/generate-token";
    private static final String CUSTOMER_ID = "customer_id";
    private static final String GET_CLIENT_TOKEN_ERROR = "[PayPal Payment Service] Errors during get client token: ";

    private PayPalConfigurationService defaultPayPalConfigurationService;

    private UserService userService;

    private RestTemplate restTemplate;

    @Override
    public PayPalGenerateClientTokenData getClientToken() {

        HttpHeaders headers = new HttpHeaders();
        String getClientTokenUrl = getPayPalApiEndpoint() + IDENTITY_GENERATE_TOKEN_URL;
        headers.setContentType(MediaType.APPLICATION_JSON);
        headers.add(PaypalcoreConstants.AUTHORIZATION_HEADER, createAPIContext().getAccessToken());
        HttpEntity<Map<String, String>> httpEntity;

        CustomerModel currentCustomer = (CustomerModel) userService.getCurrentUser();
        if (Strings.isNotBlank(currentCustomer.getVaultCustomerId())) {
            Map<String, String> entityBody = new HashMap<>();
            entityBody.put(CUSTOMER_ID, currentCustomer.getVaultCustomerId());
            httpEntity = new HttpEntity<>(entityBody, headers);
        } else {
            httpEntity = new HttpEntity<>(headers);
        }

        ResponseEntity<PayPalGenerateClientTokenData> response;
        try {
            response = restTemplate.postForEntity(getClientTokenUrl, httpEntity, PayPalGenerateClientTokenData.class);
        } catch (Exception e) {
            LOG.error(GET_CLIENT_TOKEN_ERROR, e);
            throw new AdapterException(e.getMessage());
        }
        return response.getBody();
    }

    protected PayPalEnvironment createPayPalEnvironment() {
        if (StringUtils.equals(PAYPAL_LIVE_ENVIRONMENT, defaultPayPalConfigurationService.getEnvironmentType())) {
            return new PayPalEnvironment.Live(
                    defaultPayPalConfigurationService.getClientID(),
                    defaultPayPalConfigurationService.getSecretKey());
        }
        return new PayPalEnvironment.Sandbox(
                defaultPayPalConfigurationService.getClientID(),
                defaultPayPalConfigurationService.getSecretKey());
    }

    protected APIContext createAPIContext() {
        final PayPalEnvironment payPalEnvironment = createPayPalEnvironment();
        return new APIContext(payPalEnvironment.clientId(), payPalEnvironment.clientSecret(),
                defaultPayPalConfigurationService.getEnvironmentType());
    }

    protected String getPayPalApiEndpoint() {
        return defaultPayPalConfigurationService.getPayPalApiUrl();
    }

    public void setDefaultPayPalConfigurationService(PayPalConfigurationService defaultPayPalConfigurationService) {
        this.defaultPayPalConfigurationService = defaultPayPalConfigurationService;
    }

    public void setUserService(UserService userService) {
        this.userService = userService;
    }

    public void setRestTemplate(RestTemplate restTemplate) {
        this.restTemplate = restTemplate;
    }
}
