package com.paypal.hybris.core.service.impl;

import de.hybris.platform.commerceservices.address.AddressErrorCode;
import de.hybris.platform.commerceservices.address.AddressFieldType;
import de.hybris.platform.commerceservices.address.AddressVerificationDecision;
import de.hybris.platform.commerceservices.address.AddressVerificationService;
import de.hybris.platform.commerceservices.address.data.AddressFieldErrorData;
import de.hybris.platform.commerceservices.address.data.AddressVerificationResultData;
import de.hybris.platform.commerceservices.address.util.AddressVerificationResultUtils;
import de.hybris.platform.core.model.user.AddressModel;
import de.hybris.platform.store.BaseStoreModel;
import de.hybris.platform.store.services.BaseStoreService;
import org.apache.commons.lang.StringUtils;

import java.util.ArrayList;
import java.util.List;


public class DefaultPayPalAddressVerificationService implements
    AddressVerificationService<AddressVerificationDecision, AddressFieldErrorData<AddressFieldType, AddressErrorCode>> {

    private static final String ACCEPT = "accept";
    private static final String REJECT = "reject";
    private static final String FIRST_NAME = "firstName";
    private static final String LAST_NAME = "lastName";
    private static final String ADDRESS_LINE_1 = "addressline1";
    private static final String ADDRESS_LINE_2 = "addressline2";
    private static final String REGION = "region";
    private static final String ZIP_CODE = "zipcode";
    private static final String CITY = "city";
    private static final String COUNTRY = "country";
    private static final String MISSING = "missing";
    private static final String INVALID = "invalid";
    private static final String REVIEW = "review";
    private static final int ACCEPTABLE_STRING_LENGTH = 255;
    private static final int ACCEPTABLE_CODE_LENGTH = 10;

    private BaseStoreService baseStoreService;

    @Override
    public AddressVerificationResultData<AddressVerificationDecision, AddressFieldErrorData<AddressFieldType, AddressErrorCode>> verifyAddress(
        final AddressModel addressModel) {
        final AddressVerificationResultData<AddressVerificationDecision, AddressFieldErrorData<AddressFieldType, AddressErrorCode>> acceptedResult = createVerificationResult();

        validateAddressFields(acceptedResult, addressModel);
        if (AddressVerificationResultUtils.requiresErrorHandling(acceptedResult)) {
            acceptedResult.setDecision(AddressVerificationDecision.lookup(REJECT));
        } else {
            if (REVIEW.equals(addressModel.getTown())) {
                acceptedResult.setDecision(AddressVerificationDecision.REVIEW);
                final List<AddressModel> suggestedAddresses = new ArrayList<AddressModel>();
                addressModel.setLine1(String.format("%s corrected", addressModel.getLine1()));
                suggestedAddresses.add(addressModel);
                acceptedResult.setSuggestedAddresses(suggestedAddresses);
                return acceptedResult;
            }
            acceptedResult.setDecision(AddressVerificationDecision.lookup(ACCEPT));
        }
        return acceptedResult;
    }

    @Override
    public boolean isCustomerAllowedToIgnoreSuggestions() {
        final BaseStoreModel baseStore = getBaseStoreService().getCurrentBaseStore();
        return baseStore != null && baseStore.isCustomerAllowedToIgnoreSuggestions();
    }

    protected void validateAddressFields(
        final AddressVerificationResultData<AddressVerificationDecision, AddressFieldErrorData<AddressFieldType, AddressErrorCode>> result,
        final AddressModel address) {
        final List<AddressFieldErrorData<AddressFieldType, AddressErrorCode>> errorList = new ArrayList<>();
        addErrorIfMissingOrInvalid(address.getFirstname(), FIRST_NAME, ACCEPTABLE_STRING_LENGTH, errorList);
        addErrorIfMissingOrInvalid(address.getLastname(), LAST_NAME, ACCEPTABLE_STRING_LENGTH, errorList);
        addErrorIfMissingOrInvalid(address.getTown(), CITY, ACCEPTABLE_STRING_LENGTH, errorList);
        addErrorIfMissingOrInvalid(address.getPostalcode(), ZIP_CODE, ACCEPTABLE_CODE_LENGTH, errorList);
        addErrorIfMissingOrInvalid(address.getLine1(), ADDRESS_LINE_1, ACCEPTABLE_STRING_LENGTH, errorList);

        if (StringUtils.isNotEmpty(address.getLine2())
            && StringUtils.length(address.getLine2()) > ACCEPTABLE_STRING_LENGTH) {
            addErrorToVerificationResult(ADDRESS_LINE_2, INVALID, errorList);
        }

        if (address.getRegion() != null && address.getRegion().getIsocode() == null) {
            addErrorToVerificationResult(REGION, MISSING, errorList);
        } else if (address.getRegion() != null
            && StringUtils.length(address.getRegion().getIsocode()) > ACCEPTABLE_STRING_LENGTH) {
            addErrorToVerificationResult(REGION, INVALID, errorList);
        }

        if (address.getCountry() == null || (address.getCountry() != null
            && address.getCountry().getIsocode() == null)) {
            addErrorToVerificationResult(COUNTRY, MISSING, errorList);
        } else if (address.getCountry() != null
            && StringUtils.length(address.getCountry().getIsocode()) > ACCEPTABLE_STRING_LENGTH) {
            addErrorToVerificationResult(COUNTRY, INVALID, errorList);
        }

        result.setFieldErrors(errorList);
    }

    private void addErrorIfMissingOrInvalid(final String value, final String fieldName, int acceptableLength,
        final List<AddressFieldErrorData<AddressFieldType, AddressErrorCode>> errorList) {
        if (StringUtils.isEmpty(value)) {
            addErrorToVerificationResult(fieldName, MISSING, errorList);
        } else if (StringUtils.length(value) > acceptableLength) {
            addErrorToVerificationResult(fieldName, INVALID, errorList);
        }
    }

    protected void addErrorToVerificationResult(final String titleCode, final String missing,
        final List<AddressFieldErrorData<AddressFieldType, AddressErrorCode>> errors) {
        final AddressFieldErrorData<AddressFieldType, AddressErrorCode> errorData = createFieldError();
        errorData.setFieldType(AddressFieldType.lookup(titleCode));
        errorData.setErrorCode(AddressErrorCode.lookup(missing));
        errors.add(errorData);
    }

    protected AddressFieldErrorData<AddressFieldType, AddressErrorCode> createFieldError() {
        return new AddressFieldErrorData<AddressFieldType, AddressErrorCode>();
    }

    protected AddressVerificationResultData<AddressVerificationDecision, AddressFieldErrorData<AddressFieldType, AddressErrorCode>> createVerificationResult() {
        return new AddressVerificationResultData<AddressVerificationDecision, AddressFieldErrorData<AddressFieldType, AddressErrorCode>>();
    }

    protected BaseStoreService getBaseStoreService() {
        return baseStoreService;
    }

    public void setBaseStoreService(final BaseStoreService baseStoreService) {
        this.baseStoreService = baseStoreService;
    }
}
