package com.paypal.hybris.core.jobs;

import com.paypal.hybris.core.dao.PayPalPaymentInfoDao;
import com.paypal.hybris.core.enums.ExpirationStatus;
import com.paypal.hybris.core.model.PayPalCreditCardPaymentInfoModel;
import com.paypal.hybris.core.model.ValidateExpiredCardsCronJobModel;
import com.paypal.hybris.core.util.builder.GenericBuilder;
import de.hybris.platform.core.model.user.CustomerModel;
import de.hybris.platform.core.servicelayer.data.PaginationData;
import de.hybris.platform.core.servicelayer.data.SearchPageData;
import de.hybris.platform.cronjob.enums.CronJobResult;
import de.hybris.platform.cronjob.enums.CronJobStatus;
import de.hybris.platform.servicelayer.cronjob.AbstractJobPerformable;
import de.hybris.platform.servicelayer.cronjob.PerformResult;
import org.apache.log4j.Logger;
import java.time.YearMonth;

public class ValidateExpiredCardsJobPerformable extends AbstractJobPerformable<ValidateExpiredCardsCronJobModel> {

    private static final Logger LOG = Logger.getLogger(ValidateExpiredCardsJobPerformable.class);

    private static final String VALIDATING_ERROR_MESSAGE = "Invalid expiration date on card: ";
    private static final String REGEX_FOR_VALIDATION = "^(0+[1-9]|1[012])$";

    private PayPalPaymentInfoDao payPalPaymentInfoDao;


    @Override
    public PerformResult perform(ValidateExpiredCardsCronJobModel cronJobModel) {
        int payPalExpireBetween = cronJobModel.getStore().getPayPalPluginConfiguration().getHostedFieldsConfiguration().getExpireBetween();
        setExpiredStatus(cronJobModel);
        if (payPalExpireBetween > 0) {
            setExpireSoonStatus(payPalExpireBetween, cronJobModel);
        }
        return new PerformResult(CronJobResult.SUCCESS, CronJobStatus.FINISHED);
    }


    private void setExpireSoonStatus(int payPalExpireBetween, ValidateExpiredCardsCronJobModel cronJobModel) {
        YearMonth expireSoonDate = YearMonth.now().plusMonths(payPalExpireBetween);
        SearchPageData<PayPalCreditCardPaymentInfoModel> searchPageData = new SearchPageData<>();
        searchPageData.setPagination(createPaginationData(cronJobModel.getPageSize()));
        do {
            searchPageData = payPalPaymentInfoDao.findAllExpiredSoon(expireSoonDate, searchPageData);
            searchPageData.getResults().forEach(card -> {
                if (validateExpirationDate(card)) {
                    card.setExpirationStatus(ExpirationStatus.EXPIRE_SOON);
                    CustomerModel customer =(CustomerModel) card.getUser();
                    customer.setHasNewExpireSoonCard(true);
                    modelService.saveAll(card, customer);
                } else {
                    LOG.error(VALIDATING_ERROR_MESSAGE + card.getPk());
                }
            });
            searchPageData.getPagination().setCurrentPage(searchPageData.getPagination().getCurrentPage() + 1);
        } while (searchPageData.getPagination().getHasNext());

    }


    private void setExpiredStatus(ValidateExpiredCardsCronJobModel cronJobModel) {
        SearchPageData<PayPalCreditCardPaymentInfoModel> searchPageData = new SearchPageData<>();
        searchPageData.setPagination(createPaginationData(cronJobModel.getPageSize()));
        do {
            searchPageData = payPalPaymentInfoDao.findAllExpiredCards(searchPageData);
            searchPageData.getResults().forEach(card -> {
                if (validateExpirationDate(card)) {
                    card.setExpirationStatus(ExpirationStatus.EXPIRED);
                    modelService.save(card);
                } else {
                    LOG.error(VALIDATING_ERROR_MESSAGE + card.getPk());
                }
            });
            searchPageData.getPagination().setCurrentPage(searchPageData.getPagination().getCurrentPage() + 1);
        } while (searchPageData.getPagination().getHasNext());
    }

    private boolean validateExpirationDate(PayPalCreditCardPaymentInfoModel card) {
        return card.getValidToMonth().matches(REGEX_FOR_VALIDATION) && card.getValidToYear() != null;
    }

    private PaginationData createPaginationData(int pageSize) {
        return GenericBuilder.of(PaginationData::new)
            .with(PaginationData::setCurrentPage, 0)
            .with(PaginationData::setNeedsTotal, true)
            .with(PaginationData::setPageSize, pageSize)
            .build();
    }

    public void setPayPalPaymentInfoDao(PayPalPaymentInfoDao payPalPaymentInfoDao) {
        this.payPalPaymentInfoDao = payPalPaymentInfoDao;
    }

}
