package com.paypal.hybris.core.executor.impl;

import com.paypal.hybris.core.executor.PayPalImmediateCancelRequestExecutor;
import com.paypal.hybris.core.service.PayPalPaymentService;
import de.hybris.platform.basecommerce.enums.OrderEntryStatus;
import de.hybris.platform.core.model.order.AbstractOrderEntryModel;
import de.hybris.platform.core.model.order.OrderModel;
import de.hybris.platform.ordercancel.OrderCancelEntry;
import de.hybris.platform.ordercancel.OrderCancelException;
import de.hybris.platform.ordercancel.OrderCancelNotificationServiceAdapter;
import de.hybris.platform.ordercancel.OrderCancelPaymentServiceAdapter;
import de.hybris.platform.ordercancel.OrderCancelRecordsHandler;
import de.hybris.platform.ordercancel.OrderCancelRequest;
import de.hybris.platform.ordercancel.OrderCancelRequestExecutor;
import de.hybris.platform.ordercancel.OrderCancelResponse;
import de.hybris.platform.ordercancel.OrderStatusChangeStrategy;
import de.hybris.platform.ordercancel.OrderUtils;
import de.hybris.platform.ordercancel.impl.executors.ImmediateCancelRequestExecutor;
import de.hybris.platform.ordercancel.model.OrderCancelRecordEntryModel;
import de.hybris.platform.servicelayer.model.ModelService;
import org.apache.log4j.Logger;

import java.util.Iterator;


public class DefaultPayPalImmediateCancelRequestExecutor implements PayPalImmediateCancelRequestExecutor,
    OrderCancelRequestExecutor {

    private static final Logger LOG = Logger.getLogger(ImmediateCancelRequestExecutor.class.getName());

    private ModelService modelService;
    private OrderCancelPaymentServiceAdapter paymentServiceAdapter;
    private OrderCancelNotificationServiceAdapter notificationServiceAdapter;
    private OrderCancelRecordsHandler orderCancelRecordsHandler;
    private OrderStatusChangeStrategy completeCancelStatusChangeStrategy;
    private OrderStatusChangeStrategy partialCancelStatusChangeStrategy;
    private PayPalPaymentService paymentService;

    @Override
    public void processCancelRequest(OrderCancelRequest orderCancelRequest,
        OrderCancelRecordEntryModel cancelRequestRecordEntry)
        throws OrderCancelException {
        this.modifyOrderAccordingToRequest(orderCancelRequest);
        final OrderModel order = orderCancelRequest.getOrder();
        getPaymentService().doCancel(order);
        this.modelService.refresh(order);
        if (!OrderUtils.hasLivingEntries(order)) {
            if (this.completeCancelStatusChangeStrategy != null) {
                this.completeCancelStatusChangeStrategy
                    .changeOrderStatusAfterCancelOperation(cancelRequestRecordEntry, true);
            }
        } else if (this.partialCancelStatusChangeStrategy != null) {
            this.partialCancelStatusChangeStrategy
                .changeOrderStatusAfterCancelOperation(cancelRequestRecordEntry, true);
        }

        if (this.paymentServiceAdapter != null) {
            this.paymentServiceAdapter.recalculateOrderAndModifyPayments(orderCancelRequest.getOrder());
        } else if (LOG.isDebugEnabled()) {
            LOG.debug("Missing OrderCancelPaymentServiceAdapter!");
        }

        if (this.notificationServiceAdapter == null) {
            LOG.info("order: " + orderCancelRequest.getOrder().getCode() + " has been " + (
                !orderCancelRequest.isPartialCancel() ?
                    "completely" :
                    "partially") + " cancelled");
        } else {
            this.notificationServiceAdapter.sendCancelFinishedNotifications(cancelRequestRecordEntry);
        }

        this.orderCancelRecordsHandler
            .updateRecordEntry(this.makeInternalResponse(orderCancelRequest, true, (String) null));
    }

    protected void modifyOrderAccordingToRequest(OrderCancelRequest cancelRequest) throws OrderCancelException {
        AbstractOrderEntryModel orderEntry;
        for (final Iterator var3 = cancelRequest.getEntriesToCancel().iterator(); var3.hasNext();
            this.modelService.save(orderEntry)) {
            final OrderCancelEntry oce = (OrderCancelEntry) var3.next();
            orderEntry = oce.getOrderEntry();
            final long previousQuantity = orderEntry.getQuantity();
            if (oce.getCancelQuantity() > oce.getOrderEntry().getQuantity()) {
                throw new OrderCancelException(cancelRequest.getOrder().getCode(),
                    "Trying to cancel " + oce.getCancelQuantity() + ", whereas orderEntry (" + orderEntry.getPk()
                        + ") has quantity of " + previousQuantity);
            }

            orderEntry.setQuantity(previousQuantity - oce.getCancelQuantity());
            if (previousQuantity == oce.getCancelQuantity()) {
                orderEntry.setQuantityStatus(OrderEntryStatus.DEAD);
            }
        }

    }

    public void setModelService(ModelService modelService) {
        this.modelService = modelService;
    }

    public void setPaymentServiceAdapter(OrderCancelPaymentServiceAdapter paymentServiceAdapter) {
        this.paymentServiceAdapter = paymentServiceAdapter;
    }

    public void setNotificationServiceAdapter(OrderCancelNotificationServiceAdapter notificationServiceAdapter) {
        this.notificationServiceAdapter = notificationServiceAdapter;
    }

    protected OrderCancelResponse makeInternalResponse(OrderCancelRequest request, boolean success, String message) {
        return request.isPartialCancel() ?
            new OrderCancelResponse(request.getOrder(), request.getEntriesToCancel(),
                success ? OrderCancelResponse.ResponseStatus.partial : OrderCancelResponse.ResponseStatus.error,
                message) :
            new OrderCancelResponse(request.getOrder(),
                success ? OrderCancelResponse.ResponseStatus.full : OrderCancelResponse.ResponseStatus.error, message);
    }

    public void setOrderCancelRecordsHandler(OrderCancelRecordsHandler orderCancelRecordsHandler) {
        this.orderCancelRecordsHandler = orderCancelRecordsHandler;
    }

    public void setCompleteCancelStatusChangeStrategy(OrderStatusChangeStrategy completeCancelStatusChangeStrategy) {
        this.completeCancelStatusChangeStrategy = completeCancelStatusChangeStrategy;
    }

    public void setPartialCancelStatusChangeStrategy(OrderStatusChangeStrategy partialCancelStatusChangeStrategy) {
        this.partialCancelStatusChangeStrategy = partialCancelStatusChangeStrategy;
    }

    public PayPalPaymentService getPaymentService() {
        return paymentService;
    }

    public void setPaymentService(PayPalPaymentService paymentService) {
        this.paymentService = paymentService;
    }
}
