/**
 *
 */
package com.paypal.hybris.core.dao.impl;

import com.paypal.hybris.core.dao.PayPalPaymentInfoDao;
import com.paypal.hybris.core.enums.ExpirationStatus;
import com.paypal.hybris.core.enums.PaymentProvider;
import com.paypal.hybris.core.model.PayPalCreditCardPaymentInfoModel;
import de.hybris.platform.core.servicelayer.data.SearchPageData;
import de.hybris.platform.servicelayer.exceptions.ModelNotFoundException;
import de.hybris.platform.servicelayer.internal.dao.AbstractItemDao;
import de.hybris.platform.servicelayer.search.FlexibleSearchQuery;
import de.hybris.platform.servicelayer.search.paginated.PaginatedFlexibleSearchParameter;
import de.hybris.platform.servicelayer.search.paginated.PaginatedFlexibleSearchService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import java.time.YearMonth;
import java.util.HashMap;
import java.util.Map;
import java.util.Optional;

public class DefaultPayPalPaymentInfoDao extends AbstractItemDao implements PayPalPaymentInfoDao {

    private static final Logger LOG = LoggerFactory.getLogger(DefaultPayPalPaymentInfoDao.class);

    private static final String SAVED = "saved";
    private static final String DUPLICATE = "duplicate";
    private static final String SUBSCRIPTIONID = "subscriptionId";
    private static final String PAY_PAL_ORDER_ID = "payPalOrderId";

    private static final String GET_CARD_BY_SUBSCRIPTION_ID = "SELECT {" + PayPalCreditCardPaymentInfoModel.PK + "} FROM {" + PayPalCreditCardPaymentInfoModel._TYPECODE + "} "
            + "WHERE {" + PayPalCreditCardPaymentInfoModel.SAVED + "}=?" + SAVED + " AND "
            + "{" + PayPalCreditCardPaymentInfoModel.DUPLICATE + "}=?" + DUPLICATE + " AND "
            + "{" + PayPalCreditCardPaymentInfoModel.SUBSCRIPTIONID + "}=?" + SUBSCRIPTIONID;

    private static final String GET_CREDIT_CARD_PAYMENT_INFO_BY_ORDER_ID =
            "SELECT {pk} FROM {" + PayPalCreditCardPaymentInfoModel._TYPECODE + "}"
                    + " WHERE {" + PayPalCreditCardPaymentInfoModel.PAYPALORDERID + "} = ?" + PAY_PAL_ORDER_ID
                    + " AND {" + PayPalCreditCardPaymentInfoModel.DUPLICATE + "} = ?" + DUPLICATE;

    private static final String PK = "PK";
    private static final String GET_CREDIT_CARD_PAYMENT_INFO_BY_PK =
            "SELECT {pk} FROM {" + PayPalCreditCardPaymentInfoModel._TYPECODE + "}"
                    + " WHERE {" + PayPalCreditCardPaymentInfoModel.PK + "} = ?" + PK
                    + " AND {" + PayPalCreditCardPaymentInfoModel.DUPLICATE + "} = ?" + DUPLICATE;

    private PaginatedFlexibleSearchService paginatedFlexibleSearchService;

    @Override
    public Optional<PayPalCreditCardPaymentInfoModel> findPaymentInfoByPayPalOrderId(String payPalOrderId) {
        FlexibleSearchQuery searchQuery = new FlexibleSearchQuery(GET_CREDIT_CARD_PAYMENT_INFO_BY_ORDER_ID);
        searchQuery.addQueryParameter(PAY_PAL_ORDER_ID, payPalOrderId);
        searchQuery.addQueryParameter(DUPLICATE, false);

        try {
            return Optional.of(getFlexibleSearchService().searchUnique(searchQuery));
        } catch (ModelNotFoundException exception) {
            LOG.error("The PayPalCreditCardPaymentInfoModel by paypalOrderId = {} was not found", payPalOrderId);
        }

        return Optional.empty();
    }

    @Override
    public Optional<PayPalCreditCardPaymentInfoModel> findPaymentInfoByPK(String pk) {
        FlexibleSearchQuery searchQuery = new FlexibleSearchQuery(GET_CREDIT_CARD_PAYMENT_INFO_BY_PK);
        searchQuery.addQueryParameter(PK, pk);
        searchQuery.addQueryParameter(DUPLICATE, false);

        try {
            return Optional.of(getFlexibleSearchService().searchUnique(searchQuery));
        } catch (ModelNotFoundException exception) {
            LOG.error("The PayPalCreditCardPaymentInfoModel by PK = {} was not found", pk);
        }

        return Optional.empty();
    }

    @Override
    public void removePaymentInfoByPK(String pk) {
        this.findPaymentInfoByPK(pk).ifPresent(pm -> getModelService().remove(pm));
    }

    @Override
    public Optional<PayPalCreditCardPaymentInfoModel> findCardBySubscriptionId(String subscriptionId) {
        final Map queryParams = new HashMap();
        queryParams.put(SAVED, Boolean.TRUE);
        queryParams.put(DUPLICATE, Boolean.FALSE);
        queryParams.put(SUBSCRIPTIONID, subscriptionId);

        try {
            return Optional.of(getFlexibleSearchService().searchUnique(new FlexibleSearchQuery(GET_CARD_BY_SUBSCRIPTION_ID, queryParams)));
        } catch (ModelNotFoundException e) {
            return Optional.empty();
        }
    }


    @Override
    public SearchPageData<PayPalCreditCardPaymentInfoModel> findAllExpiredCards(SearchPageData<PayPalCreditCardPaymentInfoModel> searchPageData) {
        FlexibleSearchQuery findExpiredCardsQuery = new FlexibleSearchQuery(
                "SELECT {c:" + PayPalCreditCardPaymentInfoModel.PK + "} FROM {"
                        + PayPalCreditCardPaymentInfoModel._TYPECODE + " as c join " + PaymentProvider._TYPECODE
                        + " as pp on {pp.pk}={c." + PayPalCreditCardPaymentInfoModel.PAYMENTPROVIDER + "} join "
                        + ExpirationStatus._TYPECODE + " as es on {c." + PayPalCreditCardPaymentInfoModel.EXPIRATIONSTATUS
                        + "}={es.pk}} WHERE {pp:code}='"
                        + PaymentProvider.PAYPAL_HOSTED_FIELDS
                        + "' AND {c:" + PayPalCreditCardPaymentInfoModel.SAVED
                        + "}=true AND (cast({c:" + PayPalCreditCardPaymentInfoModel.VALIDTOYEAR + "} as unsigned)<"
                        + YearMonth.now().getYear() + " OR (cast({c:" + PayPalCreditCardPaymentInfoModel.VALIDTOYEAR
                        + "} as unsigned)="
                        + YearMonth.now().getYear() + " AND cast({c:" + PayPalCreditCardPaymentInfoModel.VALIDTOMONTH
                        + "} as unsigned)<"
                        + YearMonth.now().getMonthValue() + ")) AND {es:code}!='" + ExpirationStatus.EXPIRED + "'");

        return searchCardsWithPagination(searchPageData, findExpiredCardsQuery);
    }

    @Override
    public SearchPageData<PayPalCreditCardPaymentInfoModel> findAllExpiredSoon(YearMonth expireSoonValue, SearchPageData<PayPalCreditCardPaymentInfoModel> searchPageData) {
        FlexibleSearchQuery findExpiredSoonCardsQuery = new FlexibleSearchQuery(
                "SELECT {c:" + PayPalCreditCardPaymentInfoModel.PK + "} FROM {"
                        + PayPalCreditCardPaymentInfoModel._TYPECODE + " as c join " + PaymentProvider._TYPECODE
                        + " as pp on {pp.pk}={c."
                        + PayPalCreditCardPaymentInfoModel.PAYMENTPROVIDER + "} join " + ExpirationStatus._TYPECODE
                        + " as es on {c." + PayPalCreditCardPaymentInfoModel.EXPIRATIONSTATUS + "}={es.pk}} WHERE {pp:code}='"
                        + PaymentProvider.PAYPAL_HOSTED_FIELDS
                        + "' AND {c:" + PayPalCreditCardPaymentInfoModel.SAVED
                        + "}=true AND (cast({c:" + PayPalCreditCardPaymentInfoModel.VALIDTOYEAR + "} as unsigned)<"
                        + expireSoonValue.getYear() + " OR (cast({c:" + PayPalCreditCardPaymentInfoModel.VALIDTOYEAR
                        + "} as unsigned)="
                        + expireSoonValue.getYear() + " AND cast({c:" + PayPalCreditCardPaymentInfoModel.VALIDTOMONTH
                        + "} as unsigned)<="
                        + expireSoonValue.getMonthValue() + ")) AND {es:code}!='" + ExpirationStatus.EXPIRED + "'"
                        + "AND {es:code}!='" + ExpirationStatus.EXPIRE_SOON + "'");

        return searchCardsWithPagination(searchPageData, findExpiredSoonCardsQuery);

    }

    private SearchPageData<PayPalCreditCardPaymentInfoModel> searchCardsWithPagination(SearchPageData<PayPalCreditCardPaymentInfoModel> searchPageData,
                                                                                       FlexibleSearchQuery flexibleQuery) {
        final PaginatedFlexibleSearchParameter parameter = new PaginatedFlexibleSearchParameter();
        parameter.setSearchPageData(searchPageData);
        parameter.setFlexibleSearchQuery(flexibleQuery);
        return paginatedFlexibleSearchService.search(parameter);
    }

    public void setPaginatedFlexibleSearchService(
            PaginatedFlexibleSearchService paginatedFlexibleSearchService) {
        this.paginatedFlexibleSearchService = paginatedFlexibleSearchService;
    }

}
