package com.paypal.hybris.core.commands.impl;

import com.paypal.http.HttpResponse;
import com.paypal.hybris.core.commands.PayPalAbstractCommand;
import com.paypal.hybris.core.exception.PayPalVoidAdapterException;
import com.paypal.hybris.core.results.PayPalVoidResult;
import com.paypal.hybris.core.util.PayPalCommandsUtil;
import com.paypal.hybris.core.util.builder.GenericBuilder;
import com.paypal.payments.AuthorizationsVoidRequest;
import de.hybris.platform.payment.commands.VoidCommand;
import de.hybris.platform.payment.commands.request.VoidRequest;
import de.hybris.platform.payment.commands.result.VoidResult;
import de.hybris.platform.payment.dto.TransactionStatus;
import de.hybris.platform.payment.dto.TransactionStatusDetails;
import org.apache.log4j.Logger;

import java.io.IOException;
import java.util.Optional;


public class DefaultPayPalVoidCommand extends PayPalAbstractCommand implements VoidCommand {

	private static final String CODE_RESPONSE = "204";
	private static final String PAYPAL_DEBUG_ID = "Paypal-Debug-Id";
	private static final String TRANSACTION_CANCEL_FAILED_MESSAGE = "Transaction cancel failed: ";
	private static final String REQUEST_ID_EXCEPTION_MESSAGE = "Request ID is undefined, actual Request ID is: '%s'";

	private static final Logger LOG = Logger.getLogger(DefaultPayPalVoidCommand.class);

	@Override
	public VoidResult perform(VoidRequest request) {
		AuthorizationsVoidRequest authorizationsVoidRequest = getAuthorizationsVoidRequest(request);

		HttpResponse<Void> response = performVoidRequest(authorizationsVoidRequest);

		return translateResponse(response, request);
	}

	private AuthorizationsVoidRequest getAuthorizationsVoidRequest(VoidRequest request) {
		return Optional.ofNullable(request.getRequestId())
				.map(AuthorizationsVoidRequest::new).orElseThrow(()
						-> new IllegalArgumentException(
						REQUEST_ID_EXCEPTION_MESSAGE.formatted(request.getRequestId())));
	}

	private HttpResponse<Void> performVoidRequest(AuthorizationsVoidRequest authorizationsVoidRequest) {
		try {
			return createClient().execute(authorizationsVoidRequest);
		} catch (IOException e) {
			LOG.error(TRANSACTION_CANCEL_FAILED_MESSAGE, e);
			throw new PayPalVoidAdapterException(e, authorizationsVoidRequest);
		}
	}

	private VoidResult translateResponse(HttpResponse<Void> response, VoidRequest request) {
		String resultStatus = Integer.toString(response.statusCode());
		final TransactionStatus transactionStatus = CODE_RESPONSE.equals(resultStatus) ? TransactionStatus.ACCEPTED
				: TransactionStatus.REJECTED;
		final TransactionStatusDetails transactionStatusDetails = CODE_RESPONSE.equals(resultStatus) ?
				TransactionStatusDetails.SUCCESFULL : TransactionStatusDetails.PROCESSOR_DECLINE;

		return GenericBuilder.of(PayPalVoidResult::new)
				.with(VoidResult::setAmount, request.getTotalAmount())
				.with(VoidResult::setCurrency, request.getCurrency())
				.with(VoidResult::setRequestId, request.getRequestId())
				.with(VoidResult::setMerchantTransactionCode, request.getMerchantTransactionCode())
				.with(VoidResult::setTransactionStatus, transactionStatus)
				.with(VoidResult::setTransactionStatusDetails, transactionStatusDetails)
				.with(PayPalVoidResult::setDebugId, response.headers().header(PAYPAL_DEBUG_ID))
				.with(PayPalVoidResult::setRequestField, PayPalCommandsUtil.getValueAsString(request))
				.with(PayPalVoidResult::setResponseField, PayPalCommandsUtil.getValueAsString(response.result()))
				.build();
	}
}
