package com.paypal.hybris.core.commands.impl;

import com.paypal.base.rest.JSONFormatter;
import com.paypal.core.AuthorizationProvider;
import com.paypal.core.PayPalEnvironment;
import com.paypal.core.PayPalHttpClient;
import com.paypal.core.object.AccessToken;
import com.paypal.hybris.core.commands.PayPalAbstractCommand;
import com.paypal.hybris.core.util.builder.GenericBuilder;
import com.paypal.hybris.data.PayPalIntegrationPatch;
import com.paypal.hybris.data.PayPalOrderRequestData;
import com.paypal.hybris.data.SupplementaryData;
import de.hybris.platform.payment.AdapterException;
import de.hybris.platform.payment.commands.Command;
import org.apache.log4j.Logger;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.MediaType;
import org.springframework.web.client.RestClientException;
import org.springframework.web.client.RestTemplate;

import java.io.IOException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;


public class DefaultPayPalUpdateOrderAmountCommand extends PayPalAbstractCommand
    implements Command<PayPalOrderRequestData, Boolean> {

    private static final Logger LOG = Logger.getLogger(DefaultPayPalUpdateOrderAmountCommand.class);

    private static final String AMOUNT_PATH = "/purchase_units/@reference_id=='default'/amount";
    private static final String ITEMS_PATH = "/purchase_units/@reference_id=='default'/items";
    private static final String L2_L3_DATA_PATH = "/purchase_units/@reference_id=='default'/supplementary_data/card";
    private static final String PATCH_URL = "/v2/checkout/orders/{id}";
    private static final String REPLACE = "replace";
    private static final String BEARER_PREFIX = "Bearer ";
    private static final String UPDATE_ORDER_REQUEST = "Update order request body: \n %s";

    private RestTemplate restTemplate;

    @Override
    public Boolean perform(final PayPalOrderRequestData orderDetails) {
        List<PayPalIntegrationPatch> patches = preparePatches(orderDetails);

        Map<String, String> variable = new HashMap<>();
        variable.put("id", orderDetails.getOrderId());

        PayPalEnvironment payPalEnvironment = createPayPalEnvironment();
        try {
            HttpHeaders headers = prepareHeaders();
            HttpEntity<List<PayPalIntegrationPatch>> entity = new HttpEntity<>(patches, headers);
            LOG.info(UPDATE_ORDER_REQUEST.formatted(JSONFormatter.toJSON(entity.getBody())));
            restTemplate.patchForObject(payPalEnvironment.baseUrl() + PATCH_URL, entity, Boolean.class, variable);
        } catch (RestClientException | IOException e) {
            LOG.error("Updating Order amount failed: ", e);
            throw new AdapterException(getDescriptionFromPayPalErrorMessage(e.getMessage()));
        }
        return true;
    }

    protected List<PayPalIntegrationPatch> preparePatches(PayPalOrderRequestData orderDetails) {
        List<PayPalIntegrationPatch> patches = new ArrayList<>();
        patches.add(createPatch(REPLACE, AMOUNT_PATH, orderDetails.getBreakdownAmountData()));
        patches.add(createPatch(REPLACE, ITEMS_PATH, orderDetails.getOrderItems()));
        Optional.ofNullable(orderDetails.getSupplementaryData())
                .map(SupplementaryData::getCard)
                .ifPresent(level2Level3CardData -> patches.add(createPatch(REPLACE, L2_L3_DATA_PATH, level2Level3CardData)));
        return patches;
    }

    protected PayPalIntegrationPatch createPatch(String op, String path, Object value) {
        return GenericBuilder.of(PayPalIntegrationPatch::new)
                .with(PayPalIntegrationPatch::setOp, op)
                .with(PayPalIntegrationPatch::setPath, path)
                .with(PayPalIntegrationPatch::setValue, value)
                .build();
    }
    private HttpHeaders prepareHeaders() throws IOException {
        HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        PayPalHttpClient client = createClient();
        AccessToken accessToken = AuthorizationProvider.sharedInstance().authorize(client, null);
        headers.set(HttpHeaders.AUTHORIZATION, BEARER_PREFIX + accessToken.accessToken());
        return headers;
    }

    public void setRestTemplate(RestTemplate restTemplate) {
        this.restTemplate = restTemplate;
    }
}
