package com.paypal.hybris.core.commands.impl;

import com.paypal.base.rest.PayPalRESTException;
import com.paypal.hybris.core.commands.PayPalAbstractCommand;
import com.paypal.hybris.core.constants.PaypalcoreConstants;
import com.paypal.hybris.core.exception.PayPalProcessHttpClientErrorException;
import com.paypal.hybris.core.exception.PayPalProcessPaymentException;
import com.paypal.hybris.core.results.PayPalAuthorizationResult;
import com.paypal.hybris.core.util.PayPalCommandsUtil;
import com.paypal.hybris.core.util.builder.GenericBuilder;
import com.paypal.hybris.data.PayPalOrderProcessRequestData;
import com.paypal.hybris.data.PayPalOrderResponseData;
import com.paypal.hybris.data.PayPalPaymentCollectionData;
import com.paypal.hybris.data.PayPalPaymentData;
import com.paypal.hybris.data.PayPalTokenData;
import com.paypal.hybris.data.PaymentSourceData;
import com.paypal.hybris.data.PurchaseUnitData;
import de.hybris.platform.payment.commands.SubscriptionAuthorizationCommand;
import de.hybris.platform.payment.commands.request.SubscriptionAuthorizationRequest;
import de.hybris.platform.payment.commands.result.AuthorizationResult;
import de.hybris.platform.payment.dto.TransactionStatus;
import de.hybris.platform.payment.dto.TransactionStatusDetails;
import de.hybris.platform.store.services.BaseStoreService;
import org.apache.log4j.Logger;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.client.HttpClientErrorException;
import org.springframework.web.client.RestTemplate;

import java.util.Optional;
import java.util.function.Predicate;

import static com.paypal.hybris.core.constants.PaypalcoreConstants.BILLING_AGREEMENT;
import static com.paypal.hybris.core.constants.PaypalcoreConstants.CHECKOUT_URL;
import static com.paypal.hybris.core.constants.PaypalcoreConstants.CREATED_STATUS_RESULT;
import static com.paypal.hybris.core.constants.PaypalcoreConstants.PAYPAL_CHECKOUT_PAYMENT_ERROR_MSG;


public class DefaultPayPalSubscriptionAuthorizationCommand extends PayPalAbstractCommand implements
    SubscriptionAuthorizationCommand {

    private static final String PAYPAL_METADATA_ID = "PAYPAL-CLIENT-METADATA-ID";

    private static final String DEFAULT = "default";

    private static final Logger LOG = Logger.getLogger(DefaultPayPalSubscriptionAuthorizationCommand.class);

    private static final String AUTHORIZE = "/authorize";
    private static final String PAYPAL_DEBUG_ID = "Paypal-Debug-Id";

    private RestTemplate restTemplate;
    private BaseStoreService baseStoreService;

    @Override
    public AuthorizationResult perform(SubscriptionAuthorizationRequest request) {
        final HttpEntity<PayPalOrderProcessRequestData> httpEntity = createHttpEntity(request);

        final ResponseEntity<PayPalOrderResponseData> response = performAuthorizationsCaptureRequest(request, httpEntity);

        return translateResponse(response, request);
    }

    private HttpEntity<PayPalOrderProcessRequestData> createHttpEntity(SubscriptionAuthorizationRequest request) {
        HttpHeaders headers = prepareHeaders();
        PayPalOrderProcessRequestData body = prepareRequestBody(request);
        return new HttpEntity<>(body, headers);
    }

    private HttpHeaders prepareHeaders() {
        final HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        headers.add(PaypalcoreConstants.AUTHORIZATION_HEADER, getAuthorizationHeaderValue());
        if (baseStoreService.getCurrentBaseStore() != null
                && getDefaultPayPalConfigurationService().isFraudNetEnabled()) {
            String correlationId = getCartService().getSessionCart().getPayPalClientMetadataId();
            headers.add(PAYPAL_METADATA_ID, correlationId);
            LOG.info("[FraudNet] Client MetaData ID - " + correlationId);
        }
        return headers;
    }

    private String getAuthorizationHeaderValue() {
        try {
            return createAPIContext().fetchAccessToken();
        } catch (PayPalRESTException e) {
            LOG.error("Error while fetching authorization header: " + e.getMessage());
            return null;
        }
    }

    private ResponseEntity<PayPalOrderResponseData> performAuthorizationsCaptureRequest(SubscriptionAuthorizationRequest request,
                                                                                        HttpEntity<PayPalOrderProcessRequestData> httpEntity) {
        final String captureUrl = createPayPalEnvironment().baseUrl() + CHECKOUT_URL +
                request.getSubscriptionID() + AUTHORIZE;
        try {
            return restTemplate.postForEntity(captureUrl, httpEntity, PayPalOrderResponseData.class);
        } catch (HttpClientErrorException e) {
            LOG.error("Authorization failed: ", e);
            throw new PayPalProcessHttpClientErrorException(e, request, PAYPAL_CHECKOUT_PAYMENT_ERROR_MSG);
        }
    }

    private AuthorizationResult translateResponse(ResponseEntity<PayPalOrderResponseData> response,
        SubscriptionAuthorizationRequest request) {
        PayPalOrderResponseData responseData = Optional.ofNullable(response.getBody()).orElseThrow();
        final TransactionStatus transactionStatus = getTransactionStatusMap()
                .getOrDefault(responseData.getStatus(), TransactionStatus.ERROR);
        final TransactionStatusDetails transactionStatusDetails = getTransactionStatusDetailsMap().
                getOrDefault(responseData.getStatus(), TransactionStatusDetails.GENERAL_SYSTEM_ERROR);

        PayPalPaymentData authorization = getPaymentData(responseData);

        return GenericBuilder.of(PayPalAuthorizationResult::new)
                .with(AuthorizationResult::setRequestId, authorization.getId())
                .with(AuthorizationResult::setAuthorizationCode, authorization.getId())
                .with(AuthorizationResult::setPaymentProvider, request.getPaymentProvider())
                .with(AuthorizationResult::setCurrency, request.getCurrency())
                .with(AuthorizationResult::setTotalAmount, request.getTotalAmount())
                .with(AuthorizationResult::setReconciliationId, request.getSubscriptionID())
                .with(AuthorizationResult::setMerchantTransactionCode, request.getMerchantTransactionCode())
                .with(AuthorizationResult::setTransactionStatus, transactionStatus)
                .with(AuthorizationResult::setTransactionStatusDetails, transactionStatusDetails)
                .with(PayPalAuthorizationResult::setDebugId, response.getHeaders().get(PAYPAL_DEBUG_ID).get(0))
                .with(PayPalAuthorizationResult::setRequestField, PayPalCommandsUtil.getValueAsString(request))
                .with(PayPalAuthorizationResult::setResponseField, PayPalCommandsUtil.getValueAsString(response))
                .with(PayPalAuthorizationResult::setExpirationTime, authorization.getExpirationTime())
                .with(PayPalAuthorizationResult::setCreateTime, authorization.getCreateTime())
                .with(PayPalAuthorizationResult::setUpdateTime, authorization.getUpdateTime())
                .with(PayPalAuthorizationResult::setCustomerId,
                        PayPalCommandsUtil.getCustomerIdFromPayPalOrderResponseData(responseData))
                .with(PayPalAuthorizationResult::setPaymentToken,
                        PayPalCommandsUtil.getVaultIdFromPayPalOrderResponseData(responseData))
                .build();
    }

    private PayPalPaymentData getPaymentData(PayPalOrderResponseData responseData) {
        return responseData.getPurchaseUnits().stream().filter(isDefault())
                .map(PurchaseUnitData::getPayments)
                .findFirst()
                .map(PayPalPaymentCollectionData::getAuthorizations)
                .orElseThrow(() -> new IllegalArgumentException(" Payments are empty "))
                .stream()
                .findFirst()
                .filter(isCreated())
                .orElseThrow(() -> new PayPalProcessPaymentException(PAYPAL_CHECKOUT_PAYMENT_ERROR_MSG));
    }

    private PayPalOrderProcessRequestData prepareRequestBody(SubscriptionAuthorizationRequest request) {
        if (request.getCv2() != null) {
            PayPalOrderProcessRequestData authorizationRequest = new PayPalOrderProcessRequestData();
            PaymentSourceData paymentSource = new PaymentSourceData();
            PayPalTokenData token = new PayPalTokenData();
            token.setId(request.getCv2());
            token.setType(BILLING_AGREEMENT);
            paymentSource.setToken(token);
            authorizationRequest.setPaymentSource(paymentSource);
            return authorizationRequest;
        }
        return null;
    }

    private static Predicate<PurchaseUnitData> isDefault() {
        return unit -> unit != null && DEFAULT.equals(unit.getReferenceId());
    }

    private static Predicate<PayPalPaymentData> isCreated() {
        return authorization -> authorization != null && CREATED_STATUS_RESULT.equals(authorization.getStatus());
    }

    public void setRestTemplate(RestTemplate restTemplate) {
        this.restTemplate = restTemplate;
    }

    public void setBaseStoreService(BaseStoreService baseStoreService) {
        this.baseStoreService = baseStoreService;
    }

}
