package com.paypal.hybris.core.commands.impl;

import com.paypal.http.HttpResponse;
import com.paypal.hybris.core.commands.PayPalAbstractCommand;
import com.paypal.hybris.core.exception.PayPalReauthorizeAdapterException;
import com.paypal.hybris.core.results.PayPalAuthorizationResult;
import com.paypal.hybris.core.util.PayPalCommandsUtil;
import com.paypal.hybris.core.util.builder.GenericBuilder;
import com.paypal.payments.Authorization;
import com.paypal.payments.AuthorizationsReauthorizeRequest;
import com.paypal.payments.Money;
import com.paypal.payments.ReauthorizeRequest;
import de.hybris.platform.payment.commands.SubscriptionAuthorizationCommand;
import de.hybris.platform.payment.commands.request.SubscriptionAuthorizationRequest;
import de.hybris.platform.payment.commands.result.AuthorizationResult;
import de.hybris.platform.payment.dto.TransactionStatus;
import de.hybris.platform.payment.dto.TransactionStatusDetails;
import org.apache.log4j.Logger;

import java.io.IOException;
import java.math.RoundingMode;
import java.util.Date;
import java.util.Optional;

import static com.paypal.hybris.core.constants.PaypalcoreConstants.CREATED_STATUS_RESULT;


public class DefaultPayPalReauthorizationRequestCommand extends PayPalAbstractCommand
    implements SubscriptionAuthorizationCommand {

    private static final Logger LOG = Logger.getLogger(DefaultPayPalReauthorizationRequestCommand.class);

    private static final String PAYPAL_DEBUG_ID = "Paypal-Debug-Id";
    private static final String SUBSCRIPTION_ID_EXCEPTION_MESSAGE = "Subscription ID is undefined, actual Subscription ID is: '%s'";

    @Override
    public AuthorizationResult perform(SubscriptionAuthorizationRequest request) {
        AuthorizationsReauthorizeRequest authorizationsReauthorizeRequest = getReauthorizeRequest(request);

        HttpResponse<Authorization> response = performAuthorizeRequest(authorizationsReauthorizeRequest);

        return translateResponse(response, request);
    }

    private AuthorizationsReauthorizeRequest getReauthorizeRequest(SubscriptionAuthorizationRequest request) {
        AuthorizationsReauthorizeRequest authorizationsReauthorizeRequest = Optional
                .ofNullable(request.getSubscriptionID())
                .map(AuthorizationsReauthorizeRequest::new).orElseThrow(()
                        -> new IllegalArgumentException(
                        SUBSCRIPTION_ID_EXCEPTION_MESSAGE.formatted(request.getSubscriptionID())));

        authorizationsReauthorizeRequest.requestBody(prepareRequestBody(request));
        return authorizationsReauthorizeRequest;
    }

    private HttpResponse<Authorization> performAuthorizeRequest(AuthorizationsReauthorizeRequest authorizationsReauthorizeRequest) {
        try {
            return createClient().execute(authorizationsReauthorizeRequest);
        } catch (IOException e) {
            LOG.error("Reauthorization failed: ", e);
            String errorDescription = getDescriptionFromPayPalErrorMessage(e.getMessage());
            throw new PayPalReauthorizeAdapterException(e, authorizationsReauthorizeRequest, errorDescription);
        }
    }

    private AuthorizationResult translateResponse(HttpResponse<Authorization> response,
                                                  SubscriptionAuthorizationRequest request) {
        Authorization authorizationResponse = response.result();
        final TransactionStatus transactionStatus =
                CREATED_STATUS_RESULT.equals(authorizationResponse.status()) ? TransactionStatus.ACCEPTED
                        : TransactionStatus.REJECTED;
        final TransactionStatusDetails transactionStatusDetails = CREATED_STATUS_RESULT.equals(authorizationResponse.status()) ?
                TransactionStatusDetails.SUCCESFULL : TransactionStatusDetails.PROCESSOR_DECLINE;

        return GenericBuilder.of(PayPalAuthorizationResult::new)
                .with(AuthorizationResult::setRequestId, request.getSubscriptionID())
                .with(AuthorizationResult::setAuthorizationCode, authorizationResponse.id())
                .with(AuthorizationResult::setPaymentProvider, request.getPaymentProvider())
                .with(AuthorizationResult::setCurrency, request.getCurrency())
                .with(AuthorizationResult::setTotalAmount, request.getTotalAmount())
                .with(AuthorizationResult::setReconciliationId, request.getSubscriptionID())
                .with(AuthorizationResult::setMerchantTransactionCode, request.getMerchantTransactionCode())
                .with(AuthorizationResult::setTransactionStatus, transactionStatus)
                .with(AuthorizationResult::setTransactionStatusDetails, transactionStatusDetails)
                .with(PayPalAuthorizationResult::setDebugId, response.headers().header(PAYPAL_DEBUG_ID))
                .with(PayPalAuthorizationResult::setResponseField, PayPalCommandsUtil.getValueAsString(response.result()))
                .with(PayPalAuthorizationResult::setRequestField, PayPalCommandsUtil.getValueAsString(request))
                .with(PayPalAuthorizationResult::setExpirationTime, getExpirationTime(response))
                .with(PayPalAuthorizationResult::setUpdateTime, getUpdateTime(response))
                .build();
    }

    private Date getUpdateTime(HttpResponse<Authorization> response) {
        return PayPalCommandsUtil.convertDate(response.result().updateTime());
    }

    private Date getExpirationTime(HttpResponse<Authorization> response) {
        return PayPalCommandsUtil.convertDate(response.result().expirationTime());
    }

    private static ReauthorizeRequest prepareRequestBody(SubscriptionAuthorizationRequest request) {
        ReauthorizeRequest reauthorizeRequest = new ReauthorizeRequest();
        Money money = new Money();
        money.currencyCode(request.getCurrency().getCurrencyCode());
        money.value(String.valueOf(request.getTotalAmount().setScale(2, RoundingMode.HALF_UP)));
        reauthorizeRequest.amount(money);
        return reauthorizeRequest;
    }

}
