package com.paypal.hybris.core.commands.impl;

import com.paypal.base.rest.JSONFormatter;
import com.paypal.core.PayPalEnvironment;
import com.paypal.hybris.core.commands.PayPalAbstractCommand;
import com.paypal.hybris.core.constants.PaypalcoreConstants;
import com.paypal.hybris.core.exception.PayPalProcessPaymentException;
import com.paypal.hybris.core.exception.PayPalVaultedHttpClientErrorException;
import com.paypal.hybris.core.results.PayPalVaultedPaymentResult;
import com.paypal.hybris.core.util.PayPalCommandsUtil;
import com.paypal.hybris.core.util.builder.GenericBuilder;
import com.paypal.hybris.data.CardData;
import com.paypal.hybris.data.PayPalData;
import com.paypal.hybris.data.PayPalHostedFieldsOrderRequestData;
import com.paypal.hybris.data.PayPalOrderProcessRequestData;
import com.paypal.hybris.data.PayPalOrderRequestData;
import com.paypal.hybris.data.PayPalOrderResponseData;
import com.paypal.hybris.data.PayPalPaymentData;
import com.paypal.hybris.data.PayPalVaultedPaymentProcessResultData;
import com.paypal.hybris.data.PaymentSourceData;
import com.paypal.hybris.data.PurchaseUnitData;
import com.paypal.hybris.data.StoredCredential;
import de.hybris.platform.commerceservices.util.GuidKeyGenerator;
import de.hybris.platform.payment.commands.Command;
import de.hybris.platform.payment.dto.TransactionStatus;
import de.hybris.platform.payment.dto.TransactionStatusDetails;
import org.apache.commons.lang3.StringUtils;
import org.apache.log4j.Logger;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.client.HttpClientErrorException;
import org.springframework.web.client.RestTemplate;

import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.Objects;
import java.util.Optional;
import java.util.function.Function;
import java.util.function.Predicate;
import java.util.stream.Stream;

import static com.paypal.hybris.core.constants.PaypalcoreConstants.CHECKOUT_URL;
import static com.paypal.hybris.core.constants.PaypalcoreConstants.COMPLETED_STATUS_RESULT;
import static com.paypal.hybris.core.constants.PaypalcoreConstants.CREATED_STATUS_RESULT;
import static com.paypal.hybris.core.constants.PaypalcoreConstants.PAYPAL_CHECKOUT_PAYMENT_ERROR_MSG;
import static com.paypal.hybris.core.constants.PaypalcoreConstants.PAYPAL_INTENT_AUTHORIZE;


public class DefaultPayPalProcessVaultedPaymentCommand extends PayPalAbstractCommand implements
        Command<PayPalOrderRequestData, PayPalVaultedPaymentProcessResultData> {

    private static final String TRANSACTION_CAPTURE_FAILURE_MESSAGE = "Transaction capture failed: ";
    private static final String PAYPAL_METADATA_ID = "PAYPAL-CLIENT-METADATA-ID";
    private static final String PAYPAL_REQUEST_ID_HEADER = "PayPal-Request-Id";
    private static final String CLIENT_METADATA_ID_MESSAGE = "[FraudNet] Client MetaData ID - '%s'";

    private static final Logger LOG = Logger.getLogger(DefaultPayPalProcessVaultedPaymentCommand.class);
    private RestTemplate restTemplate;
    private GuidKeyGenerator guidKeyGenerator;


    @Override
    public PayPalVaultedPaymentProcessResultData perform(PayPalOrderRequestData request) {

        final HttpHeaders headers = prepareHeaders();
        PayPalOrderProcessRequestData body = prepareRequestBody(request);
        final HttpEntity<PayPalOrderProcessRequestData> httpEntity = new HttpEntity<>(body, headers);

        final ResponseEntity<PayPalOrderResponseData> response = captureVaultedPayment(httpEntity);
        LOG.info(JSONFormatter.toJSON(httpEntity.getBody()));

        return translateResponse(response, request);
    }

    private HttpHeaders prepareHeaders() {
        final HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        headers.add(PaypalcoreConstants.AUTHORIZATION_HEADER, createAPIContext().getAccessToken());
        headers.add(PAYPAL_REQUEST_ID_HEADER, new SimpleDateFormat("yyyy.MM.dd.HH.mm.ss")
                .format(new java.util.Date()).concat(guidKeyGenerator.generate().toString()));

        if (getDefaultPayPalConfigurationService().isFraudNetEnabled()) {
            String correlationId = getCartService().getSessionCart().getPayPalClientMetadataId();
            headers.add(PAYPAL_METADATA_ID, correlationId);
            LOG.info(CLIENT_METADATA_ID_MESSAGE.formatted(correlationId));
        }
        return headers;
    }

    private ResponseEntity<PayPalOrderResponseData> captureVaultedPayment(HttpEntity<PayPalOrderProcessRequestData> httpEntity) {
        PayPalEnvironment payPalEnvironment = createPayPalEnvironment();
        String url = payPalEnvironment.baseUrl() + CHECKOUT_URL;
        try {
            return restTemplate.postForEntity(url, httpEntity, PayPalOrderResponseData.class);
        } catch (HttpClientErrorException e) {
            LOG.error(TRANSACTION_CAPTURE_FAILURE_MESSAGE, e);
            throw new PayPalVaultedHttpClientErrorException(e, httpEntity);
        } catch (Exception e) {
            LOG.error(TRANSACTION_CAPTURE_FAILURE_MESSAGE, e);
            throw new PayPalProcessPaymentException(PAYPAL_CHECKOUT_PAYMENT_ERROR_MSG);
        }
    }

    private PayPalVaultedPaymentResult translateResponse(ResponseEntity<PayPalOrderResponseData> response,
                                                         PayPalOrderRequestData request) {
        String resultStatus = Optional.ofNullable(response.getBody()).orElseThrow().getStatus();
        PayPalOrderResponseData captureOrderResponseData = response.getBody();
        final TransactionStatus transactionStatus = getTransactionStatusMap()
                .getOrDefault(resultStatus, TransactionStatus.ERROR);
        final TransactionStatusDetails transactionStatusDetails = getTransactionStatusDetailsMap().
                getOrDefault(resultStatus, TransactionStatusDetails.GENERAL_SYSTEM_ERROR);

        Function<PurchaseUnitData, Stream<PayPalPaymentData>> paymentRelatedIntent =
                PAYPAL_INTENT_AUTHORIZE.equals(request.getIntent()) ? unit -> unit.getPayments().getAuthorizations().stream() :
                        unit -> unit.getPayments().getCaptures().stream();

        Predicate<PayPalPaymentData> paymentStatusRelatedIntent = PAYPAL_INTENT_AUTHORIZE.equals(request.getIntent()) ?
                authorization -> CREATED_STATUS_RESULT.equals(authorization.getStatus()) :
                capture -> COMPLETED_STATUS_RESULT.equals(capture.getStatus());

        final String requestId = captureOrderResponseData.getPurchaseUnits().stream()
                .flatMap(paymentRelatedIntent)
                .findFirst()
                .filter(paymentStatusRelatedIntent)
                .map(PayPalPaymentData::getId)
                .orElseThrow(() -> new PayPalProcessPaymentException(PAYPAL_CHECKOUT_PAYMENT_ERROR_MSG));

        PayPalOrderResponseData responseData = response.getBody();
        return GenericBuilder.of(PayPalVaultedPaymentResult::new)
                .with(PayPalVaultedPaymentResult::setRequestId, requestId)
                .with(PayPalVaultedPaymentResult::setOrderId, captureOrderResponseData.getId())
                .with(PayPalVaultedPaymentResult::setTransactionStatus, transactionStatus)
                .with(PayPalVaultedPaymentResult::setTransactionStatusDetails, transactionStatusDetails)
                .with(PayPalVaultedPaymentResult::setCreateTime, extractCreateTime(responseData))
                .with(PayPalVaultedPaymentResult::setUpdateTime, extractUpdateTime(responseData))
                .with(PayPalVaultedPaymentResult::setExpirationTime, getExpirationTime(responseData.getPurchaseUnits().get(0)))
                .with(PayPalVaultedPaymentResult::setDebugId, getPaypalDebugId(response))
                .with(PayPalVaultedPaymentResult::setResponse, PayPalCommandsUtil.getValueAsString(response))
                .with(PayPalVaultedPaymentResult::setRequest, PayPalCommandsUtil.getValueAsString(request))
                .build();
    }

    private Date getExpirationTime(PurchaseUnitData purchaseUnit) {
        return Optional.ofNullable(purchaseUnit.getPayments().getAuthorizations())
                .flatMap(auths -> auths.stream()
                        .map(PayPalPaymentData::getExpirationTime)
                        .findFirst())
                .orElse(null);
    }

    protected String getPaypalDebugId(ResponseEntity<PayPalOrderResponseData> response) {
        return Objects.requireNonNull(response.getHeaders().get(PayPalAbstractCommand.PAYPAL_DEBUG_ID))
                .stream()
                .findFirst()
                .orElse(StringUtils.EMPTY);
    }

    protected ArrayList<PurchaseUnitData> getPurchaseUnitData(PayPalOrderRequestData data) {
        PurchaseUnitData purchaseUnitData = GenericBuilder.of(PurchaseUnitData::new)
                .with(PurchaseUnitData::setAmount, data.getBreakdownAmountData())
                .with(PurchaseUnitData::setItems, data.getOrderItems())
                .with(PurchaseUnitData::setSupplementaryData, data.getSupplementaryData())
                .build();

        ArrayList<PurchaseUnitData> purchaseUnitDataArrayList = new ArrayList<>();
        purchaseUnitDataArrayList.add(purchaseUnitData);
        return purchaseUnitDataArrayList;
    }

    private PayPalOrderProcessRequestData prepareRequestBody(PayPalOrderRequestData request) {
        ArrayList<PurchaseUnitData> purchaseUnits = getPurchaseUnitData(request);

        PaymentSourceData paymentSource = new PaymentSourceData();
        if (request instanceof PayPalHostedFieldsOrderRequestData requestData) {

            StoredCredential storedCredential = Optional.ofNullable(requestData.getStoredCredential())
                    .map(x -> new GenericBuilder<>(StoredCredential::new)
                            .with(StoredCredential::setPaymentInitiator, x.getPaymentInitiator())
                            .with(StoredCredential::setPaymentType, x.getPaymentType())
                            .with(StoredCredential::setUsage, x.getUsage())
                            .build())
                    .orElse(null);

            paymentSource.setCard(GenericBuilder.of(CardData::new)
                    .with(CardData::setVaultId, request.getVaultId())
                    .with(CardData::setStoredCredential, storedCredential)
                    .build());
        } else {
            paymentSource.setPaypal(GenericBuilder.of(PayPalData::new)
                    .with(PayPalData::setVaultId, request.getVaultId())
                    .build());
        }


        return GenericBuilder.of(PayPalOrderProcessRequestData::new)
                .with(PayPalOrderProcessRequestData::setIntent, request.getIntent().toUpperCase())
                .with(PayPalOrderProcessRequestData::setPaymentSource, paymentSource)
                .with(PayPalOrderProcessRequestData::setPurchaseUnits, purchaseUnits)
                .build();
    }

    public void setRestTemplate(RestTemplate restTemplate) {
        this.restTemplate = restTemplate;
    }

    public void setGuidKeyGenerator(GuidKeyGenerator guidKeyGenerator) {
        this.guidKeyGenerator = guidKeyGenerator;
    }

}
