package com.paypal.hybris.core.commands.impl;

import com.paypal.core.PayPalHttpClient;
import com.paypal.http.HttpResponse;
import com.paypal.hybris.core.commands.PayPalAbstractCommand;
import com.paypal.hybris.core.exception.PayPalCaptureAdapterException;
import com.paypal.hybris.core.results.PayPalCaptureResult;
import com.paypal.hybris.core.util.PayPalCommandsUtil;
import com.paypal.hybris.core.util.builder.GenericBuilder;
import com.paypal.orders.OrderCaptureRequest;
import com.paypal.orders.PaymentSource;
import com.paypal.orders.Token;
import com.paypal.payments.AuthorizationsCaptureRequest;
import com.paypal.payments.Capture;
import com.paypal.payments.Money;
import de.hybris.platform.payment.commands.CaptureCommand;
import de.hybris.platform.payment.commands.request.CaptureRequest;
import de.hybris.platform.payment.commands.result.CaptureResult;
import de.hybris.platform.payment.dto.TransactionStatus;
import de.hybris.platform.payment.dto.TransactionStatusDetails;
import de.hybris.platform.servicelayer.user.UserService;
import org.apache.log4j.Logger;

import java.io.IOException;
import java.math.RoundingMode;
import java.util.Optional;

import static com.paypal.hybris.core.constants.PaypalcoreConstants.BILLING_AGREEMENT;


public class DefaultPayPalOnDemandCaptureCommand extends PayPalAbstractCommand implements CaptureCommand {

	private static final String TRANSACTION_CAPTURE_FAILURE_MESSAGE = "Transaction capture failed: ";
	private static final String REQUEST_ID_EXCEPTION_MESSAGE = "Request ID is undefined, actual Request ID is: '%s'";
	private static final String PAYPAL_DEBUG_ID = "Paypal-Debug-Id";

	private static final Logger LOG = Logger.getLogger(DefaultPayPalOnDemandCaptureCommand.class);

	private UserService userService;

	@Override
	public CaptureResult perform(CaptureRequest request) {
		AuthorizationsCaptureRequest authorizationsCaptureRequest = getAuthorizationsCaptureRequest(request);

		HttpResponse<Capture> response = performCaptureRequest(authorizationsCaptureRequest);

		return translateResponse(response, request);
	}

	private HttpResponse<Capture> performCaptureRequest(AuthorizationsCaptureRequest authorizationsCaptureRequest) {
		try {
			return createClient().execute(authorizationsCaptureRequest);
		} catch (IOException e) {
			LOG.error(TRANSACTION_CAPTURE_FAILURE_MESSAGE, e);
			throw new PayPalCaptureAdapterException(e, authorizationsCaptureRequest);
		}
	}

	private AuthorizationsCaptureRequest getAuthorizationsCaptureRequest(CaptureRequest request) {
		AuthorizationsCaptureRequest authorizationsCaptureRequest = Optional.ofNullable(request.getRequestId())
				.map(AuthorizationsCaptureRequest::new).orElseThrow(()
						-> new IllegalArgumentException(
						REQUEST_ID_EXCEPTION_MESSAGE.formatted(request.getRequestId())));
		if (request.getRequestToken() != null) {
			authorizationsCaptureRequest.requestBody(prepareRequestBody(request));
		} else {
			authorizationsCaptureRequest.requestBody(preparePartialRequestBody(request));
		}
		return authorizationsCaptureRequest;
	}

	@Override
	protected PayPalHttpClient createClient() {
		return super.createClient();
	}

	private CaptureResult translateResponse(HttpResponse<Capture> response, CaptureRequest request) {
		Capture capture = response.result();
		String resultStatus = capture.status();
		final TransactionStatus transactionStatus = getTransactionStatusMap()
				.getOrDefault(resultStatus, TransactionStatus.ERROR);
		final TransactionStatusDetails transactionStatusDetails = getTransactionStatusDetailsMap().
				getOrDefault(resultStatus, TransactionStatusDetails.GENERAL_SYSTEM_ERROR);

		return GenericBuilder.of(PayPalCaptureResult::new)
				.with(CaptureResult::setRequestId, capture.id())
				.with(CaptureResult::setCurrency, request.getCurrency())
				.with(CaptureResult::setTotalAmount, request.getTotalAmount())
				.with(CaptureResult::setReconciliationId, request.getSubscriptionID())
				.with(CaptureResult::setMerchantTransactionCode, request.getMerchantTransactionCode())
				.with(CaptureResult::setTransactionStatus, transactionStatus)
				.with(CaptureResult::setTransactionStatusDetails, transactionStatusDetails)
				.with(PayPalCaptureResult::setDebugId, response.headers().header(PAYPAL_DEBUG_ID))
				.with(PayPalCaptureResult::setCreateTime, PayPalCommandsUtil.convertDate(capture.createTime()))
				.with(PayPalCaptureResult::setUpdateTime, PayPalCommandsUtil.convertDate(capture.updateTime()))
				.with(PayPalCaptureResult::setRequestField, PayPalCommandsUtil.getValueAsString(request))
				.with(PayPalCaptureResult::setResponseField, PayPalCommandsUtil.getValueAsString(response.result()))
				.build();
	}

	protected OrderCaptureRequest prepareRequestBody(CaptureRequest request) {
		OrderCaptureRequest orderCaptureRequest = new OrderCaptureRequest();
		PaymentSource paymentSource = new PaymentSource();
		Token token = new Token();
		token.id(request.getRequestToken());
		token.type(BILLING_AGREEMENT);
		paymentSource.token(token);
		orderCaptureRequest.paymentSource(paymentSource);
		return orderCaptureRequest;
	}

	protected com.paypal.payments.CaptureRequest preparePartialRequestBody(CaptureRequest request) {
		com.paypal.payments.CaptureRequest captureRequest = new com.paypal.payments.CaptureRequest();
		Money money = new Money();
		money.currencyCode(request.getCurrency().getCurrencyCode());
		money.value(String.valueOf(request.getTotalAmount().setScale(2, RoundingMode.HALF_UP)));
		captureRequest.amount(money);
		return captureRequest;
	}

	public UserService getUserService() {
		return userService;
	}

	public void setUserService(UserService userService) {
		this.userService = userService;
	}
}
