package com.paypal.hybris.core.commands.impl;

import com.paypal.http.HttpResponse;
import com.paypal.hybris.core.commands.PayPalAbstractCommand;
import com.paypal.hybris.core.exception.PayPalRefundAdapterException;
import com.paypal.hybris.core.results.PayPalRefundResult;
import com.paypal.hybris.core.util.PayPalCommandsUtil;
import com.paypal.hybris.core.util.builder.GenericBuilder;
import com.paypal.payments.CapturesRefundRequest;
import com.paypal.payments.Money;
import com.paypal.payments.Refund;
import com.paypal.payments.RefundRequest;
import de.hybris.platform.payment.commands.FollowOnRefundCommand;
import de.hybris.platform.payment.commands.request.FollowOnRefundRequest;
import de.hybris.platform.payment.commands.result.RefundResult;
import de.hybris.platform.payment.dto.TransactionStatus;
import de.hybris.platform.payment.dto.TransactionStatusDetails;
import org.apache.log4j.Logger;

import java.io.IOException;
import java.math.BigDecimal;
import java.math.RoundingMode;
import java.util.Optional;


public class DefaultPayPalFollowOnRefundCommand extends PayPalAbstractCommand implements
    FollowOnRefundCommand<FollowOnRefundRequest> {

    private static final String TRANSACTION_REFUND_FAILURE_MESSAGE = "Transaction refund failed: ";
    private static final String TRANSACTION_REFUND_PREFER_HEADER = "return=representation";
    private static final String REQUEST_ID_EXCEPTION_MESSAGE = "Request ID is undefined, actual Request ID is: '%s'";
    private static final String PAYPAL_DEBUG_ID = "Paypal-Debug-Id";

    private static final Logger LOG = Logger.getLogger(DefaultPayPalFollowOnRefundCommand.class);

    @Override
    public RefundResult perform(FollowOnRefundRequest request) {
        CapturesRefundRequest capturesRefundRequest = getCaptureRefundRequest(request);

        HttpResponse<Refund> response = performRefundRequest(capturesRefundRequest);

        return translateResponse(response, request);
    }

    private CapturesRefundRequest getCaptureRefundRequest(FollowOnRefundRequest request) {
        CapturesRefundRequest capturesRefundRequest = Optional.ofNullable(request)
                .map(refundRequest -> new CapturesRefundRequest(request.getRequestId())).orElseThrow(()
                        -> new IllegalArgumentException(
                        REQUEST_ID_EXCEPTION_MESSAGE.formatted(request)));
        capturesRefundRequest.requestBody(prepareRequestBody(Optional.of(request)
                .orElseThrow(IllegalArgumentException::new)));
        capturesRefundRequest.prefer(TRANSACTION_REFUND_PREFER_HEADER);
        return capturesRefundRequest;
    }

    private HttpResponse<Refund> performRefundRequest(CapturesRefundRequest capturesRefundRequest) {
        try {
            return createClient().execute(capturesRefundRequest);
        } catch (IOException e) {
            LOG.error(TRANSACTION_REFUND_FAILURE_MESSAGE, e);
            throw new PayPalRefundAdapterException(e, capturesRefundRequest);
        }
    }

    private RefundResult translateResponse(HttpResponse<Refund> response, FollowOnRefundRequest request) {
        String resultStatus = response.result().status();
        final TransactionStatus transactionStatus = getTransactionStatusMap()
            .getOrDefault(resultStatus, TransactionStatus.ERROR);
        final TransactionStatusDetails transactionStatusDetails = getTransactionStatusDetailsMap().
            getOrDefault(resultStatus, TransactionStatusDetails.GENERAL_SYSTEM_ERROR);
        BigDecimal refundedAmount = request.getTotalAmount();

        Refund refund = response.result();
        if (refund.amount() != null){
            refundedAmount = new BigDecimal(refund.amount().value());
        }

        return GenericBuilder.of(PayPalRefundResult::new)
            .with(RefundResult::setRequestId, refund.id())
            .with(RefundResult::setCurrency, request.getCurrency())
            .with(RefundResult::setTotalAmount, refundedAmount)
            .with(RefundResult::setMerchantTransactionCode, request.getMerchantTransactionCode())
            .with(RefundResult::setTransactionStatus, transactionStatus)
            .with(RefundResult::setTransactionStatusDetails, transactionStatusDetails)
            .with(PayPalRefundResult::setDebugId, response.headers().header(PAYPAL_DEBUG_ID))
            .with(PayPalRefundResult::setRequestField, PayPalCommandsUtil.getValueAsString(request))
            .with(PayPalRefundResult::setResponseField, PayPalCommandsUtil.getValueAsString(response.result()))
            .with(PayPalRefundResult::setCreateTime, PayPalCommandsUtil.convertDate(response.result().createTime()))
            .with(PayPalRefundResult::setUpdateTime, PayPalCommandsUtil.convertDate(response.result().updateTime()))
            .build();
    }

    private static RefundRequest prepareRequestBody(FollowOnRefundRequest request) {
        RefundRequest refundRequest = new RefundRequest();
        Money amount = new Money();
        amount.value(String.valueOf(request.getTotalAmount().setScale(2, RoundingMode.HALF_UP)));
        amount.currencyCode(request.getCurrency().getCurrencyCode());
        refundRequest.amount(amount);
        return refundRequest;
    }

}
