package com.paypal.hybris.core.commands.impl;

import com.paypal.base.rest.JSONFormatter;
import com.paypal.core.PayPalEnvironment;
import com.paypal.hybris.core.commands.PayPalAbstractCommand;
import com.paypal.hybris.core.constants.PaypalcoreConstants;
import com.paypal.hybris.core.util.builder.GenericBuilder;
import com.paypal.hybris.data.AddressPortableData;
import com.paypal.hybris.data.ApplePayData;
import com.paypal.hybris.data.ApplePayOrderRequestData;
import com.paypal.hybris.data.CardData;
import com.paypal.hybris.data.ExperienceContextData;
import com.paypal.hybris.data.NameData;
import com.paypal.hybris.data.PayPalAddressDetailsData;
import com.paypal.hybris.data.PayPalApplicationContextData;
import com.paypal.hybris.data.PayPalAttributesData;
import com.paypal.hybris.data.PayPalCreateOrderRequestData;
import com.paypal.hybris.data.PayPalCustomerData;
import com.paypal.hybris.data.PayPalData;
import com.paypal.hybris.data.PayPalHostedFieldsOrderRequestData;
import com.paypal.hybris.data.PayPalOrderRequestData;
import com.paypal.hybris.data.PayPalOrderResponseData;
import com.paypal.hybris.data.PayPalVaultData;
import com.paypal.hybris.data.PayPalVaultOrderRequestData;
import com.paypal.hybris.data.PaymentSourceData;
import com.paypal.hybris.data.PurchaseUnitData;
import com.paypal.hybris.data.ShippingDetailData;
import com.paypal.hybris.data.StoredCredential;
import com.paypal.hybris.data.StoredCredentialData;
import de.hybris.platform.payment.AdapterException;
import de.hybris.platform.payment.commands.Command;
import org.apache.log4j.Logger;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.client.RestTemplate;

import java.util.ArrayList;
import java.util.Locale;
import java.util.Objects;

import static com.paypal.hybris.core.constants.PaypalcoreConstants.CHECKOUT_URL;


public class DefaultPayPalCreateOrderCommand extends PayPalAbstractCommand
        implements Command<PayPalOrderRequestData, String> {

    private static final String CREATING_ORDER_ERROR_MESSAGE = "ERROR WHILE CREATING ORDER\nException: ";
    private static final String SET_PROVIDED_ADDRESS = "SET_PROVIDED_ADDRESS";
    private static final String NO_SHIPPING = "NO_SHIPPING";
    private static final String ORDER_SAVED_EXPLICITLY = "ORDER_SAVED_EXPLICITLY";
    private static final String PAYPAL_PARTNER_ATTRIBUTION_HEADER = "PayPal-Partner-Attribution-Id";
    private static final String SHIPPING_TYPE_SHIPPING = "SHIPPING";
    private static final Logger LOG = Logger.getLogger(DefaultPayPalCreateOrderCommand.class);
    private static final String RETURN_URL = "https://example.com/returnUrl";
    private static final String CANCEL_URL = "https://example.com/cancelUrl";
    private static final String CREATE_ORDER_REQUEST = "Create order request body: \n";
    private RestTemplate restTemplate;

    @Override
    public String perform(final PayPalOrderRequestData data) {
        final PayPalEnvironment payPalEnvironment = createPayPalEnvironment();
        final HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        headers.add(PaypalcoreConstants.AUTHORIZATION_HEADER, createAPIContext().getAccessToken());
        headers.add(PAYPAL_PARTNER_ATTRIBUTION_HEADER,
                getDefaultPayPalConfigurationService().getPayPalPartnerAttributionId());
        final HttpEntity<PayPalCreateOrderRequestData> httpEntity = new HttpEntity<>(createOrderRequestBody(data), headers);
        final String url = payPalEnvironment.baseUrl() + CHECKOUT_URL;
        final ResponseEntity<PayPalOrderResponseData> response;
        LOG.info(CREATE_ORDER_REQUEST.concat(JSONFormatter.toJSON(httpEntity.getBody())));
        try {
            response = restTemplate.postForEntity(url, httpEntity, PayPalOrderResponseData.class);
        } catch (Exception e) {
            LOG.error(CREATING_ORDER_ERROR_MESSAGE + e);
            throw new AdapterException();
        }
        return Objects.requireNonNull(response.getBody()).getId();
    }

    private PayPalCreateOrderRequestData createOrderRequestBody(final PayPalOrderRequestData data) {
        PayPalCreateOrderRequestData requestBody = new PayPalCreateOrderRequestData();

        requestBody.setIntent(data.getIntent().toUpperCase(Locale.US));
        requestBody.setPurchaseUnits(getPurchaseUnitData(data));


        if (data instanceof PayPalHostedFieldsOrderRequestData hostedFieldsOrderRequestData) {
            if (hostedFieldsOrderRequestData.getHostedFieldsData() != null) {
                prepareHostedFieldsRequestBody(requestBody, hostedFieldsOrderRequestData);
            } else if (hostedFieldsOrderRequestData.getStoredCredential() != null) {
                setStoredCredential(hostedFieldsOrderRequestData.getStoredCredential(), requestBody);
            }
            requestBody.setApplicationContext(GenericBuilder.of(PayPalApplicationContextData::new)
                    .with(PayPalApplicationContextData::setShippingPreference, getShippingPreference(data))
                    .build());
        } else if (data instanceof ApplePayOrderRequestData applePayOrderRequestData) {
            prepareApplePayRequestBody(requestBody, applePayOrderRequestData);
            requestBody.setApplicationContext(GenericBuilder.of(PayPalApplicationContextData::new)
                    .with(PayPalApplicationContextData::setShippingPreference, getShippingPreference(data))
                    .build());
        } else if (data instanceof PayPalVaultOrderRequestData){
            preparePayPalVaultRequestBody(requestBody, data);
        } else {
            requestBody.setApplicationContext(GenericBuilder.of(PayPalApplicationContextData::new)
                    .with(PayPalApplicationContextData::setShippingPreference, getShippingPreference(data))
                    .build());
            if (data.isSaveOrderFlowActive()) {
                requestBody.setProcessingInstruction(ORDER_SAVED_EXPLICITLY);
            }
        }

        return requestBody;
    }

    private void preparePayPalVaultRequestBody(final PayPalCreateOrderRequestData requestBody,
                                               final PayPalOrderRequestData data) {
        if (data.isSaveOrderFlowActive()) {
            requestBody.setProcessingInstruction(ORDER_SAVED_EXPLICITLY);
        }

        PayPalAttributesData attributesData = getPayPalAttributesData(data, data.isSavePaymentMethod());

        ExperienceContextData experienceContext = GenericBuilder.of(ExperienceContextData::new)
                .with(ExperienceContextData::setShippingPreference, getShippingPreference(data))
                .with(ExperienceContextData::setReturnUrl, RETURN_URL)
                .with(ExperienceContextData::setCancelUrl, CANCEL_URL)
                .build();

        PayPalData payPalData = GenericBuilder.of(PayPalData::new)
                .with(PayPalData::setVaultId, data.getVaultId())
                .with(PayPalData::setExperienceContext, experienceContext)
                .with(PayPalData::setAttributes, attributesData)
                .build();

        PaymentSourceData paymentSource = GenericBuilder.of(PaymentSourceData::new)
                .with(PaymentSourceData::setPaypal, payPalData)
                .build();

        requestBody.setPaymentSource(paymentSource);
    }

    private ArrayList<PurchaseUnitData> getPurchaseUnitData(PayPalOrderRequestData data) {
        PurchaseUnitData purchaseUnitData = GenericBuilder.of(PurchaseUnitData::new)
                .with(PurchaseUnitData::setAmount, data.getBreakdownAmountData())
                .with(PurchaseUnitData::setShipping, createShippingAddressRequest(data))
                .with(PurchaseUnitData::setItems, data.getOrderItems())
                .with(PurchaseUnitData::setSupplementaryData, data.getSupplementaryData())
                .build();

        ArrayList<PurchaseUnitData> purchaseUnitDataArrayList = new ArrayList<>();
        purchaseUnitDataArrayList.add(purchaseUnitData);
        return purchaseUnitDataArrayList;
    }

    private static void prepareHostedFieldsRequestBody(final PayPalCreateOrderRequestData requestBody,
                                                final PayPalHostedFieldsOrderRequestData data) {
        PayPalAttributesData attributesData = getPayPalAttributesData(data,
                data.getHostedFieldsData() != null && data.getHostedFieldsData().getIsShouldBeSaved());

        StoredCredential storedCredential = null;

        if (data.getStoredCredential() != null) {
            storedCredential = new GenericBuilder<>(StoredCredential::new)
                    .with(StoredCredential::setPaymentInitiator, data.getStoredCredential().getPaymentInitiator())
                    .with(StoredCredential::setPaymentType, data.getStoredCredential().getPaymentType())
                    .with(StoredCredential::setUsage, data.getStoredCredential().getUsage())
                    .build();
        }
        CardData cardData = GenericBuilder.of(CardData::new)
                .with(CardData::setVaultId, data.getVaultId())
                .with(CardData::setName, data.getHostedFieldsData().getNameOnCard())
                .with(CardData::setStoredCredential, storedCredential)
                .with(CardData::setBillingAddress,
                        createAddressPortableData(data.getHostedFieldsData().getBillingAddress()))
                .with(CardData::setAttributes, attributesData)
                .build();

        PaymentSourceData paymentSource = GenericBuilder.of(PaymentSourceData::new)
                .with(PaymentSourceData::setCard, cardData)
                .build();

        requestBody.setPaymentSource(paymentSource);

    }

    private static void prepareApplePayRequestBody(final PayPalCreateOrderRequestData requestBody, final ApplePayOrderRequestData data) {
        StoredCredential storedCredential = null;

        if (data.getStoredCredential() != null) {
            storedCredential = new GenericBuilder<>(StoredCredential::new)
                    .with(StoredCredential::setPaymentInitiator, data.getStoredCredential().getPaymentInitiator())
                    .with(StoredCredential::setPaymentType, data.getStoredCredential().getPaymentType())
                    .with(StoredCredential::setUsage, data.getStoredCredential().getUsage())
                    .build();
        }

        ApplePayData applePayData = GenericBuilder.of(ApplePayData::new)
                .with(ApplePayData::setName, data.getName())
                .with(ApplePayData::setEmailAddress, data.getEmailAddress())
                .with(ApplePayData::setPhoneNumber, data.getPhoneNumber())
                .with(ApplePayData::setStoredCredential, storedCredential)
                .build();

        PaymentSourceData paymentSource = GenericBuilder.of(PaymentSourceData::new)
                .with(PaymentSourceData::setApplepay, applePayData)
                .build();

        requestBody.setPaymentSource(paymentSource);
    }

    private static PayPalAttributesData getPayPalAttributesData(PayPalOrderRequestData data, boolean shouldStoreInVault) {
        PayPalVaultData vaultData = GenericBuilder.of(PayPalVaultData::new)
                .with(PayPalVaultData::setStoreInVault, "ON_SUCCESS")
                .build();

        if (data instanceof PayPalVaultOrderRequestData payPalVaultOrderRequestData) {
            vaultData.setUsageType(payPalVaultOrderRequestData.getUsageType());
            vaultData.setCustomerType(payPalVaultOrderRequestData.getCustomerType());
        }
        PayPalAttributesData attributesData = new PayPalAttributesData();
        if (shouldStoreInVault) {
            attributesData.setVault(vaultData);
        }

        if (data.getCustomerId() != null) {
            PayPalCustomerData customerData = GenericBuilder.of(PayPalCustomerData::new)
                    .with(PayPalCustomerData::setId, data.getCustomerId())
                    .build();
            attributesData.setCustomer(customerData);
        }

        return attributesData;
    }

    private static String getShippingPreference(final PayPalOrderRequestData data) {
        if (data.getShippingAddress() != null) {
            return SET_PROVIDED_ADDRESS;
        }
        if (data.getShippingPreference() != null) {
            return data.getShippingPreference();
        }
        return NO_SHIPPING;
    }

    private static void setStoredCredential(final StoredCredentialData data,
                                     final PayPalCreateOrderRequestData requestBody) {
        StoredCredential storedCredential = GenericBuilder.of(StoredCredential::new)
                .with(StoredCredential::setPaymentInitiator, data.getPaymentInitiator())
                .with(StoredCredential::setPaymentType, data.getPaymentType())
                .with(StoredCredential::setUsage, data.getUsage())
                .build();

        CardData cardData = GenericBuilder.of(CardData::new)
                .with(CardData::setStoredCredential, storedCredential)
                .build();
        PaymentSourceData paymentSourceData = GenericBuilder.of(PaymentSourceData::new)
                .with(PaymentSourceData::setCard, cardData)
                .build();
        requestBody.setPaymentSource(paymentSourceData);
    }

    private ShippingDetailData createShippingAddressRequest(PayPalOrderRequestData data) {
        PayPalAddressDetailsData addressData = data.getShippingAddress();
        ShippingDetailData shippingDetailData = new ShippingDetailData();

        if (data.isPickUpInStore()) {
            return null;
        } else if (addressData != null) {
            NameData name = GenericBuilder.of(NameData::new)
                    .with(NameData::setFullName, addressData.getFirstName() + " " + addressData.getLastName())
                    .build();
            AddressPortableData addressPortableData = createAddressPortableData(addressData);

            shippingDetailData.setAddress(addressPortableData);
            shippingDetailData.setName(name);
            shippingDetailData.setType(SHIPPING_TYPE_SHIPPING);
        } else {
            shippingDetailData.setType(SHIPPING_TYPE_SHIPPING);
        }
        return shippingDetailData;
    }

    private static AddressPortableData createAddressPortableData(PayPalAddressDetailsData addressData) {
        return GenericBuilder.of(AddressPortableData::new)
                .with(AddressPortableData::setAddressLine1, addressData.getLine1())
                .with(AddressPortableData::setAddressLine2, addressData.getLine2())
                .with(AddressPortableData::setAdminArea1, addressData.getRegion())
                .with(AddressPortableData::setAdminArea2, addressData.getCity())
                .with(AddressPortableData::setCountryCode, addressData.getCountryCode())
                .with(AddressPortableData::setPostalCode, addressData.getPostalCode())
                .build();
    }

    public void setRestTemplate(RestTemplate restTemplate) {
        this.restTemplate = restTemplate;
    }

}
