/*

 */
package com.paypal.hybris.core.commands.impl;

import com.paypal.hybris.core.commands.PayPalAbstractCommand;
import com.paypal.hybris.core.constants.PaypalcoreConstants;
import com.paypal.hybris.data.PayPalConnectAccessTokenData;
import com.paypal.hybris.data.PayPalConnectAccessTokenRequest;
import de.hybris.platform.core.model.user.CustomerModel;
import de.hybris.platform.payment.AdapterException;
import de.hybris.platform.payment.commands.Command;
import java.util.Base64;
import java.util.Optional;

import de.hybris.platform.servicelayer.session.SessionService;
import org.apache.log4j.Logger;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.util.LinkedMultiValueMap;
import org.springframework.util.MultiValueMap;
import org.springframework.web.client.RestTemplate;

public class DefaultPayPalConnectExchangeCodeToAccessTokenCommand extends PayPalAbstractCommand
    implements Command<PayPalConnectAccessTokenRequest, PayPalConnectAccessTokenData> {

    private static final Logger LOG = Logger.getLogger(DefaultPayPalConnectExchangeCodeToAccessTokenCommand.class);

    private static final String EXCHANGE_ENDPOINT_URL = "oauth2/token";
    private static final String AUTHORIZATION_HEADER_PREFIX = "Basic ";
    private static final String GRANT_TYPE = "grant_type";
    private static final String GRANT_TYPE_VALUE = "authorization_code";
    private static final String CLIENT_CREDENTIALS = "client_credentials";
    private static final String CODE = "code";
    private static final String RESPONSE_TYPE = "response_type";
    private static final String ID_TOKEN = "id_token";
    private static final String ERROR_MESSAGE = "Error during try exchange authorization token to access token";
    private static final String TARGET_CUSTOMER_ID = "target_customer_id";
    private static final String USER = "user";

    private SessionService sessionService;
    private RestTemplate restTemplate;

    @Override
    public PayPalConnectAccessTokenData perform(PayPalConnectAccessTokenRequest payPalConnectAccessTokenRequest) {
        MultiValueMap<String, String> map = new LinkedMultiValueMap<>();
        Optional<CustomerModel> currentCustomer = Optional.ofNullable(sessionService.getAttribute(USER));
        HttpHeaders headers = new HttpHeaders();
        Optional.ofNullable(payPalConnectAccessTokenRequest.getAuthorizationCode())
                .ifPresentOrElse(code -> {
                    map.add(CODE, code);
                    map.add(GRANT_TYPE, GRANT_TYPE_VALUE);
                }, () -> {
                    map.add(GRANT_TYPE, CLIENT_CREDENTIALS);
                    map.add(RESPONSE_TYPE, ID_TOKEN);
                });
        if (payPalConnectAccessTokenRequest.isChangePaymentButtonAvailable()) {
            currentCustomer.map(CustomerModel::getVaultCustomerId).ifPresent(customerVaultId -> map.add(TARGET_CUSTOMER_ID, customerVaultId));
        }
        String endpointUrl = getPayPalApiEndpoint().concat(EXCHANGE_ENDPOINT_URL);
        headers.setContentType(MediaType.APPLICATION_FORM_URLENCODED);
        headers.add(PaypalcoreConstants.AUTHORIZATION_HEADER, AUTHORIZATION_HEADER_PREFIX + getAuthorizationHeaderValue());
        HttpEntity<MultiValueMap<String, String>> httpEntity = new HttpEntity<>(map, headers);
        ResponseEntity<PayPalConnectAccessTokenData> response;
        try {
            response = restTemplate
                .postForEntity(endpointUrl, httpEntity, PayPalConnectAccessTokenData.class);
        } catch (Exception e) {
            LOG.error(ERROR_MESSAGE, e);
            throw new AdapterException(e.getMessage());
        }
        return response.getBody();
    }

    private String getAuthorizationHeaderValue() {
        String clientIdAndSecret = getDefaultPayPalConfigurationService().getClientID() + ":"
            + getDefaultPayPalConfigurationService().getSecretKey();
        return new String(Base64.getEncoder().encode(clientIdAndSecret.getBytes()));
    }

    public void setSessionService(SessionService sessionService) {
        this.sessionService = sessionService;
    }

    public void setRestTemplate(RestTemplate restTemplate) {
        this.restTemplate = restTemplate;
    }
}
