package com.paypal.hybris.core.commands.impl;

import com.paypal.hybris.core.commands.PayPalAbstractCommand;
import com.paypal.hybris.core.constants.PaypalcoreConstants;
import com.paypal.hybris.core.exception.PayPalHttpClientErrorException;
import com.paypal.hybris.core.exception.PayPalProcessPaymentException;
import com.paypal.hybris.core.results.PayPalCaptureResult;
import com.paypal.hybris.core.util.PayPalCommandsUtil;
import com.paypal.hybris.core.util.builder.GenericBuilder;
import com.paypal.hybris.data.PayPalOrderProcessRequestData;
import com.paypal.hybris.data.PayPalOrderResponseData;
import com.paypal.hybris.data.PayPalPaymentData;
import de.hybris.platform.payment.commands.CaptureCommand;
import de.hybris.platform.payment.commands.request.CaptureRequest;
import de.hybris.platform.payment.commands.result.CaptureResult;
import de.hybris.platform.payment.dto.TransactionStatus;
import de.hybris.platform.payment.dto.TransactionStatusDetails;
import org.apache.commons.lang3.StringUtils;
import org.apache.log4j.Logger;
import org.springframework.http.HttpEntity;
import org.springframework.http.HttpHeaders;
import org.springframework.http.MediaType;
import org.springframework.http.ResponseEntity;
import org.springframework.web.client.HttpClientErrorException;
import org.springframework.web.client.RestTemplate;

import java.util.Date;

import static com.paypal.hybris.core.constants.PaypalcoreConstants.CHECKOUT_URL;
import static com.paypal.hybris.core.constants.PaypalcoreConstants.COMPLETED_STATUS_RESULT;
import static com.paypal.hybris.core.constants.PaypalcoreConstants.PAYPAL_CHECKOUT_PAYMENT_ERROR_MSG;


public class DefaultPayPalCaptureIntentCommand extends PayPalAbstractCommand implements CaptureCommand {

    private static final String PAYPAL_METADATA_ID = "PAYPAL-CLIENT-METADATA-ID";
    private static final String PAYPAL_DEBUG_ID = "Paypal-Debug-Id";
    private static final String CORRELATION_ID_MESSAGE = "[FraudNet] Client MetaData ID - '%s" ;
    private static final String CAPTURE_SUCCESSFUL_MESSAGE = "Capture completed successfully: %s";

    private static final String TRANSACTION_CAPTURE_FAILURE_MESSAGE = "Transaction capture failed: ";

    private static final Logger LOG = Logger.getLogger(DefaultPayPalCaptureIntentCommand.class);

    private static final String CAPTURE = "/capture";

    private RestTemplate restTemplate;

    @Override
    public CaptureResult perform(CaptureRequest request) {
        final HttpHeaders headers = prepareHeaders();

        final ResponseEntity<PayPalOrderResponseData> response = performCaptureRequest(headers, request);

        return translateResponse(response, request);
    }

    private HttpHeaders prepareHeaders() {
        final HttpHeaders headers = new HttpHeaders();
        headers.setContentType(MediaType.APPLICATION_JSON);
        headers.add(PaypalcoreConstants.AUTHORIZATION_HEADER, createAPIContext().getAccessToken());

        if (getDefaultPayPalConfigurationService().isFraudNetEnabled()) {
            String correlationId = getCartService().getSessionCart().getPayPalClientMetadataId();
            headers.add(PAYPAL_METADATA_ID, correlationId);
            LOG.info(CORRELATION_ID_MESSAGE.formatted(correlationId));
        }
        return headers;
    }

    protected ResponseEntity<PayPalOrderResponseData> performCaptureRequest(HttpHeaders headers, CaptureRequest request) {
        final String captureUrl = createPayPalEnvironment().baseUrl() + CHECKOUT_URL + request.getRequestId() + CAPTURE;
        final HttpEntity<PayPalOrderProcessRequestData> httpEntity = new HttpEntity<>(null, headers);
        try {
            return restTemplate.postForEntity(captureUrl, httpEntity, PayPalOrderResponseData.class);
        } catch (HttpClientErrorException e) {
            LOG.error(TRANSACTION_CAPTURE_FAILURE_MESSAGE, e);
            throw new PayPalHttpClientErrorException(e, request);
        }
    }

    private CaptureResult translateResponse(ResponseEntity<PayPalOrderResponseData> response, CaptureRequest request) {
        PayPalOrderResponseData responseData = response.getBody();
        String resultStatus = responseData.getStatus();
        final TransactionStatus transactionStatus = getTransactionStatusMap()
                .getOrDefault(resultStatus, TransactionStatus.ERROR);
        final TransactionStatusDetails transactionStatusDetails = getTransactionStatusDetailsMap().
                getOrDefault(resultStatus, TransactionStatusDetails.GENERAL_SYSTEM_ERROR);

        String requestId = getRequestId(responseData);

        LOG.info(CAPTURE_SUCCESSFUL_MESSAGE.formatted(response.getBody().toString()));

        return GenericBuilder.of(PayPalCaptureResult::new)
                .with(CaptureResult::setRequestId, requestId)
                .with(CaptureResult::setCurrency, request.getCurrency())
                .with(CaptureResult::setTotalAmount, request.getTotalAmount())
                .with(CaptureResult::setReconciliationId, request.getSubscriptionID())
                .with(CaptureResult::setMerchantTransactionCode, request.getMerchantTransactionCode())
                .with(CaptureResult::setTransactionStatus, transactionStatus)
                .with(CaptureResult::setTransactionStatusDetails, transactionStatusDetails)
                .with(PayPalCaptureResult::setCreateTime, extractCreateTime(responseData))
                .with(PayPalCaptureResult::setUpdateTime, extractUpdateTime(responseData))
                .with(PayPalCaptureResult::setRequestField, PayPalCommandsUtil.getValueAsString(request))
                .with(PayPalCaptureResult::setResponseField, PayPalCommandsUtil.getValueAsString(response))
                .with(PayPalCaptureResult::setDebugId, getDebugId(response))
                .with(PayPalCaptureResult::setCustomerId,
                        PayPalCommandsUtil.getCustomerIdFromPayPalOrderResponseData(responseData))
                .with(PayPalCaptureResult::setPaymentToken,
                        PayPalCommandsUtil.getVaultIdFromPayPalOrderResponseData(responseData))
                .build();
    }

    private String getDebugId(ResponseEntity<PayPalOrderResponseData> response) {
        return response.getHeaders().get(PAYPAL_DEBUG_ID).stream().findFirst().orElse(StringUtils.EMPTY);
    }

    private String getRequestId(PayPalOrderResponseData order) {
        return order.getPurchaseUnits().stream()
                .flatMap(unit -> unit.getPayments().getCaptures().stream())
                .findFirst()
                .filter(captureData -> COMPLETED_STATUS_RESULT.equals(captureData.getStatus()))
                .map(PayPalPaymentData::getId)
                .orElseThrow(() -> new PayPalProcessPaymentException(PAYPAL_CHECKOUT_PAYMENT_ERROR_MSG));
    }

    public void setRestTemplate(RestTemplate restTemplate) {
        this.restTemplate = restTemplate;
    }
}
