package com.paypal.hybris.core.commands.impl;

import com.paypal.http.HttpResponse;
import com.paypal.hybris.core.commands.PayPalAbstractCommand;
import com.paypal.hybris.core.exception.PayPalAuthorizeAdapterException;
import com.paypal.hybris.core.request.PayPalSavedOrderSubscriptionAuthorizationRequest;
import com.paypal.hybris.core.results.PayPalAuthorizationResult;
import com.paypal.hybris.core.util.PayPalCommandsUtil;
import com.paypal.hybris.core.util.builder.GenericBuilder;
import com.paypal.orders.Authorization;
import com.paypal.orders.AuthorizeRequest;
import com.paypal.orders.Money;
import com.paypal.orders.Order;
import com.paypal.orders.OrdersAuthorizeRequest;
import com.paypal.orders.PaymentCollection;
import com.paypal.orders.PurchaseUnit;
import de.hybris.platform.payment.commands.Command;
import de.hybris.platform.payment.commands.request.SubscriptionAuthorizationRequest;
import de.hybris.platform.payment.commands.result.AuthorizationResult;
import de.hybris.platform.payment.dto.TransactionStatus;
import de.hybris.platform.payment.dto.TransactionStatusDetails;
import org.apache.log4j.Logger;

import java.io.IOException;
import java.math.RoundingMode;
import java.time.Instant;
import java.util.Comparator;
import java.util.Date;
import java.util.Optional;
import java.util.function.Predicate;

import static com.paypal.hybris.core.constants.PaypalcoreConstants.CREATED_STATUS_RESULT;

public class DefaultPayPalAuthorizeSavedOrderCommand extends PayPalAbstractCommand implements
    Command<PayPalSavedOrderSubscriptionAuthorizationRequest, AuthorizationResult> {

    private static final String PAYPAL_DEBUG_ID = "Paypal-Debug-Id";
    private static final String SUBSCRIPTION_ID_EXCEPTION_MESSAGE = "Subscription ID is undefined, actual Subscription ID is: '%s'";

    private static final Logger LOG = Logger.getLogger(DefaultPayPalAuthorizeSavedOrderCommand.class);

    @Override
    public AuthorizationResult perform(PayPalSavedOrderSubscriptionAuthorizationRequest request) {
        OrdersAuthorizeRequest ordersAuthorizeRequest = getOrdersAuthorizeRequest(request);

        HttpResponse<Order> response = performAuthorizeRequest(ordersAuthorizeRequest);

        return translateResponse(response, request);
    }

    private OrdersAuthorizeRequest getOrdersAuthorizeRequest(PayPalSavedOrderSubscriptionAuthorizationRequest request) {
        OrdersAuthorizeRequest ordersAuthorizeRequest = Optional.ofNullable(request.getSubscriptionID())
                .map(OrdersAuthorizeRequest::new).orElseThrow(()
                        -> new IllegalArgumentException(
                        SUBSCRIPTION_ID_EXCEPTION_MESSAGE.formatted(request.getSubscriptionID())));

        ordersAuthorizeRequest.requestBody(prepareRequestBody(request));
        return ordersAuthorizeRequest;
    }

    private HttpResponse<Order> performAuthorizeRequest(OrdersAuthorizeRequest ordersAuthorizeRequest) {
        try {
            return createClient().execute(ordersAuthorizeRequest);
        } catch (IOException e) {
            LOG.error("Authorization failed: ", e);
            throw new PayPalAuthorizeAdapterException(e, ordersAuthorizeRequest);
        }
    }

    protected AuthorizeRequest prepareRequestBody(PayPalSavedOrderSubscriptionAuthorizationRequest request){
        AuthorizeRequest authorizeRequest = new AuthorizeRequest();
        Money amount = new Money();
        amount.currencyCode(request.getCurrency().getCurrencyCode());
        amount.value(String.valueOf(request.getTotalAmount().setScale(2, RoundingMode.HALF_UP)));
        authorizeRequest.amount(amount);
        return authorizeRequest;
    }

    private AuthorizationResult translateResponse(HttpResponse<Order> response,
        PayPalSavedOrderSubscriptionAuthorizationRequest request) {

        Authorization authorization = response.result().purchaseUnits().stream()
            .filter(purchaseUnit -> request.getPurchaseUnitId().equals(purchaseUnit.id()))
            .map(PurchaseUnit::payments)
            .findFirst()
            .map(PaymentCollection::authorizations)
            .orElseThrow(() -> new IllegalArgumentException(" Payments are empty "))
            .stream().filter(isCreated()).max(Comparator.comparing(t -> Instant.parse(t.createTime())))
            .orElseThrow(() -> new IllegalArgumentException(" No Authorization with status created "));


        return getAuthorizationResult(request, authorization, response);
    }


    private AuthorizationResult getAuthorizationResult(SubscriptionAuthorizationRequest request,
                                                       Authorization authorization, HttpResponse<Order> response) {
        String resultStatus = response.result().status();
        final TransactionStatus transactionStatus = getTransactionStatusMap()
                .getOrDefault(resultStatus, TransactionStatus.ERROR);
        final TransactionStatusDetails transactionStatusDetails = getTransactionStatusDetailsMap().
                getOrDefault(resultStatus, TransactionStatusDetails.GENERAL_SYSTEM_ERROR);

        return GenericBuilder.of(PayPalAuthorizationResult::new)
                .with(AuthorizationResult::setRequestId, authorization.id())
                .with(AuthorizationResult::setAuthorizationCode, authorization.id())
                .with(AuthorizationResult::setPaymentProvider, request.getPaymentProvider())
                .with(AuthorizationResult::setCurrency, request.getCurrency())
                .with(AuthorizationResult::setTotalAmount, request.getTotalAmount())
                .with(AuthorizationResult::setReconciliationId, request.getSubscriptionID())
                .with(AuthorizationResult::setMerchantTransactionCode, request.getMerchantTransactionCode())
                .with(AuthorizationResult::setTransactionStatus, transactionStatus)
                .with(AuthorizationResult::setTransactionStatusDetails, transactionStatusDetails)
                .with(PayPalAuthorizationResult::setDebugId, response.headers().header(PAYPAL_DEBUG_ID))
                .with(PayPalAuthorizationResult::setResponseField, PayPalCommandsUtil.getValueAsString(response.result()))
                .with(PayPalAuthorizationResult::setRequestField, PayPalCommandsUtil.getValueAsString(request))
                .with(PayPalAuthorizationResult::setExpirationTime, getExpirationTime(response))
                .build();
    }

    private Date getExpirationTime(HttpResponse<Order> response) {
        return PayPalCommandsUtil.convertDate(response.result().expirationTime());
    }
    private static Predicate<Authorization> isCreated() {
        return authorization -> authorization != null && CREATED_STATUS_RESULT.equals(authorization.status());
    }
}
