package com.paypal.hybris.core.commands.impl;

import com.paypal.http.HttpResponse;
import com.paypal.hybris.core.commands.PayPalAbstractCommand;
import com.paypal.hybris.core.request.PayPalAuthorizationRequest;
import com.paypal.hybris.core.results.PayPalAuthorizationResult;
import com.paypal.hybris.core.util.PayPalCommandsUtil;
import com.paypal.orders.Authorization;
import com.paypal.orders.Order;
import com.paypal.orders.OrderActionRequest;
import com.paypal.orders.OrdersAuthorizeRequest;
import com.paypal.orders.PurchaseUnit;
import de.hybris.platform.payment.AdapterException;
import de.hybris.platform.payment.commands.AuthorizationCommand;
import de.hybris.platform.payment.commands.request.AuthorizationRequest;
import de.hybris.platform.payment.commands.result.AuthorizationResult;
import org.apache.log4j.Logger;

import java.io.IOException;
import java.math.BigDecimal;
import java.util.Date;

import static com.paypal.hybris.core.constants.PaypalcoreConstants.CREATED_STATUS_RESULT;


public class DefaultPayPalAuthorizationCommand extends PayPalAbstractCommand implements AuthorizationCommand {

    private static final String AUTHORIZATION_FAILURE_MESSAGE = "Authorization failed: ";
    private static final String PERFORM_AUTHORIZATION_MESSAGE = "Performing authorization request: %s";
    private static final String AUTHORIZATION_COMPLETED_MESSAGE = "Authorization completed successfully: %s";
    private static final String PAYPAL_DEBUG_ID = "Paypal-Debug-Id";

    private static final Logger LOG = Logger.getLogger(DefaultPayPalAuthorizationCommand.class);

    @Override
    public AuthorizationResult perform(AuthorizationRequest authorizationRequest) {
        final OrdersAuthorizeRequest ordersAuthorizeRequest = getOrdersAuthorizeRequest(authorizationRequest);

        HttpResponse<Order> response = performAuthorizationRequest(ordersAuthorizeRequest);

        return translateResponse(response, ordersAuthorizeRequest, authorizationRequest);
    }

    private OrdersAuthorizeRequest getOrdersAuthorizeRequest(AuthorizationRequest authorizationRequest) {
        OrdersAuthorizeRequest ordersAuthorizeRequest = new OrdersAuthorizeRequest(
                ((PayPalAuthorizationRequest) authorizationRequest).getOrderNumber());
        ordersAuthorizeRequest.requestBody(new OrderActionRequest());
        return ordersAuthorizeRequest;
    }

    private HttpResponse<Order> performAuthorizationRequest(OrdersAuthorizeRequest ordersAuthorizeRequest) {
        try {
            LOG.info(PERFORM_AUTHORIZATION_MESSAGE.formatted(PayPalCommandsUtil.getValueAsString(ordersAuthorizeRequest.requestBody())));
            return createClient().execute(ordersAuthorizeRequest);
        } catch (IOException e) {
            LOG.error(AUTHORIZATION_FAILURE_MESSAGE, e);
            throw new AdapterException(getDescriptionFromPayPalErrorMessage(e.getMessage()));
        }
    }

    private AuthorizationResult translateResponse(HttpResponse<Order> response, OrdersAuthorizeRequest request,
                                                  AuthorizationRequest authorizationRequest) {
        String id = ((PayPalAuthorizationRequest) authorizationRequest).getOrderNumber();
        PayPalAuthorizationResult authorizationResult = new PayPalAuthorizationResult();

        response.result().purchaseUnits()
                .stream()
                .filter(unit -> id.equals(unit.id()))
                .findFirst()
                .ifPresent(purchaseUnit -> {
                    Authorization createdAuthorization = getCreatedAuthorization(purchaseUnit);
                    BigDecimal createdAuthorizationAmount = createdAuthorization != null ?
                            new BigDecimal(createdAuthorization.amount().value()) : BigDecimal.ZERO;

                    authorizationResult.setTotalAmount(createdAuthorizationAmount);
                    authorizationResult.setDebugId(response.headers().header(PAYPAL_DEBUG_ID));
                    authorizationResult.setResponseField(PayPalCommandsUtil.getValueAsString(response.result()));
                    authorizationResult.setRequestField(PayPalCommandsUtil.getValueAsString(request));
                    authorizationResult.setExpirationTime(getExpirationTime(purchaseUnit));
                });

        LOG.info(AUTHORIZATION_COMPLETED_MESSAGE.formatted(authorizationResult.getResponseField()));
        return authorizationResult;
    }

    private Date getExpirationTime(PurchaseUnit purchaseUnit) {
        return purchaseUnit.payments().authorizations().stream()
                .map(authorization -> PayPalCommandsUtil.convertDate(authorization.expirationTime()))
                .findFirst()
                .get();
    }


    private Authorization getCreatedAuthorization(PurchaseUnit purchaseUnit) {
        for (final Authorization authorization : purchaseUnit.payments().authorizations()) {
            if (authorization != null && CREATED_STATUS_RESULT.equals(authorization.status())) {
                return authorization;
            }
        }
        return null;
    }

}
