package com.paypal.hybris.core.commands;

import com.google.gson.JsonObject;
import com.google.gson.JsonParser;
import com.paypal.base.rest.APIContext;
import com.paypal.core.PayPalEnvironment;
import com.paypal.core.PayPalHttpClient;
import com.paypal.hybris.core.service.PayPalConfigurationService;
import com.paypal.hybris.data.PayPalOrderResponseData;
import com.paypal.hybris.data.PayPalPaymentData;
import com.paypal.hybris.data.PurchaseUnitData;
import de.hybris.platform.order.CartService;
import de.hybris.platform.payment.dto.TransactionStatus;
import de.hybris.platform.payment.dto.TransactionStatusDetails;
import org.apache.commons.lang3.StringUtils;
import org.springframework.http.ResponseEntity;

import java.util.Date;
import java.util.List;
import java.util.Map;
import java.util.Objects;
import java.util.concurrent.TimeUnit;

import static com.paypal.hybris.core.constants.PaypalcoreConstants.PAYPAL_INTENT_CAPTURE;
import static com.paypal.hybris.core.constants.PaypalcoreConstants.PAYPAL_LIVE_ENVIRONMENT;


public abstract class PayPalAbstractCommand {

    private static final int FIRST_ELEMENT = 0;
    private static final long TIMEOUT = 90L;
    private static final String DETAILS = "details";
    private static final String DESCRIPTION = "description";
    private static final String ISSUE = "issue";
    protected static final String PAYPAL_DEBUG_ID = "Paypal-Debug-Id";
    private PayPalConfigurationService defaultPayPalConfigurationService;
    private Map<String, TransactionStatus> transactionStatusMap;
    private Map<String, TransactionStatusDetails> transactionStatusDetailsMap;
    private CartService cartService;

    protected PayPalHttpClient createClient() {
        final PayPalHttpClient client = new PayPalHttpClient(createPayPalEnvironment());
        changeDefaultTimeout(client);
        return client;
    }

    private void changeDefaultTimeout(PayPalHttpClient client) {
        final int readTimeout = (int) TimeUnit.SECONDS.toMillis(TIMEOUT);
        client.setReadTimeout(readTimeout);
        client.setConnectTimeout(readTimeout);
    }

    protected PayPalEnvironment createPayPalEnvironment() {
        if (StringUtils.equals(PAYPAL_LIVE_ENVIRONMENT, defaultPayPalConfigurationService.getEnvironmentType())) {
            return new PayPalEnvironment.Live(
                defaultPayPalConfigurationService.getClientID(),
                defaultPayPalConfigurationService.getSecretKey());
        }
        return new PayPalEnvironment.Sandbox(
            defaultPayPalConfigurationService.getClientID(),
            defaultPayPalConfigurationService.getSecretKey());
    }

    protected APIContext createAPIContext() {
        final PayPalEnvironment payPalEnvironment = createPayPalEnvironment();
        return new APIContext(payPalEnvironment.clientId(), payPalEnvironment.clientSecret(),
            defaultPayPalConfigurationService.getEnvironmentType());
    }


    protected Date extractCreateTime(PayPalOrderResponseData responseData)  {
        return responseData.getPurchaseUnits().stream()
                .flatMap(unit -> this.getPayPalPaymentDataList(unit).stream())
                .findFirst()
                .map(PayPalPaymentData::getCreateTime)
                .orElse(new Date());
    }

    protected Date extractUpdateTime(PayPalOrderResponseData responseData)  {
        return responseData.getPurchaseUnits().stream()
                .flatMap(unit -> this.getPayPalPaymentDataList(unit).stream())
                .findFirst()
                .map(PayPalPaymentData::getUpdateTime)
                .orElse(new Date());
    }

    protected List<PayPalPaymentData> getPayPalPaymentDataList(PurchaseUnitData purchaseUnitData) {
        return getDefaultPayPalConfigurationService().getPayPalIntent().equals(PAYPAL_INTENT_CAPTURE)
                ? purchaseUnitData.getPayments().getCaptures()
                : purchaseUnitData.getPayments().getAuthorizations();
    }

    protected String getPayPalApiEndpoint() {
        return defaultPayPalConfigurationService.getPayPalApiUrl();
    }

    public PayPalConfigurationService getDefaultPayPalConfigurationService() {
        return defaultPayPalConfigurationService;
    }

    public void setDefaultPayPalConfigurationService(PayPalConfigurationService defaultPayPalConfigurationService) {
        this.defaultPayPalConfigurationService = defaultPayPalConfigurationService;
    }

    protected String getDescriptionFromPayPalErrorMessage(String message) {
        final JsonObject jsonObject = new JsonParser().parse(message).getAsJsonObject();
        final String errorDescription = jsonObject.getAsJsonArray(DETAILS).get(FIRST_ELEMENT).getAsJsonObject()
            .get(DESCRIPTION).getAsString();
        final String firstLineOfErrorDescription = errorDescription.split("\\.")[FIRST_ELEMENT];
        return firstLineOfErrorDescription;
    }

    protected String getIssueFromPayPalErrorMessage(String message) {
        final JsonObject jsonObject = new JsonParser().parse(message).getAsJsonObject();
        return jsonObject.getAsJsonArray(DETAILS).get(FIRST_ELEMENT).getAsJsonObject()
                .get(ISSUE).getAsString();
    }

    public void setTransactionStatusMap(
        Map<String, TransactionStatus> transactionStatusMap) {
        this.transactionStatusMap = transactionStatusMap;
    }

    public void setTransactionStatusDetailsMap(
        Map<String, TransactionStatusDetails> transactionStatusDetailsMap) {
        this.transactionStatusDetailsMap = transactionStatusDetailsMap;
    }

    public void setCartService(CartService cartService) {
        this.cartService = cartService;
    }

    public Map<String, TransactionStatus> getTransactionStatusMap() {
        return transactionStatusMap;
    }

    public Map<String, TransactionStatusDetails> getTransactionStatusDetailsMap() {
        return transactionStatusDetailsMap;
    }

    public CartService getCartService() {
        return cartService;
    }
}
