package com.paypal.hybris.core.auth;

import de.hybris.platform.core.Registry;
import de.hybris.platform.core.model.user.CustomerModel;
import de.hybris.platform.core.model.user.UserModel;
import de.hybris.platform.jalo.JaloConnection;
import de.hybris.platform.jalo.JaloSession;
import de.hybris.platform.jalo.user.User;
import de.hybris.platform.jalo.user.UserManager;
import de.hybris.platform.servicelayer.exceptions.UnknownIdentifierException;
import de.hybris.platform.servicelayer.user.UserService;
import de.hybris.platform.spring.security.CoreAuthenticationProvider;
import de.hybris.platform.spring.security.CoreUserDetails;
import java.util.Collections;
import org.apache.commons.lang.StringUtils;
import org.springframework.security.authentication.AbstractAuthenticationToken;
import org.springframework.security.authentication.BadCredentialsException;
import org.springframework.security.authentication.CredentialsExpiredException;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.AuthenticationException;
import org.springframework.security.core.userdetails.UserDetails;
import org.springframework.security.core.userdetails.UserDetailsChecker;
import org.springframework.security.core.userdetails.UsernameNotFoundException;

public class PayPalAuthProvider extends CoreAuthenticationProvider {

    private static final String CORE_AUTHENTICATION_PROVIDER_BAD_CREDENTIALS = "CoreAuthenticationProvider.badCredentials";
    private static final String BAD_CREDENTIALS = "Bad credentials";
    private final UserDetailsChecker postAuthenticationChecks = new PayPalAuthProvider.DefaultPostAuthenticationChecks();
    private UserService userService;


    @Override
    public Authentication authenticate(Authentication authentication) throws AuthenticationException {
        if (!isSystemHasCurrentTenant() || !isSystemInitialized()) {
            return this.createSuccessAuthentication(authentication, new CoreUserDetails("systemNotInitialized", "systemNotInitialized", true, false, true, true, Collections.emptyList(), (String)null));
        }
        try {
            String userName = authentication.getPrincipal() == null ? "NONE_PROVIDED" : authentication.getName();
            UserModel userForUID = userService.getUserForUID(userName);
            if (userForUID instanceof CustomerModel) {
                return authenticateWithPayPalAccessToken((CustomerModel) userForUID, userName, authentication);
            } else {
                return super.authenticate(authentication);
            }
        } catch (UsernameNotFoundException | UnknownIdentifierException var6) {
            throw new BadCredentialsException(
                    this.messages.getMessage(CORE_AUTHENTICATION_PROVIDER_BAD_CREDENTIALS, BAD_CREDENTIALS), var6);
        }
    }

    private Authentication authenticateWithPayPalAccessToken(CustomerModel customer, String userName, Authentication authentication) {
        UserDetails userDetails = null;
        if (StringUtils.isNotBlank(customer.getPayPalAccessToken())) {
            userDetails = this.retrieveUser(userName);
            this.getPreAuthenticationChecks().check(userDetails);
            User user = getUser(userDetails.getUsername());
            Object credential = authentication.getCredentials();
            if (credential instanceof String) {
                if (!customer.getPayPalAccessToken().equals(credential)) {
                    throw new BadCredentialsException(
                            this.messages.getMessage(CORE_AUTHENTICATION_PROVIDER_BAD_CREDENTIALS, BAD_CREDENTIALS));
                }
            } else{
                throw new BadCredentialsException(this.messages.getMessage(CORE_AUTHENTICATION_PROVIDER_BAD_CREDENTIALS, BAD_CREDENTIALS));
            }
            this.additionalAuthenticationChecks(userDetails, (AbstractAuthenticationToken) authentication);
            this.postAuthenticationChecks.check(userDetails);
            setUserToCurrentJaloSession(user);
            return this.createSuccessAuthentication(authentication, userDetails);
        } else {
            return super.authenticate(authentication);
        }
    }

    private class DefaultPostAuthenticationChecks implements UserDetailsChecker {
        private DefaultPostAuthenticationChecks() {
        }

        public void check(UserDetails user) {
            if (!user.isCredentialsNonExpired()) {
                throw new CredentialsExpiredException(PayPalAuthProvider.this.messages.getMessage("CoreAuthenticationProvider.credentialsExpired", "User credentials have expired"));
            }
        }
    }

    @Override
    protected Authentication createSuccessAuthentication(Authentication authentication, UserDetails user) {
        ConnectWithPayPalAuthenticationToken result = new ConnectWithPayPalAuthenticationToken(user.getUsername(),
            user.getAuthorities());
        result.setDetails(authentication.getDetails());
        return result;
    }

    protected boolean isSystemHasCurrentTenant() {
        return Registry.hasCurrentTenant();
    }

    protected boolean isSystemInitialized() {
        return JaloConnection.getInstance().isSystemInitialized();
    }

    protected User getUser(String login) {
        return UserManager.getInstance().getUserByLogin(login);
    }

    protected void setUserToCurrentJaloSession(User user) {
        JaloSession.getCurrentSession().setUser(user);
    }

    public void setUserService(UserService userService) {
        this.userService = userService;
    }

}
