package com.paypal.hybris.backoffice.widgets.order.tracking;

import com.hybris.cockpitng.core.events.CockpitEventQueue;
import com.hybris.cockpitng.engine.impl.DefaultWidgetInstanceManager;
import com.hybris.cockpitng.util.DefaultWidgetController;
import com.paypal.hybris.core.model.PayPalCarrierModel;
import com.paypal.hybris.core.service.PayPalCarrierService;
import com.paypal.hybris.core.service.PaymentTransactionsService;
import de.hybris.bootstrap.annotations.UnitTest;
import de.hybris.platform.basecommerce.enums.ConsignmentStatus;
import de.hybris.platform.basecommerce.model.site.BaseSiteModel;
import de.hybris.platform.core.model.c2l.CurrencyModel;
import de.hybris.platform.core.model.order.OrderModel;
import de.hybris.platform.ordersplitting.model.ConsignmentModel;
import de.hybris.platform.payment.model.PaymentTransactionEntryModel;
import de.hybris.platform.servicelayer.model.ModelService;
import de.hybris.platform.servicelayer.session.SessionService;
import org.apache.commons.lang.StringUtils;
import org.junit.After;
import org.junit.Before;
import org.junit.Test;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.MockedStatic;
import org.mockito.MockitoAnnotations;
import org.mockito.Spy;
import org.zkoss.zk.ui.WrongValueException;
import org.zkoss.zul.Combobox;
import org.zkoss.zul.Comboitem;
import org.zkoss.zul.Messagebox;
import org.zkoss.zul.Textbox;

import java.util.Collections;

import static org.mockito.ArgumentMatchers.any;
import static org.mockito.ArgumentMatchers.anyMap;
import static org.mockito.ArgumentMatchers.anyString;
import static org.mockito.Mockito.mockStatic;
import static org.mockito.Mockito.never;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.verifyNoInteractions;
import static org.mockito.Mockito.when;

@UnitTest
public class PayPalSendTrackingInfoControllerTest {

    private static final String COMPLETED_STATUS = "COMPLETED";
    private AutoCloseable closeable;
    private static final String CONSIGNMENT_CODE = "CODE";
    private static final String CARRIER_NAME = "NOVA_POST";
    private static final String TRACKING_NUMBER = "123456789";

    @Mock
    private PayPalCarrierService payPalCarrierService;
    @Mock
    private ModelService modelService;
    @Mock
    private CockpitEventQueue cockpitEventQueue;

    @Mock
    private Combobox consignmentsComboBox;
    @Mock
    private Combobox carriersComboBox;
    @Mock
    private Combobox captureTransactionsComboBox;
    @Mock
    private Comboitem carriersComboItem;
    @Mock
    private Comboitem captureTransactionsComboItem;
    @Mock
    private Textbox trackingIdTextBox;
    @Mock
    private OrderModel orderModel;
    @Mock
    private ConsignmentModel consignmentModel;
    @Mock
    private PayPalCarrierModel carrierModel;
    @Mock
    private SessionService sessionService;
    @Mock
    private BaseSiteModel baseSiteModel;
    @Mock
    private PaymentTransactionEntryModel transactionEntryModel;
    @Mock
    private CurrencyModel currencyModel;
    @Mock
    private DefaultWidgetController defaultWidgetController;
    @Mock
    private DefaultWidgetInstanceManager defaultWidgetInstanceManager;
    @Mock
    private PaymentTransactionsService payPalPaymentTransactionsService;

    @InjectMocks
    @Spy
    private PayPalSendTrackingInfoController unit;


    @Before
    public void setup() {
        closeable = MockitoAnnotations.openMocks(this);

        when(orderModel.getConsignments()).thenReturn(Collections.singleton(consignmentModel));
        when(consignmentModel.getStatus()).thenReturn(ConsignmentStatus.SHIPPED);
        when(consignmentModel.getTrackingID()).thenReturn(StringUtils.EMPTY);
        when(consignmentModel.getCarrier()).thenReturn(StringUtils.EMPTY);
        when(consignmentModel.getCode()).thenReturn(CONSIGNMENT_CODE);
        when(payPalCarrierService.getAllCarriers()).thenReturn(Collections.singletonList(carrierModel));
        when(carrierModel.getName()).thenReturn(CARRIER_NAME);

        when(consignmentsComboBox.getValue()).thenReturn(CONSIGNMENT_CODE);
        when(carriersComboBox.getValue()).thenReturn(CARRIER_NAME);
        when(carriersComboBox.getSelectedItem()).thenReturn(carriersComboItem);
        when(trackingIdTextBox.getValue()).thenReturn(TRACKING_NUMBER);
        when(captureTransactionsComboBox.getSelectedItem()).thenReturn(captureTransactionsComboItem);
        when(captureTransactionsComboBox.getValue()).thenReturn(CARRIER_NAME);
        when(carriersComboItem.getValue()).thenReturn(carrierModel);
        when(captureTransactionsComboItem.getValue()).thenReturn(transactionEntryModel);
        when(transactionEntryModel.getCurrency()).thenReturn(currencyModel);
        when(defaultWidgetController.getWidgetInstanceManager()).thenReturn(defaultWidgetInstanceManager);
    }

    @Test
    public void shouldInitCarriersAndConsignmentsComboBoxes() {
        unit.initWidget(orderModel);
        verify(consignmentsComboBox).appendItem(CONSIGNMENT_CODE);
        verify(carriersComboBox).appendChild(any(Comboitem.class));
    }

    @Test
    public void shouldNotAppendConsignmentBoxWhenNoShippedOrTrackedConsignments() {
        when(consignmentModel.getCarrier()).thenReturn(CARRIER_NAME);
        when(consignmentModel.getTrackingID()).thenReturn(TRACKING_NUMBER);
        unit.initWidget(orderModel);
        verify(consignmentsComboBox, never()).appendItem(anyString());
    }

    @Test
    public void shouldNotAppendConsignmentBoxWhenConsignmentWaiting() {
        when(consignmentModel.getStatus()).thenReturn(ConsignmentStatus.WAITING);
        unit.initWidget(orderModel);
        verify(consignmentsComboBox, never()).appendItem(anyString());
    }

    @Test
    public void shouldNotAppendConsignmentBoxWhenNoConsignments() {
        ConsignmentModel consignment = null;
        when(orderModel.getConsignments()).thenReturn(Collections.singleton(consignment));
        unit.initWidget(orderModel);
        verify(consignmentsComboBox, never()).appendItem(anyString());
    }

    @Test
    public void shouldAddCarrierAndTrackingNumberToConsignment() {
        try (MockedStatic<Messagebox> messageboxMockedStatic = mockStatic(Messagebox.class);) {
            when(sessionService.executeInLocalViewWithParams(anyMap(), any())).thenReturn(COMPLETED_STATUS);
            when(carrierModel.getCode()).thenReturn(CARRIER_NAME);
            when(orderModel.getSite()).thenReturn(baseSiteModel);

            unit.confirmTrackingForConsignment();

            verify(consignmentModel).setCarrier(CARRIER_NAME);
            verify(consignmentModel).setTrackingID(TRACKING_NUMBER);
            verify(modelService).save(consignmentModel);
        }
    }

    @Test
    public void shouldNotSaveCarrierAndTrackingNumberWhenSendToPayPalNotCompleted() {
        try (MockedStatic<Messagebox> messageboxMockedStatic = mockStatic(Messagebox.class);) {
            when(sessionService.executeInLocalViewWithParams(anyMap(), any())).thenReturn(null);
            when(orderModel.getSite()).thenReturn(baseSiteModel);

            unit.confirmTrackingForConsignment();
            verifyNoInteractions(modelService);
        }
    }

    @Test
    public void shouldPublishEventThatConsignmentsHasChanged() {
        try (MockedStatic<Messagebox> messageboxMockedStatic = mockStatic(Messagebox.class);) {
            unit.confirmTrackingForConsignment();
            verify(cockpitEventQueue, times(1)).publishEvent(any());
        }
    }

    @Test(expected = WrongValueException.class)
    public void shouldThrowWrongValueExceptionWhenConsignmentsComboBoxValueIsEmpty() {
        when(consignmentsComboBox.getValue()).thenReturn(StringUtils.EMPTY);

        unit.validateInput();
    }

    @Test(expected = WrongValueException.class)
    public void shouldThrowWrongValueExceptionWhenCaptureTransactionsComboBoxValueIsEmpty() {
        when(captureTransactionsComboBox.getValue()).thenReturn(StringUtils.EMPTY);

        unit.validateInput();
    }

    @Test(expected = WrongValueException.class)
    public void shouldThrowWrongValueExceptionWhenTrackingIdTextBoxValueIsEmpty() {
        when(trackingIdTextBox.getValue()).thenReturn(StringUtils.EMPTY);

        unit.validateInput();
    }

    @Test(expected = WrongValueException.class)
    public void shouldThrowWrongValueExceptionWhenCarriersComboBoxSelectedItemIsNull() {
        when(carriersComboBox.getValue()).thenReturn(StringUtils.EMPTY);

        unit.validateInput();
    }

    @After
    public void close() throws Exception {
        closeable.close();
    }

}
