package com.paypal.hybris.backoffice.widgets.order.refund;

import com.hybris.cockpitng.core.events.CockpitEventQueue;
import com.hybris.cockpitng.core.events.impl.DefaultCockpitEvent;
import com.hybris.cockpitng.engine.WidgetInstanceManager;
import com.paypal.hybris.core.enums.PaymentStatusType;
import com.paypal.hybris.core.service.PayPalPaymentService;
import com.paypal.hybris.core.service.PaymentTransactionsService;
import de.hybris.bootstrap.annotations.UnitTest;
import de.hybris.platform.core.model.c2l.CurrencyModel;
import de.hybris.platform.core.model.order.OrderModel;
import de.hybris.platform.payment.dto.TransactionStatus;
import de.hybris.platform.payment.dto.TransactionStatusDetails;
import de.hybris.platform.payment.enums.PaymentTransactionType;
import de.hybris.platform.payment.model.PaymentTransactionEntryModel;
import de.hybris.platform.payment.model.PaymentTransactionModel;
import de.hybris.platform.servicelayer.model.ModelService;
import org.junit.Before;
import org.junit.Test;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;
import org.mockito.Spy;
import org.zkoss.zk.ui.WrongValueException;
import org.zkoss.zul.Combobox;
import org.zkoss.zul.Comboitem;
import org.zkoss.zul.Messagebox;
import org.zkoss.zul.Textbox;

import java.math.BigDecimal;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

import static com.paypal.hybris.backoffice.constants.PaypalbackofficeConstants.OrderManagementActions.REFUND_EMPTY_AMOUNT;
import static com.paypal.hybris.backoffice.constants.PaypalbackofficeConstants.OrderManagementActions.REFUND_ERROR;
import static com.paypal.hybris.backoffice.constants.PaypalbackofficeConstants.OrderManagementActions.REFUND_INCORRECT_AMOUNT_VALUE;
import static com.paypal.hybris.backoffice.constants.PaypalbackofficeConstants.OrderManagementActions.REFUND_INVALID_FORMAT_AMOUNT;
import static com.paypal.hybris.backoffice.constants.PaypalbackofficeConstants.OrderManagementActions.REFUND_SUCCESS;
import static com.paypal.hybris.backoffice.constants.PaypalbackofficeConstants.OrderManagementActions.REFUND_TITLE;
import static com.paypal.hybris.backoffice.constants.PaypalbackofficeConstants.OrderManagementActions.REFUND_ZERO_AMOUNT;
import static org.mockito.ArgumentMatchers.any;
import static org.mockito.Matchers.anyInt;
import static org.mockito.Matchers.anyString;
import static org.mockito.Mockito.doNothing;
import static org.mockito.Mockito.doReturn;
import static org.mockito.Mockito.eq;
import static org.mockito.Mockito.never;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

@UnitTest
public class PayPalRefundControllerTest {

    private static final String OUT_MODIFIED_ITEM = "modifiedItem";
    private static final String PARTIAL_REFUND_CONFIRM_TITLE = "paypal.backoffice.partial.refund.confirm.title";
    private static final String ORDER_CODE = "orderCode";
    private static final String LABEL = "label";
    private static final String REQUEST_ID = "requestId";
    private static final String TEN = "10.00";
    private static final String EMPTY_STRING = "";
    private static final String NAN = "NaN";
    private static final String USD = "USD";

    @Mock
    private PaymentTransactionEntryModel transaction;

    @Mock
    private CurrencyModel currencyModel;

    @Mock
    private WidgetInstanceManager widgetInstanceManager;

    @Mock
    private OrderModel order;

    @Mock
    private PaymentTransactionModel transactionModel;

    @Mock
    private PaymentTransactionEntryModel transactionEntryModel;

    @Mock
    private Textbox amount;

    @Mock
    private Combobox transactionsCombobox;

    @Mock
    private Comboitem comboitem;

    @Mock
    private PayPalPaymentService paymentService;

    @Mock
    private PaymentTransactionsService paymentTransactionsService;
    @Mock
    private CockpitEventQueue cockpitEventQueue;
    @Mock
    private ModelService modelService;

    @Spy
    @InjectMocks
    private PayPalRefundController unit;

    @Before
    public void setUp() throws Exception {
        MockitoAnnotations.initMocks(this);

        when(paymentTransactionsService.calculateRefundedAmount(transactionEntryModel)).thenReturn(BigDecimal.ZERO);
        doNothing().when(unit).showMessageBox(anyString(), anyString(), anyInt(), anyString());
    }

    @Test
    public void shouldConfirmPartialRefundWhenAreTransactionsToRefund() {
        List<PaymentTransactionEntryModel> transactionEntryModels = new ArrayList<>(List.of(transactionEntryModel));
        when(paymentTransactionsService.getTransactionsToRefund(order)).thenReturn(transactionEntryModels);
        unit.setOrder(order);

        when(order.getCode()).thenReturn(ORDER_CODE);
        when(amount.getValue()).thenReturn(BigDecimal.ONE.toPlainString());
        when(transactionsCombobox.getSelectedItem()).thenReturn(comboitem);
        when(comboitem.getLabel()).thenReturn(LABEL);
        when(transactionsCombobox.getAttribute(LABEL)).thenReturn(transactionEntryModel);
        when(transactionEntryModel.getAmount()).thenReturn(BigDecimal.TEN);

        doReturn(transactionEntryModel).when(paymentService).refund(any(), any());
        when(transactionEntryModel.getTransactionStatus()).thenReturn(TransactionStatus.ACCEPTED.name());

        when(widgetInstanceManager.getLabel(REFUND_SUCCESS)).thenReturn(REFUND_SUCCESS);
        when(widgetInstanceManager.getLabel(REFUND_TITLE)).thenReturn(REFUND_TITLE);

        unit.confirmRefund();

        verify(order).setPaymentStatusType(PaymentStatusType.PARTIAL_REFUND);
        verify(unit).showMessageBox(anyString(), anyString(), anyInt(), eq(Messagebox.INFORMATION));
        verify(unit).sendOutput(OUT_MODIFIED_ITEM, order);
        verify(cockpitEventQueue).publishEvent(any(DefaultCockpitEvent.class));
    }

    @Test
    public void shouldConfirmRefundedWhenNoTransactionsToRefund() {
        when(paymentTransactionsService.getCaptureTransactionEntries(order)).thenReturn(Collections.emptyList());
        unit.setOrder(order);

        when(order.getPaymentTransactions()).thenReturn(Collections.emptyList());
        when(order.getCode()).thenReturn(ORDER_CODE);

        when(amount.getValue()).thenReturn(BigDecimal.ONE.toPlainString());

        when(transactionsCombobox.getSelectedItem()).thenReturn(comboitem);
        when(comboitem.getLabel()).thenReturn(LABEL);
        when(transactionsCombobox.getAttribute(LABEL)).thenReturn(transactionEntryModel);

        when(transactionEntryModel.getAmount()).thenReturn(BigDecimal.TEN);

        doReturn(transactionEntryModel).when(paymentService).refund(any(), any());
        when(transactionEntryModel.getTransactionStatus()).thenReturn(TransactionStatus.ACCEPTED.name());

        when(transactionEntryModel.getCurrency()).thenReturn(currencyModel);
        when(currencyModel.getIsocode()).thenReturn(USD);

        when(widgetInstanceManager.getLabel(REFUND_SUCCESS)).thenReturn(REFUND_SUCCESS);
        when(widgetInstanceManager.getLabel(REFUND_TITLE)).thenReturn(REFUND_TITLE);

        unit.confirmRefund();

        verify(order).setPaymentStatusType(PaymentStatusType.REFUNDED);
        verify(unit).showMessageBox(anyString(), anyString(), anyInt(), eq(Messagebox.INFORMATION));
        verify(unit).sendOutput(OUT_MODIFIED_ITEM, order);
    }

    @Test
    public void shouldNotConfirmRefundWhenProcessRefundThrowsException() {
        unit.setOrder(order);

        when(order.getPaymentTransactions()).thenReturn(Collections.emptyList());
        when(order.getCode()).thenReturn(ORDER_CODE);

        when(amount.getValue()).thenReturn(BigDecimal.TEN.toPlainString());

        when(transactionsCombobox.getSelectedItem()).thenReturn(comboitem);
        when(comboitem.getLabel()).thenReturn(LABEL);
        when(transactionsCombobox.getAttribute(LABEL)).thenReturn(transactionEntryModel);

        when(transactionEntryModel.getRequestId()).thenReturn(REQUEST_ID);
        when(transactionEntryModel.getAmount()).thenReturn(BigDecimal.TEN);

        when(paymentService.refund(transactionEntryModel, BigDecimal.TEN)).thenThrow(RuntimeException.class);

        when(transactionEntryModel.getCurrency()).thenReturn(currencyModel);
        when(currencyModel.getIsocode()).thenReturn(USD);

        when(widgetInstanceManager.getLabel(REFUND_ERROR)).thenReturn(REFUND_ERROR);

        unit.confirmRefund();

        verify(unit).showMessageBox(anyString(), anyString(), anyInt(), eq(Messagebox.ERROR));
        verify(unit, never()).sendOutput(OUT_MODIFIED_ITEM, order);
    }

    @Test
    public void shouldNotConfirmRefundWhenTransactionStatusRejected() {
        unit.setOrder(order);

        when(order.getPaymentTransactions()).thenReturn(Collections.emptyList());
        when(order.getCode()).thenReturn(ORDER_CODE);

        when(amount.getValue()).thenReturn(BigDecimal.TEN.toPlainString());

        when(transactionsCombobox.getSelectedItem()).thenReturn(comboitem);
        when(comboitem.getLabel()).thenReturn(LABEL);
        when(transactionsCombobox.getAttribute(LABEL)).thenReturn(transactionEntryModel);

        when(transactionEntryModel.getRequestId()).thenReturn(REQUEST_ID);
        when(transactionEntryModel.getAmount()).thenReturn(BigDecimal.TEN);

        when(paymentService.refund(transactionEntryModel, BigDecimal.TEN)).thenReturn(transactionEntryModel);
        when(transactionEntryModel.getTransactionStatus()).thenReturn(TransactionStatus.REJECTED.name());

        when(transactionEntryModel.getCurrency()).thenReturn(currencyModel);
        when(currencyModel.getIsocode()).thenReturn(USD);

        when(transactionEntryModel.getTransactionStatusDetails()).thenReturn(TransactionStatusDetails.INVALID_REQUEST.name());
        when(widgetInstanceManager.getLabel(REFUND_TITLE)).thenReturn(REFUND_TITLE);

        unit.confirmRefund();

        verify(unit).showMessageBox(anyString(), anyString(), anyInt(), eq(Messagebox.ERROR));
        verify(unit, never()).sendOutput(OUT_MODIFIED_ITEM, order);
    }

    @Test
    public void shouldNotConfirmRefundWhenAvailableAmountLessThanValue() {
        unit.setOrder(order);

        when(order.getPaymentTransactions()).thenReturn(Collections.emptyList());
        when(order.getCode()).thenReturn(ORDER_CODE);

        when(amount.getValue()).thenReturn(BigDecimal.TEN.toPlainString());

        when(transactionsCombobox.getSelectedItem()).thenReturn(comboitem);
        when(comboitem.getLabel()).thenReturn(LABEL);
        when(transactionsCombobox.getAttribute(LABEL)).thenReturn(transactionEntryModel);

        when(transactionEntryModel.getRequestId()).thenReturn(REQUEST_ID);
        when(transactionEntryModel.getAmount()).thenReturn(BigDecimal.ONE);

        when(widgetInstanceManager.getLabel(REFUND_INCORRECT_AMOUNT_VALUE)).thenReturn(REFUND_INCORRECT_AMOUNT_VALUE);
        when(widgetInstanceManager.getLabel(REFUND_TITLE)).thenReturn(REFUND_TITLE);

        when(transactionEntryModel.getCurrency()).thenReturn(currencyModel);
        when(currencyModel.getIsocode()).thenReturn(USD);

        unit.confirmRefund();

        verify(unit).showMessageBox(anyString(), anyString(), anyInt(), eq(Messagebox.ERROR));
        verify(unit).sendOutput(OUT_MODIFIED_ITEM, order);
    }


    @Test(expected = WrongValueException.class)
    public void shouldNotConfirmRefundAndThrowWrongValueExceptionWhenValueIsBlank() {
        when(amount.getValue()).thenReturn(EMPTY_STRING);
        when(widgetInstanceManager.getLabel(REFUND_EMPTY_AMOUNT)).thenReturn(REFUND_EMPTY_AMOUNT);

        unit.confirmRefund();
    }

    @Test(expected = WrongValueException.class)
    public void shouldNotConfirmRefundAndThrowWrongValueExceptionWhenValueIsZero() {
        when(amount.getValue()).thenReturn(BigDecimal.ZERO.toPlainString());
        when(widgetInstanceManager.getLabel(REFUND_ZERO_AMOUNT)).thenReturn(REFUND_ZERO_AMOUNT);

        unit.confirmRefund();
    }

    @Test(expected = WrongValueException.class)
    public void shouldNotConfirmRefundAndThrowWrongValueExceptionWhenValueIsNaN() {
        when(amount.getValue()).thenReturn(NAN);
        when(widgetInstanceManager.getLabel(REFUND_INVALID_FORMAT_AMOUNT)).thenReturn(REFUND_INVALID_FORMAT_AMOUNT);

        unit.confirmRefund();
    }

    @Test
    public void shouldInitPartialRefundFormWhenOrderHasNotTransactionModels() {
        when(widgetInstanceManager.getLabel(PARTIAL_REFUND_CONFIRM_TITLE)).thenReturn(PARTIAL_REFUND_CONFIRM_TITLE);
        when(order.getCode()).thenReturn(ORDER_CODE);
        doNothing().when(widgetInstanceManager).setTitle(anyString());

        when(order.getPaymentTransactions()).thenReturn(Collections.emptyList());

        when(transactionsCombobox.getSelectedItem()).thenReturn(comboitem);
        when(comboitem.getLabel()).thenReturn(LABEL);
        when(transactionsCombobox.getAttribute(LABEL)).thenReturn(transactionEntryModel);
        when(transactionEntryModel.getAmount()).thenReturn(BigDecimal.TEN);

        when(transactionEntryModel.getRequestId()).thenReturn(REQUEST_ID);

        unit.initPartialRefundForm(order);

        verify(transactionsCombobox).setSelectedIndex(0);
        verify(amount).setValue(TEN);
    }

    @Test
    public void shouldInitPartialRefundFormWhenOrderHasNotEntryTransactionModels() {
        when(widgetInstanceManager.getLabel(PARTIAL_REFUND_CONFIRM_TITLE)).thenReturn(PARTIAL_REFUND_CONFIRM_TITLE);
        when(order.getCode()).thenReturn(ORDER_CODE);
        doNothing().when(widgetInstanceManager).setTitle(anyString());

        when(order.getPaymentTransactions()).thenReturn(Collections.singletonList(transactionModel));
        when(transactionModel.getEntries()).thenReturn(Collections.emptyList());

        when(transactionsCombobox.getSelectedItem()).thenReturn(comboitem);
        when(comboitem.getLabel()).thenReturn(LABEL);
        when(transactionsCombobox.getAttribute(LABEL)).thenReturn(transactionEntryModel);
        when(transactionEntryModel.getAmount()).thenReturn(BigDecimal.TEN);

        when(transactionEntryModel.getRequestId()).thenReturn(REQUEST_ID);

        unit.initPartialRefundForm(order);

        verify(transactionsCombobox).setSelectedIndex(0);
        verify(amount).setValue(TEN);
    }


    @Test
    public void shouldInitPartialRefundFormWhenTransactionStatusRejected() {
        when(widgetInstanceManager.getLabel(PARTIAL_REFUND_CONFIRM_TITLE)).thenReturn(PARTIAL_REFUND_CONFIRM_TITLE);
        when(order.getCode()).thenReturn(ORDER_CODE);
        doNothing().when(widgetInstanceManager).setTitle(anyString());

        when(order.getPaymentTransactions()).thenReturn(Collections.singletonList(transactionModel));
        when(transactionModel.getEntries()).thenReturn(Collections.singletonList(transactionEntryModel));
        when(transactionEntryModel.getTransactionStatus()).thenReturn(TransactionStatus.REJECTED.name());

        when(transactionsCombobox.getSelectedItem()).thenReturn(comboitem);
        when(comboitem.getLabel()).thenReturn(LABEL);
        when(transactionsCombobox.getAttribute(LABEL)).thenReturn(transactionEntryModel);
        when(transactionEntryModel.getAmount()).thenReturn(BigDecimal.TEN);

        when(transactionEntryModel.getRequestId()).thenReturn(REQUEST_ID);

        unit.initPartialRefundForm(order);

        verify(transactionsCombobox).setSelectedIndex(0);
        verify(amount).setValue(TEN);
    }

    @Test
    public void shouldInitPartialRefundFormWhenTransactionTypeAuthorize() {
        when(widgetInstanceManager.getLabel(PARTIAL_REFUND_CONFIRM_TITLE)).thenReturn(PARTIAL_REFUND_CONFIRM_TITLE);
        when(order.getCode()).thenReturn(ORDER_CODE);
        doNothing().when(widgetInstanceManager).setTitle(anyString());

        when(order.getPaymentTransactions()).thenReturn(Collections.singletonList(transactionModel));
        when(transactionModel.getEntries()).thenReturn(Collections.singletonList(transactionEntryModel));
        when(transactionEntryModel.getTransactionStatus()).thenReturn(TransactionStatus.ACCEPTED.name());
        when(transactionEntryModel.getType()).thenReturn(PaymentTransactionType.AUTHORIZATION);

        when(transactionsCombobox.getSelectedItem()).thenReturn(comboitem);
        when(comboitem.getLabel()).thenReturn(LABEL);
        when(transactionsCombobox.getAttribute(LABEL)).thenReturn(transactionEntryModel);
        when(transactionEntryModel.getAmount()).thenReturn(BigDecimal.TEN);

        when(transactionEntryModel.getRequestId()).thenReturn(REQUEST_ID);

        unit.initPartialRefundForm(order);

        verify(transactionsCombobox).setSelectedIndex(0);
        verify(amount).setValue(TEN);
    }

    @Test
    public void shouldInitPartialRefundFormWhenTransactionTypeCapture() {
        when(widgetInstanceManager.getLabel(PARTIAL_REFUND_CONFIRM_TITLE)).thenReturn(PARTIAL_REFUND_CONFIRM_TITLE);
        when(order.getCode()).thenReturn(ORDER_CODE);
        doNothing().when(widgetInstanceManager).setTitle(anyString());

        when(order.getPaymentTransactions()).thenReturn(Collections.singletonList(transactionModel));
        when(transactionModel.getEntries()).thenReturn(Collections.singletonList(transactionEntryModel));
        when(transactionEntryModel.getTransactionStatus()).thenReturn(TransactionStatus.ACCEPTED.name());
        when(transactionEntryModel.getType()).thenReturn(PaymentTransactionType.CAPTURE);

        when(transactionsCombobox.getSelectedItem()).thenReturn(comboitem);
        when(comboitem.getLabel()).thenReturn(LABEL);
        when(transactionsCombobox.getAttribute(LABEL)).thenReturn(transactionEntryModel);
        when(transactionEntryModel.getAmount()).thenReturn(BigDecimal.TEN);

        when(transactionEntryModel.getRequestId()).thenReturn(REQUEST_ID);

        unit.initPartialRefundForm(order);

        verify(transactionsCombobox).appendItem(REQUEST_ID);
        verify(transactionsCombobox).setAttribute(REQUEST_ID, transactionEntryModel);
        verify(transactionsCombobox).setSelectedIndex(0);
        verify(amount).setValue(TEN);
    }

    @Test
    public void shouldInitPartialRefundFormWhenTransactionTypePartialCapture() {
        when(widgetInstanceManager.getLabel(PARTIAL_REFUND_CONFIRM_TITLE)).thenReturn(PARTIAL_REFUND_CONFIRM_TITLE);
        when(order.getCode()).thenReturn(ORDER_CODE);
        doNothing().when(widgetInstanceManager).setTitle(anyString());

        when(order.getPaymentTransactions()).thenReturn(Collections.singletonList(transactionModel));
        when(transactionModel.getEntries()).thenReturn(Collections.singletonList(transactionEntryModel));
        when(transactionEntryModel.getTransactionStatus()).thenReturn(TransactionStatus.ACCEPTED.name());
        when(transactionEntryModel.getType()).thenReturn(PaymentTransactionType.PARTIAL_CAPTURE);

        when(transactionsCombobox.getSelectedItem()).thenReturn(comboitem);
        when(comboitem.getLabel()).thenReturn(LABEL);
        when(transactionsCombobox.getAttribute(LABEL)).thenReturn(transactionEntryModel);
        when(transactionEntryModel.getAmount()).thenReturn(BigDecimal.TEN);

        when(transactionEntryModel.getRequestId()).thenReturn(REQUEST_ID);

        unit.initPartialRefundForm(order);

        verify(transactionsCombobox).appendItem(REQUEST_ID);
        verify(transactionsCombobox).setAttribute(REQUEST_ID, transactionEntryModel);
        verify(transactionsCombobox).setSelectedIndex(0);
        verify(amount).setValue(TEN);
    }

    @Test
    public void shouldInitPartialRefundFormWhenAmountEqualsZero() {
        when(widgetInstanceManager.getLabel(PARTIAL_REFUND_CONFIRM_TITLE)).thenReturn(PARTIAL_REFUND_CONFIRM_TITLE);
        when(order.getCode()).thenReturn(ORDER_CODE);
        doNothing().when(widgetInstanceManager).setTitle(anyString());

        when(order.getPaymentTransactions()).thenReturn(Collections.singletonList(transactionModel));
        when(transactionModel.getEntries()).thenReturn(Collections.singletonList(transactionEntryModel));
        when(transactionEntryModel.getTransactionStatus()).thenReturn(TransactionStatus.ACCEPTED.name());
        when(transactionEntryModel.getType()).thenReturn(PaymentTransactionType.REFUND_STANDALONE);

        when(transactionsCombobox.getSelectedItem()).thenReturn(comboitem);
        when(comboitem.getLabel()).thenReturn(LABEL);
        when(transactionsCombobox.getAttribute(LABEL)).thenReturn(transactionEntryModel);
        when(transactionEntryModel.getAmount()).thenReturn(BigDecimal.TEN);

        when(transactionEntryModel.getRequestId()).thenReturn(REQUEST_ID);
        when(transactionEntryModel.getAmount()).thenReturn(BigDecimal.ZERO);

        unit.initPartialRefundForm(order);

        verify(transactionsCombobox).setSelectedIndex(0);
        verify(amount).setValue(BigDecimal.ZERO.toPlainString());
    }
}