package com.paypal.hybris.backoffice.widgets.order.reauthorize;

import com.hybris.cockpitng.engine.WidgetInstanceManager;
import com.hybris.cockpitng.engine.impl.ComponentWidgetAdapter;
import com.paypal.enums.PayPalPaymentProvider;
import com.paypal.http.HttpRequest;
import com.paypal.hybris.core.enums.PaymentStatusType;
import com.paypal.hybris.core.exception.PayPalReauthorizationTooSoonException;
import com.paypal.hybris.core.exception.PayPalReauthorizeAdapterException;
import com.paypal.hybris.core.results.PayPalAuthorizationResult;
import com.paypal.hybris.core.service.PayPalPaymentService;
import com.paypal.payments.Authorization;
import de.hybris.bootstrap.annotations.UnitTest;
import de.hybris.platform.core.model.c2l.CurrencyModel;
import de.hybris.platform.core.model.order.OrderModel;
import de.hybris.platform.core.model.user.CustomerModel;
import de.hybris.platform.payment.AdapterException;
import de.hybris.platform.payment.commands.request.SubscriptionAuthorizationRequest;
import de.hybris.platform.payment.commands.result.AuthorizationResult;
import de.hybris.platform.payment.dto.TransactionStatus;
import de.hybris.platform.payment.enums.PaymentTransactionType;
import de.hybris.platform.payment.model.PaymentTransactionEntryModel;
import de.hybris.platform.payment.model.PaymentTransactionModel;
import de.hybris.platform.servicelayer.model.ModelService;
import org.junit.Before;
import org.junit.Test;
import org.mockito.ArgumentCaptor;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;
import org.mockito.Spy;
import org.zkoss.zk.ui.WrongValueException;
import org.zkoss.zul.Combobox;
import org.zkoss.zul.Comboitem;
import org.zkoss.zul.Messagebox;
import org.zkoss.zul.Textbox;

import java.io.IOException;
import java.math.BigDecimal;
import java.util.Collections;
import java.util.Date;
import java.util.List;

import static com.paypal.hybris.backoffice.constants.PaypalbackofficeConstants.OrderManagementActions.REAUTHORIZE_EMPTY_AMOUNT;
import static com.paypal.hybris.backoffice.constants.PaypalbackofficeConstants.OrderManagementActions.REAUTHORIZE_ERROR;
import static com.paypal.hybris.backoffice.constants.PaypalbackofficeConstants.OrderManagementActions.REAUTHORIZE_INVALID_FORMAT_AMOUNT;
import static com.paypal.hybris.backoffice.constants.PaypalbackofficeConstants.OrderManagementActions.REAUTHORIZE_SUCCESS;
import static com.paypal.hybris.backoffice.constants.PaypalbackofficeConstants.OrderManagementActions.REAUTHORIZE_TITLE;
import static com.paypal.hybris.backoffice.constants.PaypalbackofficeConstants.OrderManagementActions.REAUTHORIZE_ZERO_AMOUNT;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertNotNull;
import static org.junit.Assert.assertTrue;
import static org.mockito.Mockito.any;
import static org.mockito.Mockito.anyInt;
import static org.mockito.Mockito.anyString;
import static org.mockito.Mockito.doNothing;
import static org.mockito.Mockito.doReturn;
import static org.mockito.Mockito.eq;
import static org.mockito.Mockito.never;
import static org.mockito.Mockito.times;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

@UnitTest
public class PayPalReauthorizeControllerTest {

    private static final String OUT_MODIFIED_ITEM = "modifiedItem";
    private static final String ORDER_IS_NOT_EXIST = "Order is not exist";
    private static final String ORDER_CODE = "orderCode";
    private static final String DISPLAY_NAME = "displayName";
    private static final String REQUEST_ID = "requestId";
    private static final String BLANK_LINE = "";
    private static final String NAN = "NaN";
    private static final String MERCHANT_TRANSACTION_CODE = "merchantTransactionCode";
    private static final String USD = "USD";
    private static final String AUTHORIZATION_CODE = "authorizationCode";
    private static final String DEBUG_ID = "debug_id";
    private static final String REQUEST = "request";
    private static final String RESPONSE = "response";
    private static final String REQUEST_FAILED = "{\"path\":\"path\",\"verb\":\"POST\",\"body\":null,\"responseClass\":\"com.paypal.payments.Authorization\",\"headers\":{\"mHeaders\":{},\"keyMapping\":{}}}";
    private static final Date UPDATE_TIME = new Date();

    @Mock
    private ComponentWidgetAdapter componentWidgetAdapter;

    @Mock
    private OrderModel order;

    @Mock
    private PaymentTransactionModel transactionModel;

    @Mock
    private PaymentTransactionEntryModel entryModel;

    @Mock
    private CustomerModel customerModel;

    @Mock
    private CurrencyModel currencyModel;

    @Mock
    private PayPalAuthorizationResult authorizationResult;

    @Mock
    private Textbox orderCode;

    @Mock
    private Textbox customer;

    @Mock
    private Textbox amount;

    @Mock
    private Combobox transactions;

    @Mock
    private Comboitem comboitem;

    @Mock
    private PayPalPaymentService paymentService;

    @Mock
    private ModelService modelService;

    @Mock
    private WidgetInstanceManager widgetInstanceManager;
    @Mock
    private SubscriptionAuthorizationRequest request;
    @Mock
    private PayPalAuthorizationResult payPalAuthorizationResult;
    @Mock
    private PayPalReauthorizeAdapterException exception;
    @Mock
    private IOException ioException;

    @Spy
    @InjectMocks
    private PayPalReauthorizeController unit;

    @Before
    public void setUp() throws Exception {
        MockitoAnnotations.initMocks(this);

        doNothing().when(unit).showMessageBox(any(), anyString(), anyInt(), anyString());
        when(transactionModel.getCode()).thenReturn(MERCHANT_TRANSACTION_CODE);
        when(entryModel.getCurrency()).thenReturn(currencyModel);
        when(currencyModel.getIsocode()).thenReturn(USD);
        when(transactionModel.getPaymentProvider()).thenReturn(PayPalPaymentProvider.PAYPAL.name());
        when(entryModel.getPaymentTransaction()).thenReturn(transactionModel);
        when(exception.getParentException()).thenReturn(ioException);
    }

    @Test
    public void initReauthorizeRequestFormWithComboitems() {
        doNothing().when(widgetInstanceManager).setTitle(REAUTHORIZE_TITLE + " " + ORDER_CODE);
        when(widgetInstanceManager.getLabel(REAUTHORIZE_TITLE)).thenReturn(REAUTHORIZE_TITLE);

        when(order.getCode()).thenReturn(ORDER_CODE);
        when(order.getUser()).thenReturn(customerModel);
        when(customerModel.getDisplayName()).thenReturn(DISPLAY_NAME);

        //getPossibleAmountForReauthorize
        when(order.getTotalPrice()).thenReturn(100D);

        //getAuthorizeTransactions
        when(order.getPaymentTransactions()).thenReturn(Collections.singletonList(transactionModel));
        when(transactionModel.getEntries()).thenReturn(Collections.singletonList(entryModel));
        when(entryModel.getType()).thenReturn(PaymentTransactionType.AUTHORIZATION);
        when(entryModel.getTransactionStatus()).thenReturn(TransactionStatus.ACCEPTED.name());

        //configureTransactionsCombobox
        when(entryModel.getRequestId()).thenReturn(REQUEST_ID);
        when(transactions.getItems()).thenReturn(Collections.singletonList(comboitem));

        unit.initReauthorizeRequestForm(order);

        verify(transactions).setSelectedItem(comboitem);
        verify(transactions).appendChild(any());
        verify(orderCode).setValue(ORDER_CODE);
        verify(customer).setValue(DISPLAY_NAME);
        verify(amount).setValue("100.00");
    }

    @Test
    public void initReauthorizeRequestFormWithoutComboitems() {
        doNothing().when(widgetInstanceManager).setTitle(REAUTHORIZE_TITLE + " " + ORDER_CODE);
        when(widgetInstanceManager.getLabel(REAUTHORIZE_TITLE)).thenReturn(REAUTHORIZE_TITLE);

        when(order.getCode()).thenReturn(ORDER_CODE);
        when(order.getUser()).thenReturn(customerModel);
        when(customerModel.getDisplayName()).thenReturn(DISPLAY_NAME);

        //getPossibleAmountForReauthorize
        when(order.getTotalPrice()).thenReturn(100D);

        //getAuthorizeTransactions
        when(order.getPaymentTransactions()).thenReturn(Collections.singletonList(transactionModel));
        when(transactionModel.getEntries()).thenReturn(Collections.singletonList(entryModel));
        when(entryModel.getType()).thenReturn(PaymentTransactionType.AUTHORIZATION);
        when(entryModel.getTransactionStatus()).thenReturn(TransactionStatus.ACCEPTED.name());

        //configureTransactionsCombobox
        when(entryModel.getRequestId()).thenReturn(REQUEST_ID);
        when(transactions.getItems()).thenReturn(Collections.emptyList());

        unit.initReauthorizeRequestForm(order);

        verify(transactions, never()).setSelectedItem(comboitem);
        verify(transactions).appendChild(any());
        verify(orderCode).setValue(ORDER_CODE);
        verify(customer).setValue(DISPLAY_NAME);
        verify(amount).setValue("100.00");
    }

    @Test
    public void shouldSelectTransactionEntry() {
        when(transactions.getSelectedItem()).thenReturn(comboitem);
        when(comboitem.getValue()).thenReturn(entryModel);
        when(entryModel.getAmount()).thenReturn(BigDecimal.TEN);

        unit.selectTransactionEntry();

        verify(amount).setValue("10.00");
    }

    @Test
    public void shouldNotConfirmWhenOrderNull() {
        unit.setOrder(null);
        when(transactions.getSelectedIndex()).thenReturn(1);
        when(transactions.getSelectedItem()).thenReturn(comboitem);
        when(comboitem.getValue()).thenReturn(entryModel);

        //validateAmount
        when(amount.getValue()).thenReturn(BigDecimal.TEN.toPlainString());

        when(widgetInstanceManager.getLabel(REAUTHORIZE_ERROR)).thenReturn(REAUTHORIZE_ERROR);

        unit.confirm();

        verify(unit, never()).sendOutput(OUT_MODIFIED_ITEM, order);
        verify(unit).showMessageBox(eq(ORDER_IS_NOT_EXIST), anyString(), anyInt(), eq(Messagebox.ERROR));
    }

    @Test
    public void shouldNotConfirmWhenReauthorizationTooSoon() {
        unit.setOrder(order);
        when(transactions.getSelectedIndex()).thenReturn(1);
        when(transactions.getSelectedItem()).thenReturn(comboitem);
        when(comboitem.getValue()).thenReturn(entryModel);

        //validateAmount
        when(amount.getValue()).thenReturn(BigDecimal.TEN.toPlainString());

        //reauthorize
        when(paymentService.reauthorize(any())).thenThrow(PayPalReauthorizationTooSoonException.class);

        when(widgetInstanceManager.getLabel(REAUTHORIZE_ERROR)).thenReturn(REAUTHORIZE_ERROR);

        unit.confirm();

        verify(unit, never()).sendOutput(OUT_MODIFIED_ITEM, order);
        verify(unit).showMessageBox(any(), anyString(), anyInt(), eq(Messagebox.ERROR));
    }

    @Test
    public void shouldConfirm() {
        unit.setOrder(order);
        when(transactions.getSelectedIndex()).thenReturn(1);
        when(transactions.getSelectedItem()).thenReturn(comboitem);
        when(comboitem.getValue()).thenReturn(entryModel);

        //validateAmount
        when(amount.getValue()).thenReturn(BigDecimal.TEN.toPlainString());

        //reauthorize
        when(paymentService.reauthorize(any())).thenReturn(authorizationResult);
        when(authorizationResult.getAuthorizationCode()).thenReturn(AUTHORIZATION_CODE);


        //prepareReauthorizeRequest
        when(order.getPaymentTransactions()).thenReturn(Collections.singletonList(transactionModel));
        when(transactionModel.getEntries()).thenReturn(Collections.singletonList(entryModel));
        when(entryModel.getType()).thenReturn(PaymentTransactionType.AUTHORIZATION);
        when(entryModel.getTransactionStatus()).thenReturn(TransactionStatus.ACCEPTED.name());
        when(entryModel.getAmount()).thenReturn(BigDecimal.TEN);


        when(widgetInstanceManager.getLabel(REAUTHORIZE_SUCCESS)).thenReturn(REAUTHORIZE_SUCCESS);
        when(widgetInstanceManager.getLabel(REAUTHORIZE_TITLE)).thenReturn(REAUTHORIZE_TITLE);
        when(order.getCode()).thenReturn(ORDER_CODE);

        unit.confirm();

        verify(modelService).save(transactionModel);
        verify(modelService).save(order);
        verify(order).setAlreadyReauthorized(Boolean.TRUE);
        verify(order).setPaymentStatusType(PaymentStatusType.PENDING);

        verify(unit).showMessageBox(anyString(), anyString(), anyInt(), eq(Messagebox.INFORMATION));
        verify(unit).sendOutput(OUT_MODIFIED_ITEM, order);
    }

    @Test
    public void shouldNotConfirmWhenExceptionThrowsDuringReauthorization() {
        unit.setOrder(order);
        when(transactions.getSelectedIndex()).thenReturn(1);
        when(transactions.getSelectedItem()).thenReturn(comboitem);
        when(comboitem.getValue()).thenReturn(entryModel);

        //validateAmount
        when(amount.getValue()).thenReturn(BigDecimal.TEN.toPlainString());

        //reauthorize
        when(paymentService.reauthorize(any())).thenThrow(RuntimeException.class);

        when(widgetInstanceManager.getLabel(REAUTHORIZE_ERROR)).thenReturn(REAUTHORIZE_ERROR);

        unit.confirm();

        verify(unit, never()).sendOutput(OUT_MODIFIED_ITEM, order);
        verify(order).setPaymentStatusType(PaymentStatusType.FAILED);
        verify(unit).showMessageBox(any(), anyString(), anyInt(), eq(Messagebox.ERROR));
    }

    @Test(expected = WrongValueException.class)
    public void shouldNotConfirmAndThrowWrongValueExceptionWhenValueIsBlank() {
        when(transactions.getSelectedIndex()).thenReturn(-1);

        when(amount.getValue()).thenReturn(BLANK_LINE);
        when(widgetInstanceManager.getLabel(REAUTHORIZE_EMPTY_AMOUNT)).thenReturn(REAUTHORIZE_EMPTY_AMOUNT);

        unit.confirm();
    }

    @Test(expected = WrongValueException.class)
    public void shouldNotConfirmAndThrowWrongValueExceptionWhenValueIsZero() {
        when(transactions.getSelectedIndex()).thenReturn(-1);

        when(amount.getValue()).thenReturn(BigDecimal.ZERO.toPlainString());
        when(widgetInstanceManager.getLabel(REAUTHORIZE_ZERO_AMOUNT)).thenReturn(REAUTHORIZE_ZERO_AMOUNT);

        unit.confirm();
    }

    @Test(expected = WrongValueException.class)
    public void shouldNotConfirmAndThrowWrongValueExceptionWhenValueIsNaN() {
        when(transactions.getSelectedIndex()).thenReturn(-1);

        when(amount.getValue()).thenReturn(NAN);
        when(widgetInstanceManager.getLabel(REAUTHORIZE_INVALID_FORMAT_AMOUNT)).thenReturn(REAUTHORIZE_INVALID_FORMAT_AMOUNT);

        unit.confirm();
    }

    @Test(expected = AdapterException.class)
    public void shouldAdapterExceptionWhenPayPalReauthorizeAdapterExceptionOccurs() {
        when(paymentService.reauthorize(any())).thenThrow(exception);

        unit.reauthorize(entryModel, BigDecimal.TEN);
    }

    @Test
    public void shouldUpdateAuthorizationEntryWhenReauthorizeIsSuccessful() {
        doReturn(request).when(unit).prepareReauthorizeRequest(BigDecimal.TEN, entryModel);

        when(paymentService.reauthorize(request)).thenReturn(payPalAuthorizationResult);
        when(payPalAuthorizationResult.getDebugId()).thenReturn(DEBUG_ID);
        when(payPalAuthorizationResult.getRequestField()).thenReturn(REQUEST);
        when(payPalAuthorizationResult.getResponseField()).thenReturn(RESPONSE);
        when(payPalAuthorizationResult.getUpdateTime()).thenReturn(UPDATE_TIME);
        when(payPalAuthorizationResult.getAuthorizationCode()).thenReturn(AUTHORIZATION_CODE);

        assertTrue(unit.reauthorize(entryModel, BigDecimal.TEN));

        verify(entryModel).setDebugId(DEBUG_ID);
        verify(entryModel).setUpdateTime(UPDATE_TIME);
        verify(entryModel).setAmount(BigDecimal.TEN);
        verify(entryModel).setRequestId(AUTHORIZATION_CODE);
        verify(entryModel).setRequest(REQUEST);
        verify(entryModel).setResponse(RESPONSE);
    }

    @Test
    public void shouldPopulateValuesWhenPayPalReauthorizeExceptionIsThrown() {
        HttpRequest<Authorization> authorizationHttpRequest = new HttpRequest<>("path", "POST", Authorization.class);
        when(exception.getParentException()).thenReturn(ioException);
        when(ioException.getMessage()).thenReturn(REAUTHORIZE_ERROR);
        when(exception.getRequest()).thenReturn(authorizationHttpRequest);
        when(exception.getDebugId()).thenReturn(DEBUG_ID);

        unit.createFailedReauthorizationTransactionEntryModel(exception, request, transactionModel);

        ArgumentCaptor<PaymentTransactionEntryModel> captor = ArgumentCaptor.forClass(PaymentTransactionEntryModel.class);

        verify(modelService, times(2)).save(captor.capture());
        List<PaymentTransactionEntryModel> entryList = captor.getAllValues();
        PaymentTransactionEntryModel entry = entryList.get(1);
        assertNotNull(entryList);
        assertEquals(DEBUG_ID, entry.getDebugId());
        assertEquals(REAUTHORIZE_ERROR, entry.getResponse());
        assertEquals(REQUEST_FAILED, entry.getRequest());
    }
}
