package com.paypal.hybris.backoffice.widgets.order.capture;

import com.hybris.cockpitng.engine.WidgetInstanceManager;
import com.hybris.cockpitng.engine.impl.ComponentWidgetAdapter;
import com.paypal.http.exceptions.HttpException;
import com.paypal.hybris.core.enums.PaymentStatusType;
import com.paypal.hybris.core.exception.PayPalCaptureAdapterException;
import com.paypal.hybris.core.model.PayPalCreditCardPaymentInfoModel;
import com.paypal.hybris.core.service.PayPalManualMultiCaptureService;
import de.hybris.bootstrap.annotations.UnitTest;
import de.hybris.platform.core.enums.OrderStatus;
import de.hybris.platform.core.model.c2l.CurrencyModel;
import de.hybris.platform.core.model.order.OrderModel;
import de.hybris.platform.core.model.order.payment.PaymentInfoModel;
import de.hybris.platform.core.model.user.CustomerModel;
import de.hybris.platform.orderprocessing.model.OrderProcessModel;
import de.hybris.platform.payment.AdapterException;
import de.hybris.platform.payment.PaymentService;
import de.hybris.platform.payment.dto.TransactionStatus;
import de.hybris.platform.payment.dto.TransactionStatusDetails;
import de.hybris.platform.payment.enums.PaymentTransactionType;
import de.hybris.platform.payment.methods.CardPaymentService;
import de.hybris.platform.payment.model.PaymentTransactionEntryModel;
import de.hybris.platform.payment.model.PaymentTransactionModel;
import de.hybris.platform.processengine.BusinessProcessService;
import de.hybris.platform.servicelayer.i18n.I18NService;
import de.hybris.platform.servicelayer.model.ModelService;
import de.hybris.platform.store.BaseStoreModel;
import org.junit.Before;
import org.junit.Test;
import org.mockito.InjectMocks;
import org.mockito.Mock;
import org.mockito.MockitoAnnotations;
import org.mockito.Spy;
import org.springframework.http.HttpStatus;
import org.zkoss.zk.ui.WrongValueException;
import org.zkoss.zul.Combobox;
import org.zkoss.zul.Comboitem;
import org.zkoss.zul.Messagebox;
import org.zkoss.zul.Textbox;
import com.paypal.http.HttpRequest;
import com.paypal.http.Headers;
import com.paypal.payments.Capture;

import java.math.BigDecimal;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.Collections;
import java.util.Date;
import java.util.List;

import static com.paypal.hybris.backoffice.constants.PaypalbackofficeConstants.OrderManagementActions.PARTIAL_CAPTURE_ERROR;
import static com.paypal.hybris.backoffice.constants.PaypalbackofficeConstants.OrderManagementActions.PARTIAL_CAPTURE_SUCCESS;
import static com.paypal.hybris.backoffice.constants.PaypalbackofficeConstants.OrderManagementActions.PARTIAL_CAPTURE_TITLE;
import static junit.framework.TestCase.assertEquals;
import static org.mockito.Matchers.any;
import static org.mockito.Matchers.anyInt;
import static org.mockito.Matchers.anyString;
import static org.mockito.Matchers.eq;
import static org.mockito.Mockito.doNothing;
import static org.mockito.Mockito.never;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.when;

@UnitTest
public class PayPalCaptureControllerTest {

    private static final String OUT_MODIFIED_ITEM = "modifiedItem";
    private static final String REVIEW_DECISION = "_ReviewDecision";
    private static final String ORDER_CODE = "orderCode";
    private static final String DISPLAY_MAME = "displayMame";
    private static final String REQUEST_ID = "requestId";
    private static final String SUBSCRIPTION_ID = "subscriptionId";
    private static final String REQUEST_ID1 = "requestId";
    private static final String AUTHORIZATION_ID = "authorizationId";
    private static final String PROCESS_CODE = "processCode";
    private static final String BLANK_LINE = "";
    private static final String NAN = "NaN";
    private static final String TRANSACTION_CODE = "transactionCode";
    private static final String ERROR_MESSAGE = "capture failed";
    private static final String NEW_ENTRY_CODE = "transactionCode-CAPTURE-1";
    private static final String HEADER_DATE = "Mon, 30 May 2024 15:45:00 GMT";
    private static final String HEADER_DATE_KEY = "Date";
    private static final String HEADER_DEBUG_ID_KEY = "Paypal-Debug-Id";
    private static final String HEADER_DEBUG_ID = "DebugId";
    private static final String DATE_PATTERN = "EEE, dd MMM yyyy HH:mm:ss zzz";

    @Mock
    private ComponentWidgetAdapter componentWidgetAdapter;

    @Mock
    private OrderModel order;

    @Mock
    private OrderProcessModel orderProcessModel;

    @Mock
    private BaseStoreModel baseStoreModel;

    @Mock
    private PaymentTransactionModel transactionModel;

    @Mock
    private CurrencyModel currencyModel;

    @Mock
    private PaymentTransactionEntryModel entryModel;

    @Mock
    private PaymentTransactionEntryModel secondEntryModel;

    @Mock
    private CustomerModel customerModel;

    @Mock
    private PayPalCreditCardPaymentInfoModel paymentInfoModel;

    @Mock
    private PaymentInfoModel simplePaymentInfoModel;

    @Mock
    private Textbox orderCode;

    @Mock
    private Textbox customer;

    @Mock
    private Textbox amount;

    @Mock
    private Combobox transactions;

    @Mock
    private Comboitem comboitem;

    @Mock
    private transient PaymentService paymentService;

    @Mock
    private transient ModelService modelService;

    @Mock
    private BusinessProcessService businessProcessService;

    @Mock
    private CardPaymentService cardPaymentService;

    @Mock
    private I18NService i18nService;

    @Mock
    private PayPalManualMultiCaptureService payPalManualMultiCaptureService;

    @Mock
    private WidgetInstanceManager widgetInstanceManager;

    @Spy
    @InjectMocks
    private PayPalCaptureController unit;

    @Before
    public void setUp() throws Exception {
        MockitoAnnotations.initMocks(this);

        doNothing().when(unit).showMessageBox(any(), anyString(), anyInt(), anyString());

        when(baseStoreModel.getSubmitOrderProcessCode()).thenReturn(PROCESS_CODE);
        when(orderProcessModel.getOrder()).thenReturn(order);
        when(orderProcessModel.getCode()).thenReturn(PROCESS_CODE);
        when(order.getCode()).thenReturn(ORDER_CODE);
        when(order.getOrderProcess()).thenReturn(Collections.singletonList(orderProcessModel));
        when(order.getStore()).thenReturn(baseStoreModel);
    }


    @Test
    public void shouldConfirmWhenOrderFullyCaptured() {
        unit.setOrder(order);
        when(transactions.getSelectedIndex()).thenReturn(1);
        when(transactions.getSelectedItem()).thenReturn(comboitem);
        when(comboitem.getValue()).thenReturn(entryModel);

        when(amount.getValue()).thenReturn(BigDecimal.TEN.toPlainString());

        when(order.getPaymentTransactions()).thenReturn(Collections.singletonList(transactionModel));
        when(transactionModel.getEntries()).thenReturn(List.of(entryModel, secondEntryModel));
        when(entryModel.getPaymentTransaction()).thenReturn(transactionModel);
        when(entryModel.getType()).thenReturn(PaymentTransactionType.AUTHORIZATION);
        when(entryModel.getTransactionStatus()).thenReturn(TransactionStatus.ACCEPTED.name());
        when(entryModel.getAmount()).thenReturn(BigDecimal.valueOf(100));
        when(payPalManualMultiCaptureService.doMultiCapture(BigDecimal.TEN, entryModel, PaymentTransactionType.PARTIAL_CAPTURE)).thenReturn(secondEntryModel);

        when(order.getPaymentInfo()).thenReturn(paymentInfoModel);
        when(paymentInfoModel.isSaveOrderFlowActive()).thenReturn(Boolean.FALSE);
        when(transactionModel.getPlannedAmount()).thenReturn(BigDecimal.TEN);

        when(secondEntryModel.getType()).thenReturn(PaymentTransactionType.CAPTURE);
        when(secondEntryModel.getTransactionStatus()).thenReturn(TransactionStatus.ACCEPTED.name());
        when(secondEntryModel.getSubscriptionID()).thenReturn(SUBSCRIPTION_ID);
        when(secondEntryModel.getAmount()).thenReturn(BigDecimal.valueOf(100));

        doNothing().when(unit).executeManualPaymentCaptureOperation(order);

        when(widgetInstanceManager.getLabel(PARTIAL_CAPTURE_SUCCESS)).thenReturn(PARTIAL_CAPTURE_SUCCESS);
        when(widgetInstanceManager.getLabel(PARTIAL_CAPTURE_TITLE)).thenReturn(PARTIAL_CAPTURE_TITLE);

        unit.confirm();

        verify(unit).showMessageBox(anyString(), anyString(), anyInt(), eq(Messagebox.INFORMATION));
        verify(unit).sendOutput(OUT_MODIFIED_ITEM, order);
    }

    @Test
    public void shouldConfirmPartialCaptureWhenSaveOrderFlowActive() {
        unit.setOrder(order);
        when(transactions.getSelectedIndex()).thenReturn(1);
        when(transactions.getSelectedItem()).thenReturn(comboitem);
        when(comboitem.getValue()).thenReturn(entryModel);

        when(amount.getValue()).thenReturn(BigDecimal.TEN.toPlainString());

        when(order.getPaymentTransactions()).thenReturn(Collections.singletonList(transactionModel));
        when(transactionModel.getEntries()).thenReturn(Collections.singletonList(entryModel));
        when(entryModel.getPaymentTransaction()).thenReturn(transactionModel);
        when(entryModel.getType()).thenReturn(PaymentTransactionType.AUTHORIZATION);
        when(entryModel.getTransactionStatus()).thenReturn(TransactionStatus.ACCEPTED.name());
        when(entryModel.getAmount()).thenReturn(BigDecimal.valueOf(100));
        when(payPalManualMultiCaptureService.doMultiCapture(BigDecimal.TEN, entryModel, PaymentTransactionType.PARTIAL_CAPTURE)).thenReturn(entryModel);

        when(order.getPaymentInfo()).thenReturn(paymentInfoModel);
        when(paymentInfoModel.isSaveOrderFlowActive()).thenReturn(Boolean.TRUE);
        when(entryModel.getRequestId()).thenReturn(REQUEST_ID);
        when(entryModel.getAuthorizationId()).thenReturn(AUTHORIZATION_ID);
        when(transactionModel.getPlannedAmount()).thenReturn(BigDecimal.TEN);

        when(widgetInstanceManager.getLabel(PARTIAL_CAPTURE_SUCCESS)).thenReturn(PARTIAL_CAPTURE_SUCCESS);
        when(widgetInstanceManager.getLabel(PARTIAL_CAPTURE_TITLE)).thenReturn(PARTIAL_CAPTURE_TITLE);

        unit.confirm();

        verify(transactionModel).setPlannedAmount(BigDecimal.valueOf(100));
        verify(modelService).save(transactionModel);
        verify(unit).showMessageBox(anyString(), anyString(), anyInt(), eq(Messagebox.INFORMATION));
        verify(unit).sendOutput(OUT_MODIFIED_ITEM, order);
        verify(businessProcessService, never()).triggerEvent(ORDER_CODE + REVIEW_DECISION);
    }

    @Test
    public void shouldNotConfirmWhenThrowsAdapterException() {
        unit.setOrder(order);
        when(transactions.getSelectedIndex()).thenReturn(1);
        when(transactions.getSelectedItem()).thenReturn(comboitem);
        when(comboitem.getValue()).thenReturn(entryModel);

        when(amount.getValue()).thenReturn(BigDecimal.TEN.toPlainString());

        when(order.getPaymentTransactions()).thenReturn(Collections.singletonList(transactionModel));
        when(transactionModel.getEntries()).thenReturn(Collections.singletonList(entryModel));
        when(entryModel.getPaymentTransaction()).thenReturn(transactionModel);
        when(entryModel.getType()).thenReturn(PaymentTransactionType.AUTHORIZATION);
        when(entryModel.getTransactionStatus()).thenReturn(TransactionStatus.ACCEPTED.name());
        when(entryModel.getAmount()).thenReturn(BigDecimal.TEN);
        when(payPalManualMultiCaptureService.doMultiCapture(BigDecimal.TEN, entryModel, PaymentTransactionType.CAPTURE)).thenThrow(AdapterException.class);

        when(transactionModel.getCode()).thenReturn(TRANSACTION_CODE);
        when(transactionModel.getCurrency()).thenReturn(currencyModel);

        when(widgetInstanceManager.getLabel(PARTIAL_CAPTURE_ERROR)).thenReturn(PARTIAL_CAPTURE_ERROR);
        when(widgetInstanceManager.getLabel(PARTIAL_CAPTURE_TITLE)).thenReturn(PARTIAL_CAPTURE_TITLE);

        unit.confirm();

        verify(unit).showMessageBox(anyString(), anyString(), anyInt(), eq(Messagebox.ERROR));
        verify(unit, never()).sendOutput(OUT_MODIFIED_ITEM, order);
    }

    @Test
    public void shouldNotConfirmWhenThrowsRuntimeException() {
        unit.setOrder(order);
        when(transactions.getSelectedIndex()).thenReturn(1);
        when(transactions.getSelectedItem()).thenReturn(comboitem);
        when(comboitem.getValue()).thenReturn(entryModel);

        when(amount.getValue()).thenReturn(BigDecimal.TEN.toPlainString());

        when(order.getPaymentTransactions()).thenReturn(Collections.singletonList(transactionModel));
        when(transactionModel.getEntries()).thenReturn(Collections.singletonList(entryModel));
        when(entryModel.getPaymentTransaction()).thenReturn(transactionModel);
        when(entryModel.getType()).thenReturn(PaymentTransactionType.AUTHORIZATION);
        when(entryModel.getTransactionStatus()).thenReturn(TransactionStatus.ACCEPTED.name());
        when(entryModel.getAmount()).thenReturn(BigDecimal.TEN);
        when(payPalManualMultiCaptureService.doMultiCapture(BigDecimal.TEN, entryModel, PaymentTransactionType.CAPTURE)).thenThrow(RuntimeException.class);

        when(widgetInstanceManager.getLabel(PARTIAL_CAPTURE_ERROR)).thenReturn(PARTIAL_CAPTURE_ERROR);

        unit.confirm();

        verify(unit).showMessageBox(anyString(), anyString(), anyInt(), eq(Messagebox.ERROR));
        verify(unit, never()).sendOutput(OUT_MODIFIED_ITEM, order);
    }

    @Test
    public void shouldNotConfirmWhenOrderNull() {
        unit.setOrder(null);
        when(transactions.getSelectedIndex()).thenReturn(1);
        when(transactions.getSelectedItem()).thenReturn(comboitem);
        when(comboitem.getValue()).thenReturn(entryModel);

        when(amount.getValue()).thenReturn(BigDecimal.TEN.toPlainString());

        when(widgetInstanceManager.getLabel(PARTIAL_CAPTURE_ERROR)).thenReturn(PARTIAL_CAPTURE_ERROR);
        when(widgetInstanceManager.getLabel(PARTIAL_CAPTURE_TITLE)).thenReturn(PARTIAL_CAPTURE_TITLE);

        unit.confirm();

        verify(unit).showMessageBox(anyString(), anyString(), anyInt(), eq(Messagebox.ERROR));
        verify(unit, never()).sendOutput(OUT_MODIFIED_ITEM, order);
    }

    @Test(expected = WrongValueException.class)
    public void shouldNotConfirmAndThrowWrongValueExceptionWhenValueIsBlank() {
        when(transactions.getSelectedIndex()).thenReturn(-1);
        when(amount.getValue()).thenReturn(BLANK_LINE);

        unit.confirm();
    }

    @Test(expected = WrongValueException.class)
    public void shouldNotConfirmAndThrowWrongValueExceptionWhenValueIsZero() {
        when(transactions.getSelectedIndex()).thenReturn(-1);
        when(amount.getValue()).thenReturn(BigDecimal.ZERO.toPlainString());

        unit.confirm();
    }

    @Test(expected = WrongValueException.class)
    public void shouldNotConfirmAndThrowWrongValueExceptionWhenValueIsNaN() {
        when(transactions.getSelectedIndex()).thenReturn(-1);
        when(amount.getValue()).thenReturn(NAN);

        unit.confirm();
    }

    @Test
    public void shouldInitCaptureRequestFormWhenSaveOrderFlowIsActiveAndWithNotEmptyItems() {
        when(order.getCode()).thenReturn(ORDER_CODE);
        when(widgetInstanceManager.getLabel(PARTIAL_CAPTURE_TITLE)).thenReturn(PARTIAL_CAPTURE_TITLE);
        when(order.getUser()).thenReturn(customerModel);
        when(customerModel.getDisplayName()).thenReturn(DISPLAY_MAME);

        when(order.getPaymentTransactions()).thenReturn(Collections.singletonList(transactionModel));
        when(transactionModel.getEntries()).thenReturn(List.of(entryModel, secondEntryModel));
        when(entryModel.getType()).thenReturn(PaymentTransactionType.AUTHORIZATION);
        when(entryModel.getTransactionStatus()).thenReturn(TransactionStatus.ACCEPTED.name());

        when(order.getPaymentInfo()).thenReturn(paymentInfoModel);
        when(paymentInfoModel.isSaveOrderFlowActive()).thenReturn(Boolean.TRUE);

        when(entryModel.getRequestId()).thenReturn(REQUEST_ID);
        when(transactions.getItems()).thenReturn(Collections.singletonList(comboitem));
        when(transactions.getSelectedItem()).thenReturn(comboitem);
        when(comboitem.getValue()).thenReturn(secondEntryModel);

        when(secondEntryModel.getAmount()).thenReturn(BigDecimal.TEN);

        when(secondEntryModel.getType()).thenReturn(PaymentTransactionType.CAPTURE);
        when(secondEntryModel.getTransactionStatus()).thenReturn(TransactionStatus.ACCEPTED.name());

        when(secondEntryModel.getRequestId()).thenReturn(REQUEST_ID);
        when(secondEntryModel.getAuthorizationId()).thenReturn(AUTHORIZATION_ID);

        unit.initCaptureRequestForm(order);

        verify(transactions).appendChild(any());
        verify(amount).setValue("10.00");
    }

    @Test
    public void initCaptureRequestFormWhenSaveOrderFlowIsNotActiveAndWithEmptyItems() {
        when(order.getCode()).thenReturn(ORDER_CODE);
        when(widgetInstanceManager.getLabel(PARTIAL_CAPTURE_TITLE)).thenReturn(PARTIAL_CAPTURE_TITLE);
        when(order.getUser()).thenReturn(customerModel);
        when(customerModel.getDisplayName()).thenReturn(DISPLAY_MAME);

        when(order.getPaymentTransactions()).thenReturn(Collections.singletonList(transactionModel));
        when(transactionModel.getEntries()).thenReturn(List.of(entryModel, secondEntryModel));
        when(entryModel.getType()).thenReturn(PaymentTransactionType.AUTHORIZATION);
        when(entryModel.getTransactionStatus()).thenReturn(TransactionStatus.ACCEPTED.name());

        when(order.getPaymentInfo()).thenReturn(paymentInfoModel);
        when(paymentInfoModel.isSaveOrderFlowActive()).thenReturn(Boolean.FALSE);

        when(entryModel.getRequestId()).thenReturn(REQUEST_ID);
        when(transactions.getItems()).thenReturn(Collections.emptyList());

        unit.initCaptureRequestForm(order);

        verify(transactions).appendChild(any());
    }

    @Test
    public void shouldSelectTransactionEntry() {
        when(transactions.getSelectedItem()).thenReturn(comboitem);
        when(comboitem.getValue()).thenReturn(entryModel);
        when(entryModel.getAmount()).thenReturn(BigDecimal.ZERO);

        when(entryModel.getType()).thenReturn(PaymentTransactionType.PARTIAL_CAPTURE);
        when(entryModel.getTransactionStatus()).thenReturn(TransactionStatus.ACCEPTED.name());

        when(entryModel.getSubscriptionID()).thenReturn(SUBSCRIPTION_ID);

        unit.selectTransactionEntry();

        verify(amount).setValue("0.00");
    }

    @Test
    public void shouldExecuteManualPaymentCaptureOperationWhenSaveOrderFlowNotActive() {
        unit.setOrder(order);
        when(order.getPaymentInfo()).thenReturn(simplePaymentInfoModel);
        when(order.getCode()).thenReturn(ORDER_CODE);

        when(order.getOrderProcess()).thenReturn(Collections.singletonList(orderProcessModel));
        when(orderProcessModel.getCode()).thenReturn(PROCESS_CODE);
        when(order.getStore()).thenReturn(baseStoreModel);

        unit.executeManualPaymentCaptureOperation(order);

        verify(order).setStatus(OrderStatus.PAYMENT_CAPTURED);
        verify(order).setPaymentStatusType(PaymentStatusType.COMPLETED);
        verify(modelService).save(order);
        verify(businessProcessService).triggerEvent(ORDER_CODE + REVIEW_DECISION);
    }

    @Test
    public void shouldExecuteManualPaymentCaptureOperationWhenSaveOrderFlowActive() {
        unit.setOrder(order);
        when(order.getPaymentInfo()).thenReturn(paymentInfoModel);
        when(paymentInfoModel.isSaveOrderFlowActive()).thenReturn(Boolean.TRUE);
        when(orderProcessModel.getOrder()).thenReturn(order);

        unit.executeManualPaymentCaptureOperation(order);

        verify(order).setStatus(OrderStatus.PAYMENT_CAPTURED);
        verify(order).setPaymentStatusType(PaymentStatusType.COMPLETED);
        verify(modelService).save(order);
        verify(businessProcessService).triggerEvent(ORDER_CODE + REVIEW_DECISION);
    }

    @Test
    public void shouldPopulateEntryWithProperDataWhenPayPalCaptureAdapterExceptionIsThrown() {
        PayPalCaptureAdapterException e = createAdapterException();
        when(transactionModel.getCode()).thenReturn(TRANSACTION_CODE);
        when(transactionModel.getCurrency()).thenReturn(currencyModel);

        PaymentTransactionEntryModel entry = unit.handlePayPalSDKError(transactionModel, e, BigDecimal.TEN);

        assertEquals(PaymentTransactionType.CAPTURE, entry.getType());
        assertEquals(transactionModel, entry.getPaymentTransaction());
        assertEquals(BigDecimal.TEN, entry.getAmount());
        assertEquals(currencyModel, entry.getCurrency());
        assertEquals(NEW_ENTRY_CODE, entry.getCode());
        assertEquals(TransactionStatus.ERROR.toString(), entry.getTransactionStatus());
        assertEquals(request(), entry.getRequest());
        assertEquals(ERROR_MESSAGE, entry.getFailureReason());
        assertEquals(HEADER_DEBUG_ID, entry.getDebugId());
        assertEquals(ERROR_MESSAGE, entry.getResponse());
        assertEquals(getHeaderDate(), entry.getCreateTime());
        assertEquals(TransactionStatusDetails.INVALID_REQUEST.name(), entry.getTransactionStatusDetails());
    }

    private PayPalCaptureAdapterException createAdapterException() {
        HttpRequest<Capture> captureHttpRequest = new HttpRequest<>("path", "POST", Capture.class);
        Headers headers = new Headers();
        headers.header(HEADER_DATE_KEY, HEADER_DATE);
        headers.header(HEADER_DEBUG_ID_KEY, HEADER_DEBUG_ID);
        HttpException httpException = new HttpException(ERROR_MESSAGE, HttpStatus.UNPROCESSABLE_ENTITY.value(), headers);
        return new PayPalCaptureAdapterException(httpException, captureHttpRequest);
    }
    private Date getHeaderDate() {
        SimpleDateFormat format = new SimpleDateFormat(DATE_PATTERN);
        try {
            return format.parse(HEADER_DATE);
        } catch (ParseException ex) {
            ex.printStackTrace();
        }
        return new Date();
    }
    private String request() {
        return """
                {
                  "path" : "path",
                  "verb" : "POST",
                  "body" : null,
                  "responseClass" : "com.paypal.payments.Capture",
                  "headers" : {
                    "mHeaders" : { },
                    "keyMapping" : { }
                  }
                }""";
    }
}
